#!/usr/bin/env python3
"sa_harness.py -- create sqlalchemy definitions from create table & index stmts"

import glob, collections, argparse
import sqlparse
from . import wrappers, diffing

PREAMBLE = """# autogenerated by sa_harness.py
import enum, sqlalchemy as sa
from sqlalchemy.dialects.postgresql import UUID, JSONB
"""

def read_glob_stmts(glob_pattern):
  "return wrapped stmts for files matching glob"
  stmts = []
  for fname in glob.glob(glob_pattern):
    stmts.extend(map(wrappers.wrap, sqlparse.parse(open(fname).read())))
  return stmts

def column_args(column, pkey_fields):
  "parse sql column in a horrible way and return args for column constructor"
  parsed = column.parse()
  assert parsed.success # todo: does parse() log something on failure?
  dets = {'name': parsed.name, 'type': parsed.type}
  if parsed.pkey or parsed.name in pkey_fields:
    dets['primary_key'] = True
  if parsed.default:
    # todo: figure out escaping here
    dets['server_default'] = "sa.text('%s')" % parsed.default
  if parsed.unique:
    raise NotImplementedError('todo: sa_harness unique keys')
  if parsed.not_null:
    dets['nullable'] = False
  return dets.pop('name'), dets.pop('type'), dets

TYPES = {
  'int': 'sa.Integer',
  'jsonb': 'JSONB',
  'json': 'sa.JSON',
  'text': 'sa.Text',
  'uuid': 'UUID',
  'timestamp': 'sa.DateTime',
  'boolean': 'sa.Boolean',
}

def render_col(args, col_name, col_type, details, composite_pkey):
  if composite_pkey and col_name in composite_pkey:
    details['primary_key'] = True
  params = ', '.join([TYPES[col_type]] + ['%s=%s' % pair for pair in details.items()])
  if args.classic:
    return f"sa.Column('{col_name}', {params}),"
  else:
    return f"{col_name} = sa.Column({params})"

def snake_case(raw):
  return raw.title().replace('_', '')

def transform(args, table_stmts, delim='\n  '):
  "return some representation of sqlalchemy tables"
  table_strings = []
  for tablename, stmts in diffing.group_by_table(table_stmts).items():
    indexes = []
    cols = collections.OrderedDict()
    composite_pkey = None
    for stmt in stmts:
      if isinstance(stmt, wrappers.CreateTable):
        pkey_fields = stmt.pkey_fields()
        for col in stmt.columns():
          name, type_, dets = column_args(col, pkey_fields)
          cols[name, type_] = dets
      elif isinstance(stmt, wrappers.CreateIndex):
        if stmt.index_name is None: # what case is this?
          continue
        if not args.classic: # not sure I need these in classic mode either unless SQLA being used to generate prod db
          continue
        paren = stmt.paren()
        assert isinstance(paren, sqlparse.sql.Parenthesis)
        index_cols = ['%s_table.c.%s' % (stmt.table, ident) for ident, in wrappers.split_pun(paren)]
        indexes.append(f"{stmt.index_name}_index = sa.Index('{stmt.index_name}', {', '.join(index_cols)})")
      elif isinstance(stmt, wrappers.CreateEnum):
        assert not any('"' in val for val in stmt.values), f"error: this doesn't know how to handle quotes in enum values {str(stmt.values)}"
        table_strings.append(delim.join([f"class {stmt.name}Enum(enum.Enum):"] + [f'{val} = "{val}"' for i, val in enumerate(stmt.values)]))
        TYPES[stmt.name] = f"sa.Enum({stmt.name}Enum)"
      else:
        raise TypeError('unhandled statement type', type(stmt))
    if cols:
      col_strings = [
        render_col(args, col_name, col_type, details, composite_pkey)
        for (col_name, col_type), details in cols.items()
      ]
      if args.classic:
        table_strings.append(f"{tablename}_table = sa.Table('{tablename}', META,{delim}{delim.join(col_strings)}\n)")
      else:
        table_strings.append(delim.join([
          f"class {snake_case(tablename)}(Base):",
          f"__tablename__ = '{tablename}'",
          delim.join(col_strings),
        ]))
    table_strings.extend(indexes)
  return table_strings

PARSER = argparse.ArgumentParser(description="generate sqlalchemy models from create table stmts")
PARSER.add_argument('glob')
PARSER.add_argument('--target', help="optional output path; otherwise stdout")
PARSER.add_argument('--classic', action='store_true', help="by default use sqlalchemy declarative mapping, pass this to use classic")

def main():
  args = PARSER.parse_args()
  stmts = read_glob_stmts(args.glob)
  preamble = PREAMBLE
  if args.classic:
    preamble += "META = sa.MetaData()"
  else:
    preamble += "from sqlalchemy.ext.declarative import declarative_base\nBase = declarative_base()"
  value = '\n\n'.join([preamble] + transform(args, stmts))
  if args.target:
    open(args.target, 'w').write(value)
  else:
    print(value)

if __name__ == '__main__':
  main()
