"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DenyList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const create_map_1 = require("./create-map");
const prune_1 = require("./prune");
/**
 * Manages the construct hub deny list.
 */
class DenyList extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The object key within the bucket with the deny list JSON file.
         */
        this.objectKey = 'deny-list.json';
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            versioned: true,
        });
        const directory = this.writeToFile(props.rules, this.objectKey);
        // upload the deny list to the bucket
        const upload = new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            destinationBucket: this.bucket,
            prune: true,
            retainOnDelete: false,
            sources: [s3deploy.Source.asset(directory)],
        });
        this.uploadReady = upload.node.findChild('CustomResource');
        this.prune = new prune_1.Prune(this, 'Prune', {
            packageDataBucket: props.packageDataBucket,
            packageDataKeyPrefix: props.packageDataKeyPrefix,
            monitoring: props.monitoring,
        });
        this.grantRead(this.prune.pruneHandler);
        // trigger prune when the deny list changes
        const pruneOnChange = (_a = props.pruneOnChange) !== null && _a !== void 0 ? _a : true;
        if (pruneOnChange) {
            this.prune.pruneHandler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.bucket, {
                events: [s3.EventType.OBJECT_CREATED],
                filters: [{ prefix: this.objectKey, suffix: this.objectKey }],
            }));
            // Add an explicit dep between upload and the bucket notification. We are
            // not using the whole bucket scope to reduce the likelihood of a
            // dependency cycle.
            const notificationsReady = this.bucket.node.findChild('Notifications');
            this.uploadReady.node.addDependency(notificationsReady);
        }
        // trigger prune periodically (every 5 minutes) - just in case
        const prunePeriod = (_b = props.prunePeriod) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5);
        if (prunePeriod && prunePeriod.toSeconds() > 0) {
            new events.Rule(this, 'PeriodicPrune', {
                schedule: events.Schedule.rate(prunePeriod),
                targets: [new targets.LambdaFunction(this.prune.pruneHandler)],
            });
        }
    }
    /**
     * Grants an AWS Lambda function permissions to read the deny list, and adds
     * the relevant environment variables expected by the `DenyListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment(constants_1.ENV_DENY_LIST_BUCKET_NAME, this.bucket.bucketName);
        handler.addEnvironment(constants_1.ENV_DENY_LIST_OBJECT_KEY, this.objectKey);
        this.bucket.grantRead(handler);
    }
    /**
     * Number of rules in the deny list.
     */
    metricDenyListRules(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "DenyListRuleCount" /* DENY_LIST_RULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * Writes the deny list to a temporary file and returns a path to a directory
     * with the JSON file. The contents of the JSON file is a map where keys are
     * package names (and optionally, version) and the value is the deny list
     * entry. This makes it easier to query by the service.
     *
     * Also performs some validation to make sure there are no duplicate entries.
     *
     * @param list The deny list
     * @returns a path to a temporary directory that can be deployed to S3
     */
    writeToFile(list, fileName) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'deny-list-'));
        const filePath = path.join(tmpdir, fileName);
        const map = create_map_1.createDenyListMap(list);
        fs.writeFileSync(filePath, JSON.stringify(map, null, 2));
        return tmpdir;
    }
}
exports.DenyList = DenyList;
//# sourceMappingURL=data:application/json;base64,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