# pyOCD debugger
# Copyright (c) 2017 NXP
# Copyright (c) 2018,2020 Arm Limited
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from ...core.memory_map import (FlashRegion, RomRegion, RamRegion, MemoryMap)
from ...debug.svd.loader import SVDFile
from ..family.target_imxrt import IMXRT

FLASH_ALGO_QUADSPI = {
    'load_address' : 0x20200000,

    # Flash algorithm as a hex string
    'instructions': [
    0xE00ABE00, 0x062D780D, 0x24084068, 0xD3000040, 0x1E644058, 0x1C49D1FA, 0x2A001E52, 0x4770D1F2,
    0x4605b570, 0x4616460c, 0xcc0fe002, 0x3e10c50f, 0xd2fa2e10, 0xd3022e08, 0xc503cc03, 0x2e043e08,
    0xcc01d307, 0x1f36c501, 0x7821e003, 0x1c647029, 0x1e761c6d, 0xbd70d2f9, 0x4770ba40, 0x4770ba40,
    0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770bac0, 0x4770bac0, 0x4770bac0,
    0x4770bac0, 0x4770bac0, 0x4770bac0, 0x4770bac0, 0x4875b510, 0x08408900, 0x49730040, 0x48738108,
    0x08408900, 0x49710040, 0x486f8108, 0x21048800, 0x28004008, 0x486cd004, 0x43888800, 0x8008496a,
    0x8800486a, 0x40082104, 0xd0042800, 0x88004867, 0x49664388, 0x48668008, 0x21806800, 0x28004008,
    0x4864d00c, 0x60484962, 0x60884863, 0x68004608, 0x43882180, 0x43082120, 0x6008495d, 0x6900485f,
    0x0fc007c0, 0xd0052800, 0x6900485c, 0x00400840, 0x6108495a, 0x6940485a, 0x04092101, 0x42884008,
    0xf000d101, 0xbd10f855, 0x4605b5fe, 0x4617460e, 0xffaef7ff, 0x90004853, 0x4953466a, 0x20004449,
    0xfb87f003, 0x2c004604, 0x2001d001, 0x494ebdfe, 0x20004449, 0xf8aef001, 0xd0012800, 0xe7f52001,
    0xe7f32000, 0x20004601, 0xb5104770, 0x44494946, 0xf0012000, 0x2800fa7d, 0x2001d001, 0x2000bd10,
    0xb510e7fc, 0x20034604, 0x1a240740, 0x031b2301, 0x493d4622, 0x20004449, 0xfbc8f003, 0xd0012800,
    0xbd102001, 0xe7fc2000, 0x4604b570, 0x4615460e, 0x07402003, 0x462b1a24, 0x49334622, 0x20004449,
    0xf9e4f001, 0xd0012800, 0xbd702001, 0xe7fc2000, 0x2300b510, 0x34804c2a, 0xbf006063, 0xbf00bf00,
    0x8f4ff3bf, 0xbf00bf00, 0x4b25bf00, 0x2401695b, 0x43a30424, 0x61634c22, 0xbf00bf00, 0xf3bfbf00,
    0xbf008f4f, 0xbf00bf00, 0x33804b1d, 0x4b1f6818, 0x0b594003, 0x4b1ebf00, 0x08da4003, 0x014bbf00,
    0x40234c1c, 0x0fa40794, 0x432307a4, 0x63634c1a, 0xbf00bf00, 0x1e524613, 0xd1f02b00, 0x1e49460b,
    0xd1e82b00, 0xbf00bf00, 0xf3bfbf00, 0xbf008f4f, 0xbf00bf00, 0xbf00bf00, 0xf3bfbf00, 0xbf008f6f,
    0xbf00bf00, 0x0000bd10, 0x400b8000, 0x400d0000, 0x400bc000, 0xd928c520, 0x0000ffff, 0xe000e000,
    0xe000ed00, 0xc0000406, 0x00000004, 0x0fffe000, 0x00001ff8, 0x00003fe0, 0xe000ef40, 0x48cfb510,
    0x444949cf, 0xbf006008, 0x6b0048ce, 0x03092101, 0x49cc4388, 0xbf006308, 0x6b0048ca, 0x07c00fc0,
    0xd0f92800, 0x680048c8, 0x400849c8, 0x49c83018, 0x49c54308, 0x20036008, 0x49c20300, 0xbf006108,
    0x690048c0, 0x07c00fc0, 0xd0f92800, 0x690048bd, 0x04092101, 0x49bb4388, 0x48bb6108, 0x6b003840,
    0x400849ba, 0x49bb3023, 0x49b74308, 0x63083940, 0x49b92001, 0x48b96108, 0x444949b9, 0x24016008,
    0x1c64e000, 0x48b64621, 0x68004448, 0xfe15f004, 0x428849b4, 0x48b0d8f5, 0x49b36940, 0x1e614008,
    0x22070409, 0x40110412, 0x1e614308, 0x22030209, 0x40110212, 0x49a84308, 0x46086148, 0x09c06a40,
    0x1cc001c0, 0x46086248, 0x02916980, 0x18404388, 0x618849a1, 0x69804608, 0x400849a4, 0x06892105,
    0x499d4308, 0x46086188, 0x49a169c0, 0x05514008, 0x21071840, 0x430805c9, 0x49971c40, 0x489161c8,
    0x21016b00, 0x43880409, 0x6308498e, 0x69004608, 0x04092101, 0x498b4388, 0xbd106108, 0x3180498e,
    0x22036809, 0x43110292, 0x32804a8b, 0x47706011, 0x31804989, 0x22036809, 0x43910292, 0x32804a86,
    0x47706011, 0xb08cb5f7, 0x4615460c, 0x27002600, 0x900b2000, 0x2214900a, 0x44794986, 0xf7ffa805,
    0x2214fe0f, 0x44794983, 0x46683108, 0xfe08f7ff, 0x2d00bf00, 0x2d01d002, 0xe048d000, 0x38404872,
    0x09866b00, 0x487401b6, 0x210769c0, 0x438805c9, 0x23004607, 0xd1012d00, 0xe000ab05, 0x2c01466b,
    0x2c09db34, 0x2401dd00, 0x5c180060, 0x0060900b, 0x784018c0, 0x980b900a, 0x0e800680, 0x980a4306,
    0x05c01e40, 0x05c92107, 0x43074008, 0x6800486a, 0x43082102, 0x60084968, 0xf7ff980c, 0x485affa9,
    0x6b003840, 0xd00242b0, 0x38404857, 0x485a6306, 0x42b869c0, 0x4858d001, 0x980c61c7, 0xff8ef7ff,
    0x6800485d, 0x43882102, 0x6008495b, 0xbf00bf00, 0xbdf0b00f, 0xb085b5f0, 0x4616460d, 0x20002400,
    0x484f9004, 0x68004448, 0x2d009002, 0x2d01d004, 0x2d02d006, 0xe010d12e, 0x44484849, 0xe02c6804,
    0x69404845, 0x02092103, 0x0a004008, 0x46391c47, 0xf0049802, 0x4604fd32, 0x483fe01f, 0x483a69c0,
    0x6b003840, 0x0e800680, 0x48449001, 0xf0049901, 0x2112fd24, 0x90004348, 0x69c04837, 0x05c92107,
    0x0dc04008, 0x90031c40, 0x98009903, 0xfd15f004, 0xe0024604, 0x90042004, 0xbf00bf00, 0x98046034,
    0xbdf0b005, 0x4603b510, 0xbf002004, 0xd1012b00, 0xd1002900, 0x2a01e008, 0x4c31d102, 0xe001600c,
    0x600c4c2f, 0xbf002000, 0xbd10bf00, 0x44484824, 0x47706800, 0x4820b510, 0x21036940, 0x40080209,
    0x1c440a00, 0x481e4621, 0x68004448, 0xfce5f004, 0xb5f8bd10, 0x460d4604, 0xffe8f7ff, 0x49214606,
    0xfcdbf004, 0xe00c4607, 0x46382103, 0xfcd5f004, 0xe0009000, 0x9800bf00, 0x91001e41, 0xd1f92800,
    0x4629bf00, 0x462a4620, 0x1e642300, 0x4615419a, 0x40594058, 0xd1e74308, 0x0000bdf8, 0x016e3600,
    0x00000204, 0x400d8000, 0x400d8100, 0xc0c0c0c0, 0x18131800, 0x0f1a2300, 0x400fc000, 0x1f78a400,
    0x0000020c, 0x08954400, 0xfff8e0ff, 0xe3ffffcf, 0x9c7fff80, 0x00004dc2, 0x402a8000, 0x1c9c3800,
    0x09e4f580, 0x000f4240, 0x460cb5f8, 0x485a4e59, 0x46379000, 0x46204635, 0xfa57f003, 0xd0042800,
    0x6ee06e26, 0x6e679000, 0x6d206ea5, 0xd1022800, 0x28006d60, 0x6d60d04f, 0xd0042800, 0x484f2106,
    0x484f6381, 0x6d206386, 0xd0042800, 0x484b2101, 0x484b64c1, 0x211164c6, 0x65014848, 0x65074848,
    0x48462101, 0x48466541, 0x48446545, 0x48446581, 0x48426585, 0x484265c1, 0x484065c5, 0x48406601,
    0x20456605, 0x28085d00, 0x483cd10f, 0x483c6401, 0x483a6405, 0x483a63c1, 0x483863c5, 0x48386381,
    0x48366385, 0x48366341, 0x7b206345, 0xd0022803, 0x28017b20, 0x2111d105, 0x64814830, 0x99004830,
    0x46206481, 0xfa0bf003, 0xd0042800, 0x482b2101, 0x482b6441, 0x6da06447, 0xd1022800, 0x28006de0,
    0x6da0d01e, 0xd0042800, 0x48242104, 0x48246441, 0x21116486, 0x64414821, 0x64474821, 0x481f2101,
    0x481f6401, 0x481d6405, 0x481d63c1, 0x481b63c5, 0x481b6381, 0x48196385, 0x48196341, 0xbf006345,
    0x4601bdf8, 0xbf002004, 0xd1002900, 0x7b0ae014, 0xd10f2a03, 0x6c0a2340, 0x2a00401a, 0x230fd003,
    0x54532278, 0x2279e006, 0x2a015c52, 0x231eda02, 0x54532278, 0xbf002000, 0x4770bf00, 0x48094601,
    0x20006281, 0x46014770, 0x6a804806, 0x20006008, 0x00004770, 0x000010f1, 0x000130f1, 0x401f8124,
    0x401f8298, 0x400f8000, 0x4607b5bf, 0x463849f7, 0x01019102, 0x99020d0a, 0x68091851, 0x0edb06c3,
    0x409a2201, 0x29004011, 0x2101d001, 0x2100e000, 0xd1022900, 0xb0042000, 0x48ecbdb0, 0x90019702,
    0x01389f02, 0x98010d01, 0x68001808, 0x03892103, 0x0b804008, 0xd1022800, 0xfcfff000, 0x2000e000,
    0x48e24604, 0x0d090139, 0x68091809, 0x04122201, 0x0c094011, 0xd0012900, 0xe7dc4620, 0x183848dc,
    0xd0292800, 0x05492101, 0xd0071a40, 0x1a400049, 0x49d8d02d, 0x28001840, 0xe068d16b, 0x304048d3,
    0x90026a00, 0x49d14620, 0x69093140, 0xfc92f004, 0x90009101, 0x9a022300, 0xfa72f004, 0x48cb4605,
    0x07c06b00, 0x28000fc0, 0x2016d002, 0xe0014344, 0x43442014, 0xe04e1964, 0x690048c4, 0x40082102,
    0xd0012800, 0xe0002016, 0x43442014, 0x48bfe043, 0x6b003040, 0x0e400640, 0x48bc9003, 0x69003080,
    0x46209002, 0x318049b9, 0xf0046809, 0x9101fc63, 0x23009000, 0xf0049a02, 0x4605fa43, 0x98034621,
    0x194c4341, 0x304048b1, 0x21036b00, 0x400804c9, 0xd0042800, 0x04c92101, 0xd1041a40, 0x08a4e001,
    0x0864e002, 0xbf00e000, 0x48abbf00, 0x49ab6b00, 0x21014008, 0x1a4003c9, 0x0209d004, 0xd1031a40,
    0xe00208a4, 0xe0000864, 0xbf00bf00, 0x4ca4e003, 0x2400e001, 0xbf00bf00, 0xe75c4620, 0x4604b570,
    0xf7ff48a0, 0x4605ff41, 0xd0062c00, 0xd00e2c01, 0xd0182c02, 0xd12e2c03, 0x4897e021, 0x68003840,
    0x0e890681, 0xf0044628, 0x4605fb08, 0x4892e025, 0x68003840, 0x0212223f, 0x0a014010, 0xf0044628,
    0x4605fafc, 0x488ce019, 0x68003840, 0x0412223f, 0x0c014010, 0xf0044628, 0x4605faf0, 0x4886e00d,
    0x68003840, 0x0612223f, 0x0e014010, 0xf0044628, 0x4605fae4, 0x2500e001, 0xbf00bf00, 0x43452012,
    0xbd704628, 0x4604b570, 0x42404879, 0xfefcf7ff, 0x2c004605, 0x2c01d006, 0x2c02d00e, 0x2c03d018,
    0xe021d12e, 0x38804874, 0x06816b00, 0x46280e89, 0xfac3f004, 0xe0254605, 0x3880486f, 0x223f6b00,
    0x40100212, 0x46280a01, 0xfab7f004, 0xe0194605, 0x38804869, 0x223f6b00, 0x40100412, 0x46280c01,
    0xfaabf004, 0xe00d4605, 0x38804863, 0x223f6b00, 0x40100612, 0x46280e01, 0xfa9ff004, 0xe0014605,
    0xbf002500, 0x2012bf00, 0x46284345, 0xb510bd70, 0x6940485d, 0x06492101, 0x28004008, 0x485ad02e,
    0x21036980, 0x40080309, 0xd0082800, 0x03092101, 0xd00a1a40, 0xd00c1a40, 0xd10f1a40, 0x484ce00f,
    0xf7ff4240, 0x4604fea1, 0xf000e00b, 0x4604fbc6, 0x484ce007, 0xfe98f7ff, 0xe0024604, 0x2400bf00,
    0xbf00bf00, 0x69404848, 0x06d22207, 0x0ec04010, 0x46201c41, 0xfa61f004, 0xe0234604, 0x69804842,
    0x04892103, 0x28004008, 0x2101d008, 0x1a400489, 0x1a40d009, 0x1a40d00c, 0xe00ed111, 0xf7ff4839,
    0x4604fe73, 0x2003e00d, 0xff6cf7ff, 0xe0084604, 0xf7ff2003, 0x4604ff23, 0x4c31e003, 0x2400e001,
    0xbf00bf00, 0xbd104620, 0x4604b510, 0xd1002c00, 0x2299e004, 0xa03aa12d, 0xf9a4f004, 0x07802001,
    0x61884925, 0x483abf00, 0x21016b00, 0x40080409, 0xd0f82800, 0x49204608, 0xbf006148, 0x6900481e,
    0x03c92101, 0x28004008, 0x0048d0f8, 0x6188491a, 0x2001bd10, 0x49180780, 0x47706148, 0xd1032800,
    0x4a2b2110, 0xe0026351, 0x4a292110, 0x47706391, 0x6b004827, 0x43082101, 0x63084925, 0x48244770,
    0x08406b00, 0x49220040, 0x47706308, 0xf7ffb510, 0x4604ff5e, 0x6940480c, 0x02922207, 0x0a804010,
    0x46201c41, 0xf9e9f004, 0x0000bd10, 0x400d8000, 0xffeffff3, 0xff8ffff7, 0x400d8140, 0x00808000,
    0x1dcd6500, 0x0030000d, 0x400fc000, 0x2e5c2e2e, 0x2e2e5c2e, 0x5c2e2e5c, 0x69766564, 0x5c736563,
    0x584d494d, 0x30315452, 0x645c3531, 0x65766972, 0x665c7372, 0x635f6c73, 0x6b636f6c, 0x0000632e,
    0x70796221, 0x58737361, 0x4f6c6174, 0x00006373, 0x400d8240, 0x48feb510, 0x21406940, 0x28004008,
    0x48fbd00f, 0x21806940, 0x28004008, 0x2001d004, 0xfec8f7ff, 0xe0074604, 0xf7ff2002, 0x4604fe7f,
    0xf7ffe002, 0x4604ff04, 0x694048f1, 0x04122207, 0x0c004010, 0x46201c41, 0xf98ff004, 0x46204604,
    0xb510bd10, 0xff92f7ff, 0x48e94604, 0x22036940, 0x40100212, 0x1c410a00, 0xf0044620, 0xbd10f97e,
    0x4605b570, 0xf004002b, 0x0c15fa9d, 0x1915110d, 0x2b26211d, 0x3f3a3530, 0x534e4944, 0x5d586262,
    0xbf000062, 0xff72f7ff, 0xe0524604, 0xffb2f7ff, 0xe04e4604, 0xffd5f7ff, 0xe04a4604, 0xf84cf000,
    0xe0464604, 0xfaa9f000, 0xe0424604, 0x2401bf00, 0xbf0003e4, 0x48cfe03d, 0xfd76f7ff, 0xe0384604,
    0xf7ff2000, 0x4604fe6f, 0x2001e033, 0xfe6af7ff, 0xe02e4604, 0xf7ff2002, 0x4604fe65, 0x2003e029,
    0xfe60f7ff, 0xe0244604, 0xf7ff48c3, 0x4604fd5d, 0x2000e01f, 0xfe12f7ff, 0xe01a4604, 0xf7ff2001,
    0x4604fe0d, 0x2002e015, 0xfe08f7ff, 0xe0104604, 0xf7ff2003, 0x4604fe03, 0x48b8e00b, 0xfd44f7ff,
    0xe0064604, 0xf7ff48b6, 0x4604fd3f, 0x2400e001, 0xbf00bf00, 0xbd704620, 0x48adb510, 0x214069c0,
    0x28004008, 0xf000d003, 0x4604fa58, 0x2003e003, 0xff86f7ff, 0x48a64604, 0x068069c0, 0x1c410e80,
    0xf0044620, 0x4604f8fa, 0xbd104620, 0x4602b508, 0x3080489f, 0x23036800, 0x4b9d4318, 0x60183380,
    0x680048a0, 0x43182302, 0x60184b9e, 0x90002000, 0xbf00e003, 0x1c409800, 0x4b9b9000, 0x42989800,
    0x489ad3f7, 0x231f6a00, 0x4398021b, 0x43184b98, 0x62184b96, 0xbd082001, 0x69094996, 0x03922203,
    0x22014391, 0x43110412, 0x03927842, 0x039b2303, 0x4311401a, 0x61114a8f, 0x69094611, 0x43912202,
    0x019222c1, 0x78024311, 0x23020052, 0x4311401a, 0x61114a88, 0x4987bf00, 0x0fc96909, 0x290007c9,
    0x4984d0f9, 0x22016909, 0x43910412, 0x61114a81, 0xb5384770, 0x460d4604, 0x90002000, 0x6900487d,
    0x03492101, 0x28004008, 0x487ad006, 0x21406900, 0x49784308, 0xe0026108, 0xf7ff4668, 0x4876ffbd,
    0x00406b00, 0x49740840, 0x46086308, 0x04896b00, 0x49714388, 0x20006308, 0x46086008, 0x21636b00,
    0x43080389, 0x6308496c, 0xbd382001, 0x69004869, 0x43882140, 0x61084967, 0x6b004867, 0x07892101,
    0x49654308, 0x47706308, 0x6b094962, 0x03922203, 0x22014391, 0x43110412, 0x03927b02, 0x039b2303,
    0x4311401a, 0x63114a5b, 0x6b094611, 0x40114a5b, 0x03522201, 0x78024311, 0x0fd207d2, 0x4a554311,
    0x68416311, 0x08890089, 0x32404a52, 0x68816111, 0x08890089, 0x8a416211, 0x0c490449, 0x03d27c02,
    0x03db2301, 0x4311401a, 0x041289c2, 0x04120c12, 0x4a484311, 0x60113240, 0x4946bf00, 0x0fc96b09,
    0x290007c9, 0x4943d0f9, 0x22016b09, 0x43910412, 0x63114a40, 0x20014770, 0x493e0300, 0x47706308,
    0x493c2000, 0x47706108, 0x4601b530, 0x4b392200, 0x6b1b3340, 0x03a42403, 0x240143a3, 0x43230424,
    0x03a47b0c, 0x03ad2503, 0x4323402c, 0x34404c31, 0x684b6323, 0x089b009b, 0x3c804c2c, 0x688b6023,
    0x089b009b, 0x4b2b6123, 0x6b1b3340, 0x3c7e4c2b, 0x24014023, 0x43230364, 0x0664780c, 0x43230e64,
    0x784b4618, 0xd0122b02, 0xd00b2b04, 0xd0042b08, 0xd1102b10, 0x4a22bf00, 0x2301e011, 0x431804db,
    0xe00c4a1f, 0x051b2301, 0x4a1d4318, 0x2301e007, 0x431804db, 0x2301e003, 0x4318051b, 0xbf00bf00,
    0x33404b12, 0x4c166b1b, 0x402343e4, 0x4c0f4313, 0x63233440, 0x33404b0f, 0xbf006318, 0x33404b0d,
    0x0fdb6b1b, 0x2b0007db, 0x4b0ad0f8, 0xe0193340, 0x400fc000, 0x0010000d, 0x0030000d, 0x00e00016,
    0x0070000d, 0x402e0140, 0x00061a80, 0x400d8100, 0x00001701, 0x400d8000, 0x400d9000, 0xffffeffe,
    0x00808000, 0x24016b1b, 0x43a30424, 0x63234c79, 0x2001bd30, 0x49770300, 0x47706308, 0x2100b510,
    0x32804a74, 0x23036a12, 0x439a039b, 0x041b2301, 0x7903431a, 0x2403039b, 0x402303a4, 0x4b6d431a,
    0x621a3380, 0x2a007842, 0x2201d002, 0x43110592, 0x32804a68, 0x23016a12, 0x439a031b, 0x4b65430a,
    0x621a3380, 0x4a63bf00, 0x6a123280, 0x07d20fd2, 0xd0f82a00, 0x32804a5f, 0x23016a12, 0x439a041b,
    0x33804b5c, 0xbd10621a, 0x03002001, 0x31804959, 0x47706208, 0x460ab570, 0x4c564601, 0x682434c0,
    0x25bf00ce, 0x43ac40b5, 0x00cd4623, 0x40ac2480, 0x4d50431c, 0x602c35c0, 0x0ea40694, 0x40ac00cd,
    0x4d4c431c, 0x602c35c0, 0x494abd70, 0x680931c0, 0x228000c3, 0x4311409a, 0x32c04a46, 0x47706011,
    0x460ab570, 0x4c434601, 0x6b243480, 0x25bf00ce, 0x43ac40b5, 0x00cd4623, 0x40ac2480, 0x4d3d431c,
    0x632c3580, 0x0ea40694, 0x40ac00cd, 0x4d39431c, 0x632c3580, 0x4937bd70, 0x6b093180, 0x228000c3,
    0x4311409a, 0x32804a33, 0x47706311, 0x4606b5f8, 0xd0002e00, 0x4a30e004, 0xa03da130, 0xfe72f003,
    0x27ffbf00, 0x483e37f5, 0x21016bc0, 0x40080609, 0xd0044288, 0x6bc0483a, 0x49394308, 0x483963c8,
    0x21016800, 0x40080649, 0xd0004288, 0x4a22e005, 0xa122320b, 0xf003a034, 0x4832fe55, 0x07c06800,
    0x28000fc0, 0x482fd105, 0x21016800, 0x492d4308, 0x482c6008, 0x25006844, 0x46314638, 0xffdaf003,
    0x414d1904, 0x43c02000, 0x1b002100, 0xd20c41a9, 0x21004628, 0x41881c64, 0xbf004605, 0x68404821,
    0x462a2100, 0x418a1a20, 0xbf00d3f8, 0x6840481d, 0x1b002100, 0xd3f941a9, 0x482bbdf8, 0x21106b00,
    0x28004008, 0x4829d001, 0x48294770, 0x68004448, 0x0000e7fa, 0x400d8040, 0x00000397, 0x2e5c2e2e,
    0x2e2e5c2e, 0x5c2e2e5c, 0x69766564, 0x5c736563, 0x584d494d, 0x30315452, 0x645c3531, 0x65766972,
    0x665c7372, 0x635f6c73, 0x6b636f6c, 0x0000632e, 0x21205530, 0x6564203d, 0x5f79616c, 0x00007375,
    0xe000edc0, 0xe0001000, 0x5f545744, 0x4c525443, 0x434f4e5f, 0x4e434359, 0x734d5f54, 0x3d21206b,
    0x57442820, 0x433e2d54, 0x204c5254, 0x57442026, 0x54435f54, 0x4e5f4c52, 0x4359434f, 0x5f544e43,
    0x296b734d, 0x00000000, 0x400d8240, 0x016e3600, 0x00000204, 0x4606b570, 0x2504460c, 0x4621bf00,
    0xf0034630, 0x4605fa88, 0xd0002d00, 0x2301e00c, 0x32804622, 0x46302100, 0xffdcf002, 0x28017c60,
    0x2000d101, 0xbf007420, 0x4628bf00, 0xb5f0bd70, 0x4607b089, 0x4615460c, 0x2100461e, 0x70014668,
    0x9002200f, 0x90032001, 0x74054668, 0x98029601, 0x46210100, 0x18423180, 0x46382101, 0xf0029b03,
    0x2001ffb9, 0x46699002, 0xf0024638, 0xb009fe7a, 0xb5ffbdf0, 0x4606b085, 0x2504460c, 0x2c00bf00,
    0xe03dd100, 0x30ce20ff, 0x28005d00, 0x4621d009, 0x9b084630, 0xf7ff9a07, 0x4605ffca, 0xd0002d00,
    0x20ffe02e, 0x5d0030cd, 0xd1132802, 0x30804620, 0x6b426b83, 0x6bc06b01, 0xc70eaf01, 0x46209004,
    0x46013080, 0x6c4b6c8f, 0x6cc96c0a, 0x63436387, 0x63c16302, 0x46304621, 0x9a079b08, 0xffc1f002,
    0x20ff4605, 0x5d0030cd, 0xd1052802, 0x30804620, 0xc98ea901, 0xc08e3030, 0xd0002d00, 0xbf00e000,
    0x4628bf00, 0xbdf0b009, 0xb089b5ff, 0x4615460c, 0x2704461e, 0x4633bf00, 0x4621462a, 0xf7ff9809,
    0x4607ffa8, 0xd0002f00, 0x2100e01b, 0x70014668, 0x9002200e, 0x90032001, 0x74054668, 0x98029601,
    0x46210100, 0x18423180, 0x9b032101, 0xf0029809, 0x2001ff41, 0x46699002, 0xf0029809, 0x4607fe02,
    0xbf00bf00, 0xb00d4638, 0xb5ffbdf0, 0x4606b085, 0x2504460c, 0x2c00bf00, 0xe023d100, 0x30cd20ff,
    0x28025d00, 0x4620d113, 0x69833080, 0x69016942, 0xaf0169c0, 0x9004c70e, 0x30804620, 0x6a8f4601,
    0x6a0a6a4b, 0x61876ac9, 0x61026143, 0x462161c1, 0x9b084630, 0xf0029a07, 0x4605ffbb, 0xd0002d00,
    0xbf00e000, 0x2c00bf00, 0x20ffd00a, 0x5d0030cd, 0xd1052802, 0x30804620, 0xc98ea901, 0xc08e3010,
    0xb0094628, 0xb5f7bdf0, 0x460cb084, 0xbf004615, 0xd0012c00, 0xd9002d01, 0x20ffe065, 0x5d0030c9,
    0xdd012800, 0xe0002001, 0x90032000, 0x28009803, 0xe058d100, 0x6c202140, 0x09864008, 0xf0039804,
    0x2d00fa42, 0x20ffd119, 0x5d0030c9, 0xd00a2800, 0x30c920ff, 0x46325d01, 0xf7fe9804, 0x4621ff73,
    0xf0029804, 0x20fffb2d, 0x5d0030cf, 0xd01c2800, 0x98042101, 0xfa4bf003, 0x20ffe017, 0x5d0030c9,
    0xd0092800, 0x5d012046, 0x98044632, 0xff5af7fe, 0x98044621, 0xfb14f002, 0x30cf20ff, 0x28005d00,
    0x2100d003, 0xf0039804, 0xaa02fa32, 0x98042102, 0xffb0f7fe, 0xd0022e00, 0x08409802, 0xaa019002,
    0x98042100, 0xffa6f7fe, 0x98019902, 0xfcf5f003, 0x210a1c40, 0x08874348, 0xbf00e000, 0x1e7f4638,
    0xd1fa2800, 0xbf00bf00, 0xbdf0b007, 0xb08bb5ff, 0x4617460c, 0x46304626, 0xfa9df002, 0x22009001,
    0x980b4621, 0xff7ff7ff, 0x463bbf00, 0x9a014621, 0xf7ff980b, 0x4605febe, 0xd0002d00, 0x2002e04c,
    0x72084669, 0x90052001, 0x90042009, 0x5d802047, 0xd00c2800, 0x309120ff, 0x28005d80, 0x20ffd007,
    0x5d803092, 0x20ff9004, 0x5d803091, 0x97039005, 0x98014669, 0x980e7608, 0x20ff9007, 0x590030c1,
    0x98049008, 0x46210100, 0x18423180, 0x9b052101, 0xf002980b, 0x2001fe3f, 0xa9029004, 0xf002980b,
    0x4605fd00, 0xd0002d00, 0x463be016, 0x9a014621, 0xf7ff980b, 0x4605fefa, 0xd0002d00, 0x20ffe00c,
    0x5d0030d0, 0xd0062800, 0x4621463b, 0x980b9a01, 0xfebaf7ff, 0xbf004605, 0x980bbf00, 0xf997f003,
    0x46212201, 0xf7ff980b, 0x4628ff1e, 0xbdf0b00f, 0xb08db5f3, 0x2700460e, 0x900c2000, 0x900a900b,
    0x46352400, 0x46682100, 0x20017101, 0x200b9004, 0x46689003, 0x20477501, 0x28005d40, 0x20ffd00c,
    0x5d403095, 0xd0072800, 0x309620ff, 0x90035d40, 0x309520ff, 0x90045d40, 0x46312200, 0xf7ff980d,
    0x2000fef2, 0x462f900b, 0x900a3750, 0xcf01e02b, 0x980c900c, 0xd01f2800, 0x46312200, 0x980d9b0b,
    0xfe27f7ff, 0x2c004604, 0xe01fd000, 0x9002980b, 0x01009803, 0x31804631, 0x21011842, 0x980d9b04,
    0xfdc8f002, 0x90032001, 0x980da901, 0xfc89f002, 0x2c004604, 0xe009d000, 0x980b990c, 0x900b1840,
    0x1c40980a, 0x980a900a, 0xd3d02804, 0x2c00bf00, 0x2000d12e, 0x462f900b, 0x900a3750, 0xcf01e024,
    0x980c900c, 0xd0182800, 0x46312200, 0x980d9b0b, 0xfe6bf7ff, 0x2c004604, 0xe018d000, 0x30d020ff,
    0x28005d80, 0x2200d009, 0x9b0b4631, 0xf7ff980d, 0x4604fe2b, 0xd0002c00, 0x990ce009, 0x1840980b,
    0x980a900b, 0x900a1c40, 0x2804980a, 0xbf00d3d7, 0xf003980d, 0x2201f8fc, 0x980d4631, 0xfe83f7ff,
    0xb00f4620, 0xb5f7bdf0, 0x460eb08a, 0x46354617, 0xf0024628, 0x9000f990, 0x46312200, 0xf7ff980a,
    0xbf00fe72, 0x4631463b, 0x980a9a00, 0xfdb1f7ff, 0x2c004604, 0xe048d000, 0x20009702, 0x71084669,
    0x90042001, 0x90032005, 0x75089800, 0x5d402047, 0xd00c2800, 0x308d20ff, 0x28005d40, 0x20ffd007,
    0x5d40308e, 0x20ff9003, 0x5d40308d, 0x98039004, 0x46310100, 0x18423180, 0x9b042101, 0xf002980a,
    0x2001fd39, 0xa9019003, 0xf002980a, 0x4604fbfa, 0xd0002c00, 0x463be019, 0x9a004631, 0xf7ff980a,
    0x4604fdf4, 0xd0002c00, 0x20ffe00f, 0x5d8030d0, 0xd0092800, 0x4631463b, 0x980a9a00, 0xfdb4f7ff,
    0x2c004604, 0xe000d000, 0xbf00bf00, 0xf003980a, 0x2201f88e, 0x980a4631, 0xfe15f7ff, 0xb00d4620,
    0xb5f7bdf0, 0x460eb08a, 0x46354617, 0xf0024628, 0x9000f922, 0x46312200, 0xf7ff980a, 0xbf00fe04,
    0x4631463b, 0x980a9a00, 0xfd43f7ff, 0x2c004604, 0xe048d000, 0x20009702, 0x71084669, 0x90042001,
    0x90032008, 0x75089800, 0x5d402047, 0xd00c2800, 0x309d20ff, 0x28005d40, 0x20ffd007, 0x5d40309e,
    0x20ff9003, 0x5d40309d, 0x98039004, 0x46310100, 0x18423180, 0x9b042101, 0xf002980a, 0x2001fccb,
    0xa9019003, 0xf002980a, 0x4604fb8c, 0xd0002c00, 0x463be019, 0x9a004631, 0xf7ff980a, 0x4604fd86,
    0xd0002c00, 0x20ffe00f, 0x5d8030d0, 0xd0092800, 0x4631463b, 0x980a9a00, 0xfd46f7ff, 0x2c004604,
    0xe000d000, 0xbf00bf00, 0xf003980a, 0x2201f820, 0x980a4631, 0xfda7f7ff, 0xb00d4620, 0xb5f0bdf0,
    0x4607b08b, 0x4614460d, 0x2004461e, 0xbf009001, 0xd1002c00, 0x2124e012, 0xf003a802, 0x2003faec,
    0x72084669, 0x90042001, 0x94099005, 0x9503960a, 0x4638a902, 0xfb45f002, 0xbf009001, 0x9801bf00,
    0xbdf0b00b, 0xb08fb5ff, 0x4616460c, 0x2004461d, 0xbf00900e, 0x20002700, 0x900b900a, 0x900d900c,
    0x28056830, 0x6870d202, 0xd31b2840, 0x74202001, 0x02406c30, 0x00030f40, 0xfbe4f003, 0x0b050f07,
    0x0907060d, 0xbf00000f, 0x2702bf00, 0x2704e009, 0x2701e007, 0x2703e005, 0x2700e003, 0x74202000,
    0xe002bf00, 0x05008828, 0x2f000f07, 0x2000d070, 0x90029000, 0x46682100, 0x21037501, 0x90087101,
    0x90092001, 0x90049003, 0xd00a2f01, 0xd0042f02, 0xd00a2f03, 0xd10c2f04, 0xbf00e000, 0x900a48fd,
    0x48fce008, 0x900a3830, 0x48fae004, 0x900a300a, 0xbf00e000, 0x2301bf00, 0x4619aa0a, 0xf002980f,
    0xa901fc19, 0xf002980f, 0x900efadc, 0x2800980e, 0xe071d000, 0x01006828, 0x28000f00, 0x88a8d008,
    0x0e000400, 0xd0032800, 0x040088a8, 0x90000e00, 0x74202000, 0xd0062f01, 0xd0172f02, 0xd0382f03,
    0xd1462f04, 0x2140e025, 0x40089800, 0xd10c2800, 0x20c049e1, 0x21c35101, 0x40089800, 0x21409000,
    0x43089800, 0x20019000, 0xe0357420, 0x98002102, 0x28004008, 0x49d9d10b, 0x510120c0, 0x98002102,
    0x90004308, 0x02009800, 0x20019000, 0xe0237420, 0x2102e02f, 0x40089800, 0xd1092800, 0x313049ce,
    0x510120c0, 0x98002102, 0x90004308, 0x74202001, 0x2180e012, 0x40089800, 0xd1092800, 0x313d49c7,
    0x510120c0, 0x98002180, 0x90004308, 0x74202001, 0x2000e002, 0xbf007420, 0x7c20bf00, 0xd0072800,
    0x75202001, 0x75612104, 0x61a09800, 0x74602001, 0x2000bf00, 0xbf00900e, 0x980ebf00, 0xbdf0b013,
    0xb08fb5f0, 0x460c4606, 0x25044617, 0x2000bf00, 0x900b9001, 0x900d900c, 0x9003900e, 0x46682100,
    0x21037601, 0xa8017201, 0x20019009, 0x9004900a, 0x48a89005, 0x900b3050, 0xaa0b2301, 0x46304619,
    0xfb78f002, 0x4630a902, 0xfa3bf002, 0x2d004605, 0xe01fd000, 0x980121f7, 0x90014008, 0x74202001,
    0x04009801, 0x040921ff, 0x499d4008, 0x06894308, 0x20ff1841, 0x51013061, 0x75202001, 0x7561210e,
    0x61a09801, 0x74602004, 0x20ff2101, 0x550130d0, 0xbf002500, 0x4628bf00, 0xbdf0b00f, 0xb098b5f7,
    0x2504460c, 0x01e82700, 0x20009010, 0x7308a910, 0x20019011, 0xbf009012, 0xd1002c00, 0x4889e0ec,
    0x46014478, 0x68c0c90e, 0xc60eae07, 0x2003900a, 0x7708a908, 0x9016a803, 0x90172010, 0x01009811,
    0x31804621, 0x21011842, 0x98189b12, 0xfb22f002, 0x90112001, 0x9818a90f, 0xf9e3f002, 0x2d004605,
    0xe0c9d000, 0x90022001, 0x1c409803, 0xd10b2800, 0x1c409804, 0xd1072800, 0x1c409805, 0xd1032800,
    0x1c409806, 0xd00a2800, 0xa9032210, 0xf003a807, 0x2800f917, 0x4d6cd001, 0x2000e0ae, 0x98029002,
    0xd04a2800, 0x461a2300, 0x98184621, 0xfb41f7ff, 0x2d004605, 0xe09fd000, 0x90112009, 0xa9082002,
    0xa8077708, 0x20109014, 0x98119015, 0x46210100, 0x18423180, 0x9b122101, 0xf0029818, 0x2001fadb,
    0xa90f9011, 0xf0029818, 0x4605f99c, 0xd0002d00, 0x2300e082, 0x4621461a, 0xf0029818, 0x4605fb79,
    0xd0002d00, 0x2000e078, 0x20039011, 0x7708a908, 0x9016a803, 0x90172010, 0x9818a90f, 0xf981f002,
    0x2d004605, 0xe067d000, 0xa9032210, 0xf003a807, 0x2800f8c7, 0x4d44d001, 0x2032e05e, 0x20009001,
    0x26029000, 0x6c202140, 0x64204308, 0x5d012046, 0x98182201, 0xfb46f7fe, 0x2000e03f, 0x900c900b,
    0x900e900d, 0x06092101, 0x42886d20, 0x4837d902, 0xe001900b, 0x900b4836, 0x30ffb2f0, 0x300230ff,
    0x0309210b, 0x21011840, 0x18400489, 0x184001c9, 0x06892129, 0x900c4308, 0xaa0b2301, 0x98182100,
    0xfa78f002, 0x90112000, 0xa9082003, 0xa8037708, 0x20109016, 0xa90f9017, 0xf0029818, 0x4605f932,
    0xd0002d00, 0x2210e011, 0xa807a903, 0xf878f003, 0xd1012800, 0xe0082701, 0x2f001c76, 0x9800d105,
    0x90001c40, 0x42889901, 0xbf00d3b7, 0xd0022f00, 0x6006981a, 0xbf00e000, 0x2f00bf00, 0x4d12d100,
    0xb01b4628, 0xb5ffbdf0, 0x4604b088, 0x4629460d, 0x46283108, 0x9007305c, 0x900568e8, 0x0fc09805,
    0x280007c0, 0x9805d016, 0x20011ec3, 0x90064098, 0x0000e014, 0x24010435, 0x20010401, 0x20020401,
    0x00000481, 0x000035c4, 0x00004e2a, 0x8a2004ee, 0x8a1804ed, 0x1c409805, 0x900608c0, 0x65209806,
    0x28406868, 0x23ffd205, 0x20ff3301, 0x510330c1, 0x8d08e010, 0x0f030600, 0x40982001, 0x23019004,
    0x980403db, 0xd1014298, 0xe00011c3, 0x20ff9b04, 0x510330c1, 0x900348fc, 0x90022000, 0xe01cbf00,
    0x460b0046, 0x5d9b331c, 0xd0152b00, 0x460f0046, 0x5dbf371c, 0x40be2601, 0x9e034633, 0xd20142b3,
    0x90009303, 0x42b39e02, 0x2601d905, 0x42b30536, 0x9302d201, 0xbf009001, 0x28041c40, 0x23ffd3e0,
    0x980333c5, 0x23ff5118, 0x980233d1, 0x9b025118, 0x42989803, 0x2301d104, 0x30ca20ff, 0xe0035503,
    0x20ff2300, 0x550330ca, 0x061b2301, 0x42986d20, 0x2058d935, 0x28005d40, 0x9807d00b, 0x9b001d00,
    0x60105cc0, 0x1d009807, 0x5cc39b01, 0x6003980b, 0x9800e034, 0x46080043, 0x1818301c, 0x28207840,
    0x28d8d002, 0xe002d106, 0x60102021, 0x20dce002, 0xbf006010, 0x9801bf00, 0x46080043, 0x1818301c,
    0x28207840, 0x28d8d002, 0xe003d108, 0x980b2321, 0xe0036003, 0x980b23dc, 0xbf006003, 0x9800e00e,
    0x46080043, 0x1818301c, 0x60107840, 0x00439801, 0x301c4608, 0x78431818, 0x6003980b, 0xb00c2000,
    0xb5ffbdf0, 0x460cb08d, 0x2004461f, 0xbf00900c, 0x4605980f, 0x980f3508, 0x900b305c, 0x900a2000,
    0x02006838, 0x28010f00, 0x6828d107, 0x0fc00300, 0x980a900a, 0xd1002800, 0x2000e288, 0x78387320,
    0x0f090701, 0x55012046, 0x90092000, 0x20189008, 0x20009005, 0xab069004, 0x4620aa07, 0xf7ff990f,
    0x2101ff02, 0x6d200609, 0xd9014288, 0x90052020, 0x04008838, 0x90030f00, 0x28009803, 0x463bd165,
    0x9a0f4621, 0xf7ff980d, 0x900cfcbd, 0x2800980c, 0xe25bd000, 0x02806828, 0x28000fc0, 0x2002d00a,
    0x7a289004, 0x0f400600, 0x7a289008, 0x0ec006c0, 0xe00c9009, 0x02406828, 0x28000fc0, 0x68a8d007,
    0x0f400200, 0x68a89008, 0x0ec002c0, 0x98059009, 0xd1322820, 0x3040980f, 0x28007e00, 0x980bd024,
    0x05c08800, 0x28000fc0, 0x4980d007, 0x301120ff, 0x497f5101, 0x51011d00, 0x980be028, 0x06007800,
    0x28000fc0, 0x497bd007, 0x301120ff, 0x49785101, 0x51011d00, 0x4977e01a, 0x20ff3922, 0x51013011,
    0x1d004975, 0xe0115101, 0x39224972, 0x301120ff, 0x49715101, 0x51011d00, 0x4970e008, 0x301120ff,
    0x496d5101, 0x51011d00, 0xe1ffe000, 0x73202001, 0x20b0496b, 0x496b5101, 0x51012090, 0xb2c09807,
    0x02892101, 0x99051840, 0x22ff0409, 0x40110412, 0x21014308, 0x184106c9, 0x510120d0, 0xb2c09806,
    0x02892101, 0x99051840, 0x40110409, 0x21014308, 0x184106c9, 0x300120ff, 0x21235101, 0x20ff0149,
    0x51013031, 0x01006838, 0x28000f00, 0x7938d00d, 0xd00a2800, 0x90082000, 0x90097938, 0x2800980a,
    0x9809d02a, 0x90090040, 0x980ae026, 0xd0232800, 0x01006838, 0x28010f00, 0x9805da1e, 0xd1062820,
    0x39214944, 0x51012080, 0x20844947, 0x46215101, 0xf7ff980d, 0x900cf88f, 0x2800980c, 0xe1a5d000,
    0x4621aa09, 0xf7ff980d, 0x900cfd29, 0x2800980c, 0xe19bd000, 0x90082000, 0x28189805, 0x980ad11a,
    0xd0012800, 0xe04526ed, 0x02806828, 0x28000fc0, 0x8928d003, 0x0e060400, 0x6828e03c, 0x0fc00240,
    0xd0022800, 0x0e0668a8, 0x2603e034, 0x90092000, 0xe02f9008, 0x90042002, 0x2800980a, 0x26eed001,
    0x980fe028, 0x7e003040, 0xd0112800, 0x7800980b, 0x0fc00680, 0xd0012800, 0xe01b26ec, 0x7800980b,
    0x0fc006c0, 0xd0152800, 0x2000266c, 0xe0119004, 0x02806828, 0x28000fc0, 0x26ecd001, 0x6828e00a,
    0x0fc00240, 0xd0012800, 0xe003266c, 0x20002613, 0x90089009, 0x2800980a, 0x2140d028, 0x43086c20,
    0xb2f06420, 0x18400109, 0x04099905, 0x041222ff, 0x43084011, 0x06492101, 0x21111840, 0x184106c9,
    0x51012080, 0x0000e027, 0x00ffffff, 0x0a20043e, 0x00002204, 0x08200434, 0x00002004, 0x08180402,
    0x00000406, 0x24040405, 0x00012404, 0x2101b2f0, 0x18400289, 0x04099905, 0x041222ff, 0x43084011,
    0x06099904, 0x06122203, 0x43084011, 0x06c92101, 0x20801841, 0x20005101, 0x78389002, 0x0f000600,
    0xd1062800, 0x98089909, 0x90091808, 0x90082000, 0x7838e06e, 0x0f000600, 0xd1682801, 0x6840980f,
    0xd3042840, 0x05808f28, 0x28000fc0, 0x9909d106, 0x18089808, 0x20009009, 0xe0599008, 0x03006ba8,
    0x90010f00, 0x07c09801, 0x28000fc0, 0x2104d104, 0x40089801, 0xd0022800, 0x900220a5, 0x2102e011,
    0x40089801, 0xd00c2800, 0x4621463b, 0x980d9a0f, 0xfc06f7ff, 0x980c900c, 0xd0002800, 0x2001e0c6,
    0x8f289002, 0x0e800400, 0x21029000, 0x40089800, 0xd1042800, 0x98002108, 0x28004008, 0x49f7d00d,
    0x307120ff, 0x1d005101, 0x98055101, 0xd1192820, 0x20ff1409, 0x51013079, 0x9800e014, 0x0fc007c0,
    0xd1042800, 0x98002110, 0x28004008, 0x49ecd009, 0x307120ff, 0x1d005101, 0x49ea5101, 0x51011d00,
    0xe093e000, 0x20ff2101, 0x550130ce, 0xe08de000, 0x28009808, 0x9809d12f, 0xd0282800, 0x2800980a,
    0x9809d012, 0x30ffb2c0, 0x300230ff, 0x0309210b, 0x21011840, 0x18400489, 0x184001c9, 0x06892129,
    0x21844308, 0xe0635108, 0xb2c09809, 0x30ff30ff, 0x21033002, 0x18400309, 0x04892101, 0x01c91840,
    0x21091840, 0x43080689, 0x51082184, 0x49cee050, 0x51012084, 0x990ae04c, 0xd0242900, 0x29019908,
    0x2026d104, 0x09099902, 0xe0009102, 0x99022027, 0x31ffb2c9, 0x310231ff, 0x233f0282, 0x401a029b,
    0x9a094311, 0x23ff0412, 0x401a041b, 0x22014311, 0x18890652, 0x0712220b, 0x2184188a, 0x4abb510a,
    0x510a2188, 0x9908e023, 0xd1042901, 0x99022006, 0x91020909, 0x2007e000, 0xb2c99902, 0x31ff31ff,
    0x02823102, 0x029b233f, 0x4311401a, 0x04129a09, 0x041b23ff, 0x4311401a, 0x06522201, 0x22031889,
    0x188a0712, 0x510a2184, 0x21884aa9, 0xbf00510a, 0x2800980a, 0x2101d007, 0x30cf20ff, 0x21405501,
    0x43086c20, 0x20006420, 0xbf00900c, 0x980cbf00, 0xbdf0b011, 0xb09ab5f7, 0x4f9e460c, 0x2c00bf00,
    0x2704d101, 0x2308e08e, 0x2100aa18, 0xf7ff981a, 0x4607fa26, 0xd0002f00, 0x4997e084, 0x42889818,
    0x4f94d001, 0xa818e07e, 0x60207900, 0x7980a818, 0x90171c40, 0x280a9817, 0x200ad901, 0x9817e000,
    0x25089002, 0x2800981c, 0x022dd000, 0x00c39802, 0x4629aa03, 0xf7ff981a, 0x4607fa02, 0xd0002f00,
    0x2164e060, 0xf0024620, 0x2600fcf6, 0x00f0e055, 0x5c08a903, 0xaa0300f1, 0x79c91889, 0x18400209,
    0x21ff9001, 0x98010209, 0xd0034288, 0x9801497b, 0xd1414288, 0x20022500, 0x022de007, 0xaa0300f1,
    0x1d091889, 0x430d5c09, 0x28001e40, 0x00f1daf5, 0x1808a803, 0x008078c0, 0x981c9000, 0xd0002800,
    0x21ff022d, 0x98010209, 0xd1124288, 0x28509800, 0x2050d901, 0x46229000, 0x46293208, 0x981a9b00,
    0xf9bdf7ff, 0x2f004607, 0xe019d000, 0x60609800, 0x4962e011, 0x42889801, 0x4622d10d, 0x4629325c,
    0x981a9b00, 0xf9abf7ff, 0x2f004607, 0xe007d000, 0x20582101, 0xbf005501, 0x98021c76, 0xd3a64286,
    0xbf00bf00, 0x4638bf00, 0xbdf0b01d, 0xb0a9b5f0, 0x460c4607, 0x26044615, 0x90252000, 0x90279026,
    0x22309028, 0x4479494e, 0xf7fda819, 0xbf00fce9, 0x03006828, 0x28000f00, 0xe044d000, 0x20452104,
    0x49485501, 0x51012080, 0x20844947, 0x46215101, 0xf7fe4638, 0x4606fe1f, 0xd0002e00, 0x7828e033,
    0x0f000600, 0xd1122801, 0x20ff4934, 0x51013071, 0x51011d00, 0x1d001409, 0x23005101, 0x4621461a,
    0xf7fe4638, 0x4606fe24, 0xd0002e00, 0x6828e01b, 0x0f000300, 0xa9190100, 0x23011842, 0x46384619,
    0xfde8f001, 0x46692200, 0xf7ff4638, 0x4606ff13, 0xd0002e00, 0x462be007, 0x4621466a, 0xf7ff4638,
    0x4606fc60, 0xbf00bf00, 0xb0294630, 0xb5f0bdf0, 0x4607b08b, 0x4615460c, 0x2103461e, 0x72014668,
    0x90042000, 0x90052001, 0x90030060, 0x46682100, 0x95097601, 0xa902960a, 0xf0014638, 0x9001fc82,
    0xb00b9801, 0xb5f0bdf0, 0x4607b08b, 0x4615460c, 0x2102461e, 0x72014668, 0x90042001, 0x00609005,
    0x21009003, 0x76014668, 0x96089507, 0x4638a902, 0xfc67f001, 0x98019001, 0xbdf0b00b, 0x06ff06ff,
    0x06000600, 0x32101e00, 0x00002404, 0x7c01a604, 0x7c012604, 0x00004e27, 0x50444653, 0x0000ff84,
    0x00002bae, 0x08180403, 0x00012404, 0xb087b5f0, 0x460c4607, 0x4dfe4616, 0x2003bf00, 0x210873e0,
    0x55012045, 0x64202058, 0x73202003, 0xd0032e00, 0x6c202101, 0x64204308, 0x20472101, 0x210f5501,
    0x5301207c, 0x207e2101, 0x02495301, 0x30c120ff, 0x02495101, 0x51011d00, 0x20ff5901, 0x510130d1,
    0x1fc02101, 0x46215501, 0xf7fe4638, 0x4605fd63, 0xd0002d00, 0x2000e13e, 0x90049003, 0x90069005,
    0x900348e4, 0x900448e4, 0xaa032301, 0x46384619, 0xfd40f001, 0x90032000, 0x90059004, 0x48dd9006,
    0x90033080, 0x900448dd, 0xaa032301, 0x46382100, 0xfd30f001, 0x02c02013, 0x23029002, 0x49d8aa02,
    0xf7ff4638, 0x4605ff70, 0xd0002d00, 0x2308e112, 0x2110466a, 0xf7ff4638, 0x4605ff4a, 0xd0002d00,
    0x9801e108, 0x9001b280, 0x980049ce, 0xd1044288, 0x02092159, 0x42889801, 0x4dc5d001, 0x2308e0fa,
    0x2127466a, 0xf7ff4638, 0x4605ff32, 0xd0002d00, 0x9800e0f0, 0x9000b280, 0x0a009800, 0x98009000,
    0xd3072817, 0x28209800, 0x2001d804, 0x40889900, 0xe0016520, 0xe0dd4db6, 0x0300200f, 0x23029002,
    0x2100aa02, 0xf7ff4638, 0x4605ff2e, 0xd0002d00, 0x49b0e0d0, 0x20803180, 0x49b35101, 0x51012084,
    0x208849b2, 0x49b25101, 0x51012090, 0x209449b1, 0x49b15101, 0x51012098, 0x209c49b0, 0x49a55101,
    0x20a03180, 0x49ae5101, 0x510120a4, 0x20a849a7, 0x49a75101, 0x510120b0, 0x20b449a6, 0x49a65101,
    0x510120b8, 0x20bc49a3, 0x49a15101, 0x510120c0, 0x20c449a4, 0x49a45101, 0x510120c8, 0x20cc49a1,
    0x499b5101, 0x301120ff, 0x499a5101, 0x51011d00, 0x1d004999, 0x499d5101, 0x51011d00, 0x3920498d,
    0x51011d00, 0x1d00499a, 0x49915101, 0x510120d0, 0x20d44990, 0x49905101, 0x510120d8, 0x20dc4995,
    0x498b5101, 0x510120e0, 0x20e4498a, 0x498a5101, 0x510120e8, 0x20ec4987, 0x49855101, 0x510120f0,
    0x20f44988, 0x49885101, 0x510120f8, 0x20fc4985, 0x49785101, 0x1d003920, 0x49875101, 0x51011d00,
    0x1d004986, 0x497a5101, 0x303120ff, 0x49795101, 0x51011d00, 0x1d004978, 0x497e5101, 0x51011d00,
    0x1d004973, 0x49735101, 0x51011d00, 0x1d004972, 0x49705101, 0x51011d00, 0x1d00496d, 0x49715101,
    0x51011d00, 0x1d004970, 0x496e5101, 0x51011d00, 0x1d004967, 0x49675101, 0x51011d00, 0x1d004966,
    0x496f5101, 0x51011d00, 0x20ff2102, 0x55013089, 0x1c402103, 0x21025501, 0x55011f40, 0x1c402101,
    0x21025501, 0x309120ff, 0x21095501, 0x55011c40, 0x1f402104, 0x21055501, 0x55011c40, 0x1dc02104,
    0x210b5501, 0x55011c40, 0x20ff2101, 0x550130cd, 0xbf002500, 0x4628bf00, 0xbdf0b007, 0xb094b5f7,
    0x4615460c, 0x90132004, 0x2d00bf00, 0xe0fbd100, 0x78297868, 0xd1024288, 0x90132000, 0x2603e0f4,
    0x21082701, 0x55012045, 0x28427868, 0x7868d002, 0xd1032841, 0x21042602, 0x55012045, 0x28827868,
    0x7868d002, 0xd1002842, 0x20002721, 0x2100900b, 0x7601a808, 0x20017201, 0x2120900c, 0xf002a802,
    0x78e8fa3a, 0xd01a2806, 0xd02f2807, 0xd1792808, 0x21030230, 0x40080209, 0x43082106, 0x02892101,
    0x90021840, 0x21030230, 0x40080209, 0x430821ff, 0x02892101, 0x90061840, 0x900d2002, 0x0230e07d,
    0x02092103, 0x21664008, 0x21014308, 0x18400289, 0x02309002, 0x02092103, 0x21994008, 0x21014308,
    0x18400289, 0x20029006, 0xe066900d, 0x21030230, 0x40080209, 0x43082166, 0x223f02b9, 0x40110292,
    0x21994308, 0x18400409, 0x22030631, 0x40110612, 0x06b94308, 0x06890e89, 0x90024308, 0x14110230,
    0x21994008, 0x02b94308, 0x0292223f, 0xe02a4011, 0x00004e29, 0x8b188720, 0xa3028f10, 0xa7048f10,
    0x00000555, 0x52005100, 0xb3068f10, 0x0000a704, 0x87008700, 0x87aa8700, 0x87058700, 0x87708700,
    0xb70b8f10, 0x87558700, 0x87028700, 0x87a08700, 0xa3808f10, 0x87808700, 0x87008f10, 0x00008730,
    0x87108700, 0x4308e016, 0x04492133, 0x06311840, 0x06122203, 0x43084011, 0x0e8906b9, 0x43080689,
    0x2f219006, 0x2140d103, 0x43086c20, 0x20026420, 0xe002900d, 0x900d2001, 0xbf00bf00, 0x20462101,
    0x06085501, 0x48fe6520, 0x48fe6020, 0x20036060, 0x73a07360, 0x30c920ff, 0x200a5501, 0x46216760,
    0xf7fe9814, 0x9013fb37, 0x28009813, 0xe01bd000, 0x2101aa02, 0x98149b0d, 0xfb1cf001, 0x9814a90a,
    0xf9dff001, 0x217d9013, 0x48ef00c9, 0x68004448, 0xf973f002, 0xe0039001, 0x9801bf00, 0x90011e40,
    0x28009801, 0xbf00d1f8, 0x9813bf00, 0xbdf0b017, 0xb09eb5f7, 0x48e5460c, 0x9820901d, 0x02006800,
    0x28050f00, 0x2001d101, 0x2000e000, 0x22309011, 0x447949df, 0xf7fda805, 0xbf00f9b3, 0x88009820,
    0x0f070400, 0x68009820, 0x0f050300, 0xd0042f00, 0xd0022f03, 0x901d2004, 0x2d00e267, 0x2d03d004,
    0x2004d002, 0xe260901d, 0xd1042d03, 0xd1022f00, 0x901d2004, 0x2d00e259, 0xa802d11a, 0xfd43f7fd,
    0x2000901d, 0xfd3af7fd, 0x2800981d, 0xaa02d109, 0x981e4621, 0xfeb2f7ff, 0x981d901d, 0xd0002800,
    0x2000e243, 0x20107320, 0x20036420, 0xe011901c, 0xd10f2d03, 0x73202003, 0x64202010, 0x68009820,
    0x0f000200, 0xd1032804, 0x6c202140, 0x64204308, 0x901c2006, 0xd1032f00, 0xd1012d00, 0x90112001,
    0xd1032f03, 0x20452108, 0xe0025501, 0x20452101, 0x46215501, 0xf7fe981e, 0x901dfa9d, 0x2800981d,
    0xe212d000, 0x90042000, 0xd1012d00, 0xe0079004, 0x28009811, 0x2001d002, 0xe0019004, 0x90042002,
    0x01009804, 0x1842a905, 0x46192301, 0xf001981e, 0x2000fa71, 0x21009015, 0x7001a818, 0xa8102103,
    0x20017401, 0x90179016, 0x901ba812, 0x981ea914, 0xf927f001, 0x981d901d, 0xd0002800, 0xa812e1e5,
    0x98039003, 0x28c27800, 0xe1ded000, 0xd1072d00, 0x78819803, 0x20013930, 0x40880400, 0xe0066520,
    0x79409803, 0x21013830, 0x40810409, 0x21016521, 0x20ff0309, 0x510130c5, 0x1f001109, 0x02095101,
    0x30d120ff, 0x20035101, 0x20107320, 0x21016420, 0x40489811, 0x2d004606, 0x2f03d156, 0x2001d154,
    0x20027420, 0x20017460, 0x21068260, 0x75207561, 0x46682100, 0x21077201, 0x2e0072c1, 0x2082d01e,
    0x72484669, 0x61a02002, 0x20e04972, 0x49715101, 0x20e43972, 0x49705101, 0x510120e8, 0x78009820,
    0x0f000600, 0xd1042803, 0x20ff2101, 0x550130cb, 0x2100e026, 0x30cb20ff, 0xe0215501, 0x46692081,
    0x20027248, 0x496361a0, 0x510120e0, 0x20e44963, 0x49615101, 0x510120e8, 0x77202001, 0x20212107,
    0x21015501, 0x55012020, 0x63202001, 0x20f04959, 0x49585101, 0x20f43972, 0x49575101, 0x510120f8,
    0xf7fd9802, 0xbf00fc4b, 0x20b04955, 0x49555101, 0x51012090, 0xd11e2f00, 0x73202001, 0x20804952,
    0x49525101, 0x51012084, 0x1d89494f, 0x301120ff, 0x494f5101, 0x51011d00, 0x3115494b, 0x510120d0,
    0x31d04949, 0x300120ff, 0x21235101, 0x20ff0149, 0x51013031, 0x2e00e138, 0x2221d001, 0x2201e000,
    0x2e004610, 0x2222d001, 0x2202e000, 0x2e004611, 0x222cd001, 0x220ce000, 0x2e009202, 0x2229d001,
    0x2209e000, 0x2e009201, 0x2228d001, 0x2208e000, 0x02829200, 0x029b233f, 0x4b31401a, 0x431a3b18,
    0x041b2311, 0x230318d2, 0x18d2061b, 0x0e9b0683, 0x431a069b, 0x511a2380, 0x233f028a, 0x401a029b,
    0x015b2319, 0x2303431a, 0x18d2045b, 0x18d201db, 0x069b9b02, 0x069b0e9b, 0x2384431a, 0x9a01511a,
    0x233f0292, 0x401a029b, 0x009b23c1, 0x2388431a, 0x0282511a, 0x029b233f, 0x4b1e401a, 0x23f9431a,
    0x18d2041b, 0x061b2303, 0x0683431a, 0x069b0e9b, 0x23c0431a, 0x0282511a, 0x029b233f, 0x4b15401a,
    0x431a1e5b, 0x045b237d, 0x230318d2, 0x431a061b, 0x0e9b0683, 0x431a069b, 0x511a23a0, 0xe01b0282,
    0x42464346, 0x56010400, 0x0000020c, 0x00004e29, 0x00002572, 0x04000472, 0x20010400, 0x04020400,
    0x00000406, 0x24040405, 0x0820040c, 0x24043308, 0x00002004, 0x00000306, 0x029b233f, 0x2303401a,
    0x431a021b, 0x18d2041b, 0x0e9b0683, 0x431a069b, 0x511a23a4, 0x233f0282, 0x401a029b, 0x021b2303,
    0x041b431a, 0x068318d2, 0x069b0e9b, 0x23a8431a, 0x9a01511a, 0x233f0292, 0x401a029b, 0x009b23c1,
    0x23ac431a, 0x0282511a, 0x029b233f, 0x4bfa401a, 0x23ed431a, 0x18d2041b, 0x061b2303, 0x0683431a,
    0x069b0e9b, 0x22ff4313, 0x51133211, 0x233f028a, 0x401a029b, 0x015b2319, 0x2301431a, 0x18d2049b,
    0x061b2303, 0x9a0018d3, 0x0e920692, 0x43130692, 0x321522ff, 0x02825113, 0x029b233f, 0x4be6401a,
    0x431a330f, 0x045b236f, 0x230318d2, 0x431a061b, 0x0e9b0683, 0x431a069b, 0x511a23d0, 0x233f028a,
    0x401a029b, 0x015b2319, 0x23d4431a, 0x0282511a, 0x029b233f, 0x23f7401a, 0x431a009b, 0x041b2323,
    0x230318d2, 0x18d3061b, 0x0e920682, 0x43130692, 0x320122ff, 0x028a5113, 0x029b233f, 0x2319401a,
    0x431a015b, 0x330523ff, 0x0282511a, 0x029b233f, 0x231b401a, 0x431a015b, 0x041b239f, 0x230318d2,
    0x431a061b, 0x0e9b0683, 0x4313069b, 0x323122ff, 0x2e005113, 0x2340d003, 0x431a6c22, 0x22026422,
    0x33cd23ff, 0xbf00551a, 0xbf00bf00, 0xb021981d, 0xb5f7bdf0, 0x460cb0a6, 0x4eb84615, 0x49b82220,
    0xa81e4479, 0xff2cf7fc, 0x02006828, 0x28070f00, 0x2001d101, 0x2000e000, 0xbf00901d, 0x03006828,
    0x28000f00, 0x4668d116, 0xfacdf7fd, 0x20004606, 0xfac4f7fd, 0xd1082e00, 0x4621466a, 0xf7ff9826,
    0x4606fc3d, 0xd0002e00, 0x2000e104, 0x20107320, 0xe0036420, 0x73202003, 0x64202050, 0x20452108,
    0x46215501, 0xf7fe9826, 0x4606f845, 0xd0002e00, 0x6828e0f0, 0x0f000300, 0xd1062800, 0xaa1e2301,
    0x98264619, 0xf826f001, 0x6828e00c, 0x0f000300, 0xd1062803, 0xaa222301, 0x98264619, 0xf81af001,
    0xe0d7e000, 0x20452108, 0x22005501, 0x9826a904, 0xf940f7ff, 0x2e004606, 0xe0cbd000, 0xaa03ab02,
    0x4620a904, 0xfdbff7fe, 0x2800981d, 0x2050d103, 0x20036420, 0x49807320, 0x20b031f4, 0x49815101,
    0x51012090, 0x90012000, 0x03006828, 0x28030f00, 0x8828d004, 0x0f000400, 0xd1012803, 0x90012001,
    0x28009801, 0x8828d15a, 0x0f000400, 0xd1542800, 0x061b2301, 0x429a6d22, 0x2020d302, 0xe001218e,
    0x21c22018, 0xb2d29a03, 0x029b2301, 0x040318d2, 0x043f27ff, 0x431a403b, 0x06db2301, 0x22d018d3,
    0x9a025113, 0x2301b2d2, 0x18d2029b, 0x403b0403, 0x2301431a, 0x18d306db, 0x320122ff, 0xb2ca5113,
    0x029b2301, 0x040318d2, 0x431a403b, 0x061b2303, 0x230118d2, 0x18d306db, 0x321122ff, 0x23475113,
    0x1d1201db, 0x9a1d5113, 0xd1062a00, 0x22804b56, 0x4b565113, 0x51132284, 0x4b55e005, 0x51132280,
    0x22844b54, 0x23235113, 0x22ff015b, 0x51133231, 0x22ff2301, 0x551332cf, 0xe04be04b, 0x20a0494e,
    0x494e5101, 0x51012080, 0x20844948, 0x494c5101, 0x510120c0, 0x39dc4949, 0x510120d0, 0x39214947,
    0x300120ff, 0x213b5101, 0x20ff0149, 0x51013031, 0x39eb4942, 0x301120ff, 0x49425101, 0x51011d00,
    0x03006828, 0x28000f00, 0x2100d11f, 0x70014668, 0x70c12106, 0x70412182, 0xf7fd9800, 0x2001f9cf,
    0x20027420, 0x20e77460, 0x210661a0, 0x20017561, 0x82607520, 0x20e04934, 0x49335101, 0x20e43981,
    0x49325101, 0x510120e8, 0x2102bf00, 0x30cd20ff, 0xbf005501, 0x4630bf00, 0xbdf0b029, 0xb0bcb5f7,
    0x481e460c, 0x983e903b, 0x02006800, 0x28090f00, 0x2001d101, 0x2000e000, 0x22504606, 0x44794924,
    0xf7fca827, 0xbf00fded, 0x90262000, 0x20452104, 0x983e5501, 0x03006800, 0x90250f00, 0x8800983e,
    0x0f050400, 0x64202010, 0x28009825, 0x4668d138, 0xf989f7fd, 0x2000903b, 0xf980f7fd, 0x2800983b,
    0x466ad128, 0x983c4621, 0xfaf8f7ff, 0x983b903b, 0xd01f2800, 0x0000e1d9, 0x00000312, 0x00004e29,
    0x00002094, 0x24040405, 0x8b2004fd, 0xa704b306, 0x0b2004fc, 0x27043306, 0xa7040705, 0x8b2007fd,
    0x00000706, 0x0000a304, 0x04000481, 0x00002001, 0x00001e36, 0x20452101, 0x20005501, 0xe0157320,
    0x28029825, 0x2104d105, 0x55012045, 0x73202003, 0x9825e00c, 0xd1082803, 0x21087320, 0x55012045,
    0xd1032e00, 0x90262001, 0xe19ee000, 0xd0042d00, 0xd0022d02, 0xd0002d03, 0x9825e197, 0xd1022800,
    0xd1002d00, 0x2d022601, 0x2104d102, 0x55012045, 0xd1022d03, 0x20452108, 0x2e005501, 0x2140d004,
    0x43886c20, 0xe0036420, 0x6c202140, 0x64204308, 0x983c4621, 0xfeaef7fd, 0x983b903b, 0xd0002800,
    0x2000e173, 0x2e009024, 0x9825d00f, 0xd0022802, 0xd1062803, 0x2001e002, 0xe0059024, 0x90242002,
    0x2000e002, 0xbf009024, 0x9825e00f, 0xd0022802, 0xd1062803, 0x2003e002, 0xe0059024, 0x90242004,
    0x2000e002, 0xbf009024, 0x9824bf00, 0xa9270100, 0x23011842, 0x983c4619, 0xfe6cf000, 0x9a26a90b,
    0xf7fe983c, 0x903bff97, 0x2800983b, 0xe13cd000, 0xaa0aab09, 0x4620a90b, 0xfc15f7fe, 0x73202003,
    0x20b049fd, 0x49fd5101, 0x51012090, 0x9508462f, 0x95069507, 0xd0012e00, 0xe0002002, 0x90052022,
    0xd0012e00, 0xe000200c, 0x9004202c, 0xd0012e00, 0xe0002009, 0x90032029, 0xd0012e00, 0xe0002108,
    0x91022128, 0x21030228, 0x40080209, 0x4308210b, 0x02892101, 0x02c91840, 0x06391840, 0x06122203,
    0x43084011, 0x06899905, 0x06890e89, 0x21804308, 0x98085108, 0x14110200, 0x21044008, 0x99044308,
    0x223f0289, 0x40110292, 0x21014308, 0x18400489, 0x06099906, 0x06122203, 0x43084011, 0x06899903,
    0x06890e89, 0x21844308, 0x02285108, 0x40081411, 0x43082120, 0x18400149, 0x184102c9, 0x40100638,
    0x99054308, 0x0e890689, 0x43080689, 0x510821d0, 0x14110228, 0x21d84008, 0x21014308, 0x18400289,
    0x184002c9, 0x40110639, 0x99054308, 0x0e890689, 0x43010689, 0x300120ff, 0x02285101, 0x40081411,
    0x43082102, 0x18400249, 0x184002c9, 0x40110639, 0x99054308, 0x0e890689, 0x43010689, 0x301120ff,
    0x98075101, 0x14110200, 0x21804008, 0x99024308, 0x223f0289, 0x40110292, 0x21ff4308, 0x51083115,
    0x21030228, 0x40080209, 0x43082160, 0x02892101, 0x21ff1840, 0x51083131, 0xd0012d02, 0xd1252d03,
    0x21030228, 0x40080209, 0x43082105, 0x02892101, 0x02091840, 0x99061840, 0x22030609, 0x40110612,
    0x99034308, 0x0e890689, 0x43080689, 0x510821a0, 0x14110228, 0x21064008, 0x21014308, 0x18400289,
    0x510821c0, 0x20ff2102, 0x550130cd, 0x61a02000, 0xd1042d02, 0x69a00469, 0x61a04308, 0x2d03e006,
    0x2101d104, 0x69a004c9, 0x61a04308, 0xd1052e00, 0x05c92101, 0x430869a0, 0xe00661a0, 0x6c202180,
    0x64204308, 0x20792101, 0x49855501, 0x510120e0, 0x20e44984, 0x20015101, 0x2d007420, 0x2002d002,
    0xe0017460, 0x74602000, 0x75612106, 0x75202001, 0x20008260, 0x71084669, 0xd00e2e00, 0xd1022d02,
    0x71482041, 0x2d03e018, 0x2081d103, 0x71484669, 0x2000e012, 0x71484669, 0x2d02e00e, 0x2042d103,
    0x71484669, 0x2d03e008, 0x2082d103, 0x71484669, 0x2000e002, 0x71484669, 0x46692008, 0x980171c8,
    0xff9cf7fc, 0x903b2000, 0xbf00bf00, 0xb03f983b, 0xb5f8bdf0, 0x460c4607, 0x26044615, 0x2c00bf00,
    0x2d00d001, 0xe05ed100, 0x02492101, 0xf0014620, 0x2101fb72, 0x55012046, 0x65200608, 0x6020485a,
    0x6060485a, 0x73602003, 0x20ff73a0, 0x550130c9, 0x02006828, 0x00030f00, 0xfc84f001, 0x0e07060a,
    0x241d1c15, 0x342d2c25, 0x462abf00, 0x46384621, 0xfebcf7fe, 0xe0274606, 0x46212201, 0xf7fe4638,
    0x4606ff65, 0x2200e020, 0x46384621, 0xff5ef7fe, 0xe0194606, 0x462abf00, 0x46384621, 0xf9d8f7ff,
    0xe0114606, 0x462abf00, 0x46384621, 0xfc61f7ff, 0xe0094606, 0x462abf00, 0x46384621, 0xfd8ef7ff,
    0xe0014606, 0xbf002604, 0x2e00bf00, 0x4620d107, 0xff17f7fc, 0x07017828, 0x20460f09, 0x21015501,
    0x55012044, 0xbf00bf00, 0xbdf84630, 0xb083b5ff, 0x2704460c, 0x2c00bf00, 0xe070d100, 0x30c520ff,
    0x42455900, 0x400d9905, 0x98059906, 0x42401840, 0x31c521ff, 0x42495909, 0x42464008, 0x30ca20ff,
    0x28005d00, 0x20ffd104, 0x590030d1, 0xd1122800, 0x462ae00e, 0x98034621, 0xff15f7fd, 0x2f004607,
    0x4638d002, 0xbdf0b007, 0x30c520ff, 0x19455900, 0xd3ee42b5, 0xe03fe042, 0x30d120ff, 0x46285901,
    0x28004388, 0x2001d101, 0x2000e000, 0x1b709002, 0x98029001, 0xd0202800, 0x30d120ff, 0x98015901,
    0xd81a4281, 0x4621462a, 0xf7fd9803, 0x4607ff5a, 0xd00d2f00, 0xe7d54638, 0x00000406, 0x24040405,
    0x04000471, 0x00002003, 0x42464346, 0x56010400, 0x30d120ff, 0x19455900, 0x462ae00d, 0x98034621,
    0xfed1f7fd, 0x2f004607, 0x4638d001, 0x20ffe7ba, 0x590030c5, 0xbf001945, 0xd3bd42b5, 0xbf00bf00,
    0xe7af4638, 0xb08db5ff, 0x4614460f, 0x20049d16, 0xbf00900c, 0xd0032f00, 0xd0012c00, 0xd1002d00,
    0x9701e02d, 0xf0009801, 0x9002f846, 0x46692003, 0x20017308, 0x20009006, 0x98029005, 0xe01a7708,
    0x42854811, 0xe000d900, 0x46064628, 0x90049810, 0x960b940a, 0x980da903, 0xfad3f000, 0x980c900c,
    0xd0002800, 0x1bade009, 0x19809810, 0x08b09010, 0x19040080, 0x2d00bf00, 0xbf00d1e2, 0xbf00bf00,
    0xb011980c, 0x0000bdf0, 0x0000ffff, 0x22016801, 0x60014311, 0x6801bf00, 0x0fc907c9, 0xd1fa2900,
    0x49fe4770, 0x21026181, 0x477061c1, 0x618149fb, 0x61c12101, 0x46014770, 0x6c082204, 0x28004010,
    0x2001d001, 0x20004770, 0x4601e7fc, 0x6c082220, 0x28004010, 0x2001d001, 0x20004770, 0x4601e7fc,
    0x07c06c08, 0x28000fc0, 0x2001d001, 0x20004770, 0x4601e7fc, 0x6c082208, 0x28004010, 0x2001d001,
    0x20004770, 0x4601e7fc, 0x6c082202, 0x28004010, 0x2001d001, 0x20004770, 0x4601e7fc, 0x6c082240,
    0x28004010, 0x2001d001, 0x20004770, 0x4601e7fc, 0x29002000, 0x008ad103, 0x447b4bd9, 0x47705898,
    0xb08db5f3, 0x2004460c, 0xbf00900c, 0xf7ff980d, 0x4605ffee, 0x900a2001, 0xd0042d00, 0xd0022c00,
    0x28037b20, 0xe0cfdd00, 0x28007b20, 0x2801d006, 0x2802d005, 0x2803d004, 0xe004d110, 0xbf00bf00,
    0x900a2001, 0x4620e00b, 0xffbdf7ff, 0xd0022800, 0x900a2001, 0x2000e001, 0xe000900a, 0xbf00bf00,
    0x2800980a, 0x20ffd004, 0x90073001, 0xe07e9008, 0x2102aa09, 0xf7fc980d, 0x2000fc6d, 0x49b99006,
    0x42889809, 0x2180d306, 0x40086c20, 0xd1012800, 0x90062001, 0x28009806, 0x2079d003, 0x90089007,
    0x2000e064, 0xe05e9005, 0x00419805, 0x30784620, 0x78401808, 0x98059004, 0x46200041, 0x5c403078,
    0x98049003, 0xd13b2800, 0x00419805, 0x30784620, 0x21645c47, 0x2f00434f, 0x4620d127, 0xff7df7ff,
    0xa9029001, 0x980d9a01, 0xfc7cf7fc, 0x28009801, 0x489dd10d, 0xf0019902, 0x0840f980, 0x00c9217d,
    0x214b4348, 0xf0019000, 0x1c46f978, 0x4896e00c, 0xf0019902, 0x0880f972, 0x00c9217d, 0x214b4348,
    0xf0019000, 0x1c46f96a, 0x214be00b, 0xf0014638, 0x4606f964, 0x4370204b, 0xd20242b8, 0xe0001c76,
    0x273f9e04, 0xd90042be, 0x0270463e, 0x0249213f, 0x21ff4008, 0x43083101, 0x00899905, 0x5050aa07,
    0x1c409805, 0x98059005, 0xd39d2802, 0x6828bf00, 0x40082102, 0xd0022800, 0x900b2000, 0x2001e001,
    0x980b900b, 0xd0032800, 0x21026828, 0x60284308, 0x28006d20, 0x6d60d102, 0xd0022800, 0x990720c0,
    0x6da05141, 0xd1022800, 0x28006de0, 0x20c4d002, 0x51419908, 0x2800980b, 0x6828d003, 0x43882102,
    0x20006028, 0xbf00900c, 0x980cbf00, 0xbdf0b00f, 0xb083b5ff, 0x460f4606, 0x2004461d, 0xbf009002,
    0xd0042e00, 0x28009805, 0x2d00d001, 0xe017d100, 0x9905485d, 0xf901f001, 0x98019001, 0x46014368,
    0xf0014638, 0x4604f8fa, 0x1c64e000, 0x99014620, 0x43684348, 0xd3f842b8, 0x20006034, 0xbf009002,
    0x9802bf00, 0xbdf0b007, 0x4606b57f, 0x484f460c, 0x90029003, 0xf7ff4630, 0x4605feda, 0xd0012d00,
    0xd1022c00, 0xb0042004, 0xaa01bd70, 0x46302102, 0xfb80f7fc, 0x2101466a, 0xf7fc4630, 0x2301fb7b,
    0x4943029b, 0x9a01a803, 0xffb2f7ff, 0x029b2301, 0xa802493f, 0xf7ff9a00, 0x493cffab, 0x42889803,
    0x4608d901, 0x49399003, 0x42889802, 0x4608d901, 0x98039002, 0x0c000400, 0x99020400, 0x4308b289,
    0x20006068, 0x21bce7cf, 0x22015809, 0x22bc4311, 0xbf005011, 0x580921bc, 0x0fc907c9, 0xd1f92900,
    0x21b84770, 0x22015809, 0x22b84311, 0xbf005011, 0x580921b8, 0x0fc907c9, 0xd1f92900, 0xbf004770,
    0x580921e0, 0x0fc907c9, 0xd0f92900, 0x21e0bf00, 0x22025809, 0x29024011, 0x4770d1f9, 0x221e6941,
    0x61414311, 0xb5f34770, 0x460cb085, 0x90042004, 0x2000bf00, 0x98059001, 0xfe69f7ff, 0x2f004607,
    0x2c00d001, 0xe078d100, 0x30504620, 0x26009000, 0x9800e06e, 0x90036800, 0x1d009800, 0x98039000,
    0x00b20a81, 0x30604638, 0x7ba05081, 0x0ec006c0, 0x01497b61, 0x0152221f, 0xe00b4011, 0x5af05af0,
    0x0000176a, 0x05f5e100, 0x3b9aca00, 0x0000ffff, 0x1dcd6500, 0x7be14308, 0x220f02c9, 0x401102d2,
    0x46054308, 0xf7ff4620, 0x2800fe14, 0x2001d002, 0x43050280, 0x28006f60, 0xaa02d00f, 0x98052102,
    0xfad8f7fc, 0x6f612301, 0x9a02a801, 0xff10f7ff, 0x04009801, 0x04000c00, 0x00b14305, 0x30704638,
    0x25005045, 0x5d002047, 0xd0192800, 0x309120ff, 0x01405d00, 0x400821e0, 0xd0112800, 0x309220ff,
    0x02005d00, 0x0209210f, 0x43054008, 0x309120ff, 0x1e405d00, 0x21070340, 0x40080349, 0xe0024305,
    0x02002009, 0x00b14305, 0x30804638, 0x1c765045, 0xd38e2e04, 0x90042000, 0xbf00bf00, 0xb0079804,
    0xb5f0bdf0, 0x460d4603, 0xbf002704, 0xd0012b00, 0xd1002d00, 0x2044e025, 0x28015d40, 0x68d8d10c,
    0x00760846, 0x43062040, 0xf7ff4628, 0x2800fd94, 0x2001d001, 0x60de4306, 0x212068d8, 0x60d84308,
    0xe00a2400, 0x461800a1, 0x58403020, 0x400849fd, 0x461a00a1, 0x50503220, 0x2c031c64, 0x2700d3f2,
    0xbf00bf00, 0xbdf04638, 0x4606b570, 0x4630bf00, 0xfdadf7ff, 0x2d004605, 0xe010d100, 0xe00b2400,
    0x462800a1, 0x58403080, 0x07c92101, 0x00a24308, 0x31804629, 0x1c645088, 0xd3f12c04, 0xbf00bf00,
    0xb5f3bd70, 0x460db089, 0x90082004, 0xf7ff9809, 0x4604fd8e, 0x2c00bf00, 0x2d00d004, 0x7828d002,
    0xdd002803, 0x2000e122, 0x7c289007, 0x46209006, 0xfef5f7ff, 0xf7ff9809, 0x4620ffc7, 0xfefef7ff,
    0x686920a0, 0x78285101, 0xd1022802, 0xb2808b28, 0x78289007, 0xd1022803, 0xb2808c28, 0x89289007,
    0x210f0400, 0x40080409, 0x1e497b29, 0x22070609, 0x40110612, 0x99064308, 0x0fc907c9, 0x430807c9,
    0x43089907, 0x21a49007, 0x51089807, 0x28037828, 0x6a2ed167, 0x403869ef, 0xd0002800, 0x2080e0e6,
    0x98059005, 0x90040900, 0x08409805, 0x98049003, 0x00801e40, 0x4008213c, 0x510821b8, 0xf7ff4620,
    0x20b0fea0, 0x21015900, 0x21b04308, 0xe0445108, 0x30f91de0, 0x428e9903, 0x9a03d314, 0x69620891,
    0x401a2320, 0xd00c2a00, 0xc804e001, 0x460ac704, 0x2a001e49, 0x9a03d1f9, 0x69621ab6, 0x431a2320,
    0xe0236162, 0x590921f0, 0x0d490609, 0xd31d42b1, 0xc802e002, 0x1f36c702, 0xd2fa2e04, 0xd0142e00,
    0x91026801, 0x9101a902, 0x21009700, 0x9a01e00a, 0x9a007813, 0x9a017013, 0x92011c52, 0x1c529a00,
    0x1c499200, 0xd3f242b1, 0x2600bf00, 0x22086961, 0x29004011, 0xe002d000, 0x2e00bf00, 0xbf00d1b8,
    0x7828bf00, 0xd0022802, 0x28017828, 0x69aed15a, 0x07b8696f, 0x28000f80, 0xe077d000, 0x90052080,
    0x09009805, 0x98059004, 0x90030840, 0x1e409804, 0x213c0080, 0x21bc4008, 0x46205108, 0xfe23f7ff,
    0x90022000, 0x4620e03a, 0x308130ff, 0x22406961, 0x29004011, 0x9a03d02b, 0x9a030891, 0xd3094296,
    0xcf04e001, 0x460ac004, 0x2a001e49, 0x9a03d1f9, 0xe00c1ab6, 0x2e00e007, 0x1f36dd03, 0xc004cf04,
    0x2200e001, 0x460ac004, 0x2a001e49, 0x9a02d1f3, 0xd1072a00, 0x591222b0, 0x431a2301, 0x511a23b0,
    0x92022201, 0x23406962, 0x6162431a, 0x6961bf00, 0x40112208, 0xd0002900, 0xbf00e002, 0xdcc22e00,
    0xbf00bf00, 0x28007828, 0x20b0d105, 0x21015900, 0x21b04308, 0x46205108, 0xfdf1f7ff, 0x21086960,
    0x28004008, 0x20e4d00f, 0x210f5900, 0x40080609, 0x280e0e00, 0x4854d002, 0xe0039008, 0x1e404852,
    0xbf009008, 0x2000e001, 0xbf009008, 0x9808bf00, 0xbdf0b00b, 0xb085b5ff, 0x4616460c, 0x2004461d,
    0xbf009004, 0xf7ff9805, 0x4607fc52, 0xd0092f00, 0xd0072e00, 0x19622000, 0x23104140, 0x1a9a2100,
    0xd2004181, 0x4638e029, 0xfdb9f7ff, 0x90032000, 0x46389002, 0xfbf5f7ff, 0x900300a0, 0x00801960,
    0x98039002, 0x1df80081, 0x30fa30ff, 0x90011808, 0x6831e009, 0x60019801, 0x1c409803, 0x98019003,
    0x90011d00, 0x99021d36, 0x42889803, 0x4638d3f1, 0xfbdcf7ff, 0x90042000, 0xbf00bf00, 0xb0099804,
    0xb5f0bdf0, 0x4607b08b, 0x4615460c, 0x2004461e, 0xbf00900a, 0xd1002c00, 0x2100e02c, 0x71014668,
    0x90042001, 0x90032003, 0x75054668, 0x20479602, 0x28005d00, 0x20ffd00c, 0x5d003089, 0xd0072800,
    0x308a20ff, 0x90035d00, 0x308920ff, 0x90045d00, 0x01009803, 0x31804621, 0x21011842, 0x9b044638,
    0xff88f7ff, 0x90032001, 0x4638a901, 0xfe49f7ff, 0xbf00900a, 0x980abf00, 0xbdf0b00b, 0x4604b5f8,
    0x90002004, 0x4625bf00, 0x4610460e, 0xd0072b00, 0x0fff07df, 0xd1032f00, 0xd0012c00, 0xd1052900,
    0x0000e013, 0xfcf0ff00, 0x00001b59, 0x7807e008, 0x1c40702f, 0x78071c6d, 0x1c407037, 0x1e9b1c76,
    0xd1f42b00, 0x97002700, 0xbf00bf00, 0xbdf89800, 0xb093b5ff, 0x2004460c, 0xbf009012, 0xd1002c00,
    0x2000e0a6, 0x207c900e, 0x27015b01, 0x207e408f, 0x900f5b00, 0x90049816, 0x46692003, 0x20017308,
    0x90059006, 0x900aa810, 0x28009815, 0x2008d001, 0x2004e000, 0x4669900b, 0x77089815, 0x5d002047,
    0xd00c2800, 0x308520ff, 0x28005d00, 0x20ffd007, 0x5d003086, 0x20ff9005, 0x5d003085, 0x98059006,
    0x46210100, 0x18423180, 0x98059b06, 0x90014601, 0xf7ff9813, 0x6f20ff0f, 0xd0012800, 0xe0002001,
    0x90022000, 0x6f252600, 0xa903bf00, 0xf7ff9813, 0x9012fdc8, 0x28009812, 0xe057d000, 0x28009815,
    0x2308d022, 0xa90caa10, 0xf7ffa80d, 0x980fff77, 0xd00d2800, 0x990d4638, 0x46394388, 0x43919a0c,
    0x28004308, 0x2001d001, 0x2000e000, 0xe020900e, 0x4038980d, 0x4039990c, 0x28004308, 0x2001d001,
    0x2000e000, 0xe014900e, 0x2800980f, 0x4638d009, 0x43889910, 0xd0012800, 0xe0002001, 0x900e2000,
    0x9810e007, 0x28004038, 0x2001d001, 0x2000e000, 0x980e900e, 0xd0162800, 0x28009802, 0x2100d013,
    0x40694608, 0x43014070, 0x207dd00a, 0x210000c0, 0xf82ff7fc, 0x21004630, 0x41881e6d, 0xe0024606,
    0x901248fe, 0x980ee002, 0xd19e2800, 0xbf00bf00, 0x9812bf00, 0xbdf0b017, 0xb08ab5f7, 0x4615460c,
    0xbf002604, 0xf7ff980a, 0x4607faf2, 0xd0012f00, 0xd1002c00, 0x2001e050, 0x59c00240, 0x0f800580,
    0x98099009, 0xd0042800, 0x28027c60, 0x2600d101, 0x2101e042, 0x70014668, 0x7d609501, 0x7d209002,
    0x21009003, 0x74014668, 0x30184620, 0x20049005, 0x462b9006, 0x46212200, 0xf7ff980a, 0x9802feb2,
    0x46210100, 0x18423180, 0x9b032101, 0xf7ff980a, 0x2001fe61, 0x46699002, 0xf7ff980a, 0x4606fd22,
    0xd0002e00, 0x8a60e018, 0xd10d2800, 0x28027c60, 0x7c60d00a, 0xd0072803, 0x2200462b, 0x980a4621,
    0xfef6f7ff, 0xe0064606, 0x21648a60, 0x46024348, 0xf7fb2100, 0xbf00ffbe, 0x4630bf00, 0xbdf0b00d,
    0xb083b5f3, 0x2004460c, 0xbf009002, 0xd1002c00, 0x2500e01a, 0x30504620, 0x26009001, 0x9801e010,
    0x9001c880, 0xd00a2f00, 0x4621462a, 0xf7ff9803, 0x9002ff83, 0x28009802, 0xe003d000, 0x1c7619ed,
    0xd3ec2e04, 0xbf00bf00, 0x9802bf00, 0xbdf0b005, 0xb08cb5f7, 0x4617460c, 0xbf002604, 0xf7ff980c,
    0x900bfa66, 0x2800980b, 0x2c00d001, 0xe077d100, 0x46682101, 0x97037201, 0x76012100, 0x90082004,
    0xe06a2500, 0x462000a9, 0x18083020, 0x28007840, 0x980bdd61, 0x30ff30ff, 0x68003002, 0x0f800580,
    0x98019001, 0xd0052800, 0x301d4620, 0x28025d40, 0xe051d100, 0x462000a9, 0x18083020, 0x90047840,
    0x462000a9, 0x5c403020, 0x00a99005, 0x30304620, 0x90071808, 0x2200463b, 0x980c4621, 0xfe11f7ff,
    0x2e004606, 0x4630d002, 0xbdf0b00f, 0x01009804, 0x31804621, 0x21011842, 0x980c9b05, 0xfdbaf7ff,
    0x90042001, 0x980ca902, 0xfc7bf7ff, 0x2e004606, 0x4630d001, 0x8a60e7e8, 0xd1142800, 0x301d4620,
    0x28025d40, 0x4620d00f, 0x5d40301d, 0xd00a2803, 0x2200463b, 0x980c4621, 0xfe4af7ff, 0x2e004606,
    0x4630d008, 0x8a60e7d0, 0x43482164, 0x21004602, 0xff0ff7fb, 0xbf00bf00, 0x2d031c6d, 0xbf00d392,
    0x4630bf00, 0xb5f3e7c0, 0x460cb083, 0x90022004, 0x2c00bf00, 0xe01ad100, 0x46202500, 0x90013050,
    0xe0102600, 0xc8809801, 0x2f009001, 0x462ad00a, 0x98034621, 0xff5cf7ff, 0x98029002, 0xd0002800,
    0x19ede003, 0x2e041c76, 0xbf00d3ec, 0xbf00bf00, 0xb0059802, 0xb5f3bdf0, 0x460db083, 0xbf002704,
    0xf7ff9803, 0x4604f9b4, 0xd0012c00, 0xd1002d00, 0x7c28e0b5, 0xd1022800, 0x28007f28, 0x2110d00b,
    0x40086c28, 0xd1042800, 0xf7ff4628, 0x2800f996, 0x2001d001, 0x2000e000, 0x49499002, 0x42886828,
    0xe09cd000, 0xf7fb9803, 0x4629fdd3, 0xf7fb9803, 0x9802fefb, 0xd0052800, 0x21012200, 0xf7fb9803,
    0xe009fdd1, 0xf7ff4628, 0x9001f978, 0x5d412046, 0x98039a01, 0xfdc6f7fb, 0xf7fb9803, 0x6820fdaf,
    0x43882102, 0x46206020, 0xf920f7ff, 0x21026820, 0x60204308, 0x49336826, 0x2045400e, 0x28085d40,
    0x0280d101, 0x48304306, 0x7b284306, 0x21300100, 0x43064008, 0x46296026, 0xf7ff9803, 0x68a0fa75,
    0x03c92101, 0x60a04388, 0xf7ff4628, 0x2800f928, 0x68a0d004, 0x04c92101, 0x60a04308, 0x46204629,
    0xfb5ff7ff, 0x98034629, 0xfacdf7ff, 0x98034629, 0xf946f7ff, 0x21026820, 0x60204388, 0xf7ff4620,
    0x7c28f8e5, 0xd0072800, 0x98034629, 0xfe98f7ff, 0x2f004607, 0xe032d000, 0x28007f28, 0x4629d007,
    0xf7ff9803, 0x4607ff40, 0xd0002f00, 0x9802e027, 0xd0222800, 0x21026820, 0x60204308, 0xf7ff4628,
    0x9001f90c, 0x5d412046, 0x98039a01, 0xfd5af7fb, 0x98034629, 0xfa28f7ff, 0xe0074629, 0x00001b5a,
    0x42464346, 0x0000df0f, 0xffff0000, 0xf7ff9803, 0x6820f907, 0x43882102, 0x27006020, 0xbf00bf00,
    0xb0054638, 0xb570bdf0, 0xbf004605, 0xf7ff4628, 0x4604f8ee, 0xd1002c00, 0xbf00e007, 0x590020e0,
    0x40082102, 0xd0f92800, 0xbf00bf00, 0xb570bd70, 0xbf004605, 0xf7ff4628, 0x4604f8da, 0xd1002c00,
    0x4620e003, 0xf882f7ff, 0xbf00bf00, 0xb570bd70, 0x460d4606, 0x4630bf00, 0xf8c9f7ff, 0x2c004604,
    0xe010d100, 0xf7ff4620, 0x2d00fa3a, 0x6820d005, 0x02c92101, 0x60204308, 0x6820e004, 0x02c92101,
    0x60204388, 0xbf00bf00, 0x0000bd70, 0xb5104770, 0x890048c1, 0x00400840, 0x810849bf, 0x890048bf,
    0x00400840, 0x810849bd, 0x880048bb, 0x40082104, 0xd0042800, 0x880048b8, 0x49b74388, 0x48b78008,
    0x21048800, 0x28004008, 0x48b4d004, 0x43888800, 0x800849b2, 0x49b348b2, 0x48b36048, 0x46086088,
    0x21806800, 0x21204388, 0x49ae4308, 0x48af6008, 0x07c06900, 0x28000fc0, 0x48acd005, 0x08406900,
    0x49aa0040, 0x48aa6108, 0x21016940, 0x40080449, 0xd0014288, 0xf8bbf000, 0x694048a5, 0x04092101,
    0x42884008, 0xf000d001, 0xf7fff8ed, 0xbd10ffaf, 0x48a0b5fe, 0x21016b00, 0x40080409, 0xd0012800,
    0xe00b4d9d, 0x6b00489b, 0x0fc007c0, 0xd0012800, 0xe0002016, 0x49982014, 0x434d4605, 0x30404895,
    0x48956a07, 0x31404993, 0xf0006909, 0x463afb53, 0x91012300, 0xf0009000, 0x1945f933, 0x6900488d,
    0x4008498e, 0xd0012800, 0xe00b4e8b, 0x69004889, 0x40082102, 0xd0012800, 0xe0002016, 0x49862014,
    0x434e4606, 0x69404886, 0x06492101, 0x28004008, 0x4883d025, 0x21036980, 0x40080309, 0xd0082800,
    0x03092101, 0xd0061a40, 0xd0061a40, 0xd1061a40, 0x4634e006, 0x4c78e006, 0x4c77e004, 0xbf00e002,
    0xbf002400, 0x4876bf00, 0x22076940, 0x401006d2, 0x1c410ec0, 0xf0004620, 0x4604fa00, 0x4870e039,
    0x21036980, 0x40080489, 0xd0082800, 0x04892101, 0xd0061a40, 0xd0121a40, 0xd1271a40, 0x462ce01c,
    0x4864e026, 0x6b0030c0, 0x0612223f, 0x0e014010, 0xf0004630, 0x2112f9e2, 0x46044348, 0x4861e018,
    0x223f6800, 0x40100612, 0x46280e01, 0xf9d5f000, 0x43482112, 0xe00b4604, 0x69004859, 0x0f400740,
    0x48591c41, 0xf9c9f000, 0xe0014604, 0xbf002400, 0x4853bf00, 0x22076940, 0x40100292, 0x1c410a80,
    0xf0004620, 0x4951f9ba, 0x60084449, 0xbf00bdfe, 0xbf00bf00, 0x8f4ff3bf, 0xbf00bf00, 0xbf00bf00,
    0xbf00bf00, 0x8f6ff3bf, 0xbf00bf00, 0x2000bf00, 0x61084947, 0xbf00bf00, 0xf3bfbf00, 0xbf008f4f,
    0xbf00bf00, 0xbf00bf00, 0xf3bfbf00, 0xbf008f6f, 0xbf00bf00, 0x69404836, 0x04492101, 0x49344308,
    0xbf006148, 0xbf00bf00, 0x8f4ff3bf, 0xbf00bf00, 0xbf00bf00, 0xbf00bf00, 0x8f6ff3bf, 0xbf00bf00,
    0x4770bf00, 0x2300b510, 0x34804c29, 0xbf006063, 0xbf00bf00, 0x8f4ff3bf, 0xbf00bf00, 0x4623bf00,
    0x4b2c6818, 0x0b594003, 0x4b2bbf00, 0x08da4003, 0x014bbf00, 0x40234c29, 0x0fa40794, 0x432307a4,
    0x62234c23, 0xbf00bf00, 0x1e524613, 0xd1f02b00, 0x1e49460b, 0xd1e82b00, 0xbf00bf00, 0xf3bfbf00,
    0xbf008f4f, 0xbf00bf00, 0x695b4b11, 0x43234c13, 0x61634c0f, 0xbf00bf00, 0xf3bfbf00, 0xbf008f4f,
    0xbf00bf00, 0xbf00bf00, 0xf3bfbf00, 0xbf008f6f, 0xbf00bf00, 0x0000bd10, 0x400b8000, 0x400d0000,
    0xd928c520, 0x400bc000, 0x0000ffff, 0xe000e000, 0xe000ed00, 0x400d8000, 0x016e3600, 0x00010000,
    0x400fc000, 0x400d8100, 0x1dcd6500, 0x0000020c, 0xe000ef40, 0x0fffe000, 0x00001ff8, 0x00003fe0,
    0x4605b5fe, 0x460c4610, 0xd0734318, 0x468c46ae, 0x1aad2000, 0x419c4601, 0x4666d367, 0x24012700,
    0x1ab6463d, 0xd302419d, 0x463a4613, 0x46652421, 0x042f4676, 0x433e0c36, 0x1ab60c2d, 0xd304419d,
    0x041b0c15, 0x0412432b, 0x46653410, 0x062f4676, 0x433e0a36, 0x1ab60a2d, 0xd304419d, 0x021b0e15,
    0x0212432b, 0x46653408, 0x072f4676, 0x433e0936, 0x1ab6092d, 0xd304419d, 0x011b0f15, 0x0112432b,
    0x46651d24, 0x07af4676, 0x433e08b6, 0x1ab608ad, 0xd304419d, 0x009b0f95, 0x0092432b, 0x46651ca4,
    0x07ef4676, 0x433e0876, 0x1ab6086d, 0xd31a419d, 0x415b1892, 0xe0161c64, 0x46761800, 0x41494665,
    0x419d1ab7, 0x90009101, 0x4660d309, 0x41981ab1, 0x4684468e, 0x99019800, 0x1c402500, 0x07dd4169,
    0x432a0852, 0x1e64085b, 0x4672d5e6, 0xb0034663, 0xe7ffbdf0, 0x46012000, 0x46c046c0, 0x4623462a,
    0x0000e7f5, 0x4604b50e, 0x460e4615, 0xf000a012, 0x4620f93b, 0xf938f000, 0xf000a015, 0x4630f935,
    0xf932f000, 0xf000a014, 0x466cf92f, 0x72e02000, 0x72a0200a, 0xe007340a, 0x4628210a, 0xf877f000,
    0x46053130, 0x70211e64, 0xdcf52d00, 0xf0004620, 0xf000f91b, 0x0000f90f, 0x202a2a2a, 0x65737361,
    0x6f697472, 0x6166206e, 0x64656c69, 0x0000203a, 0x6966202c, 0x0020656c, 0x696c202c, 0x0020656e,
    0xb5104603, 0x079b430b, 0x2a04d10f, 0xc808d30d, 0x1f12c910, 0xd0f842a3, 0xba21ba18, 0xd9014288,
    0xbd102001, 0x43c02000, 0x2a00bd10, 0x07d3d003, 0x1c52d003, 0x2000e007, 0x7803bd10, 0x1c40780c,
    0x1b1b1c49, 0x7803d107, 0x1c40780c, 0x1b1b1c49, 0x1e92d101, 0x4618d1f1, 0xe001bd10, 0x1f09c004,
    0xd2fb2904, 0xd501078b, 0x1c808002, 0xd00007c9, 0x47707002, 0xd00b2900, 0xd00207c3, 0x1c407002,
    0x29021e49, 0x0783d304, 0x8002d502, 0x1e891c80, 0x2200e7e3, 0x2200e7ee, 0x2200e7df, 0x428b0903,
    0x0a03d32c, 0xd311428b, 0x469c2300, 0x4603e04e, 0xd43c430b, 0x08432200, 0xd331428b, 0x428b0903,
    0x0a03d31c, 0xd301428b, 0xe03f4694, 0x428b09c3, 0x01cbd301, 0x41521ac0, 0x428b0983, 0x018bd301,
    0x41521ac0, 0x428b0943, 0x014bd301, 0x41521ac0, 0x428b0903, 0x010bd301, 0x41521ac0, 0x428b08c3,
    0x00cbd301, 0x41521ac0, 0x428b0883, 0x008bd301, 0x41521ac0, 0x428b0843, 0x004bd301, 0x41521ac0,
    0xd2001a41, 0x41524601, 0x47704610, 0x0fcae05d, 0x4249d000, 0xd3001003, 0x40534240, 0x469c2200,
    0x428b0903, 0x0a03d32d, 0xd312428b, 0x018922fc, 0x0a03ba12, 0xd30c428b, 0x11920189, 0xd308428b,
    0x11920189, 0xd304428b, 0xd03a0189, 0xe0001192, 0x09c30989, 0xd301428b, 0x1ac001cb, 0x09834152,
    0xd301428b, 0x1ac0018b, 0x09434152, 0xd301428b, 0x1ac0014b, 0x09034152, 0xd301428b, 0x1ac0010b,
    0x08c34152, 0xd301428b, 0x1ac000cb, 0x08834152, 0xd301428b, 0x1ac0008b, 0xd2d94152, 0x428b0843,
    0x004bd301, 0x41521ac0, 0xd2001a41, 0x46634601, 0x105b4152, 0xd3014610, 0x2b004240, 0x4249d500,
    0x46634770, 0xd300105b, 0xb5014240, 0x46c02000, 0xbd0246c0, 0x2000b510, 0xf81cf000, 0x46c046c0,
    0xf0002001, 0xbd10f811, 0x4604b510, 0x1c64e002, 0xf804f000, 0x28007820, 0xbd10d1f9, 0x4669b508,
    0x20037008, 0xbd08beab, 0x20184901, 0xe7febeab, 0x00020026, 0xf000b510, 0xf000f80b, 0xbd10f802,
    0xb5104770, 0xd0012800, 0xffeef7ff, 0x0000bd10, 0x2100b510, 0xf000a002, 0x2001f813, 0x0000bd10,
    0x41474953, 0x3a545242, 0x6e624120, 0x616d726f, 0x6574206c, 0x6e696d72, 0x6f697461, 0x0000006e,
    0x4605b570, 0x200a460c, 0x1c6de000, 0xffc6f7ff, 0xd0062d00, 0x28007828, 0xe002d1f7, 0xf7ff1c64,
    0x2c00ffbd, 0x7820d002, 0xd1f72800, 0xf7ff200a, 0xbd70ffb5, 0xb282b5f8, 0x0c04b28b, 0x46100c0d,
    0x21004358, 0x435e4626, 0x04360c37, 0x41791980, 0x436e4616, 0x04360c37, 0x41791980, 0x436e4626,
    0xbdf81989, 0x4674b430, 0x78251e64, 0x42ab1c64, 0x5d63d304, 0x18e3005b, 0x4718bc30, 0xe7f8461d,
    0x08220000, 0x06180816, 0x0612041e, 0x0411060e, 0x0216060c, 0x06180000, 0x060c0416, 0x0312021e,
    0x01210216, 0x0116011a, 0x33221100, 0x77665544, 0xbbaa9988, 0xffeeddcc, 0x0818045a, 0x24ff3008,
    0x00000000, 0x00000000, 0x0918055a, 0x25ff3108, 0x00000000, 0x00000000, 0x0a18065a, 0x26ff3208,
    0x00000000, 0x00000000, 0x2403049f, 0x00000000, 0x00000000, 0x00000000, 0x0760079f, 0x27040b20,
    0x00000000, 0x00000000, 0x8760879f, 0xa7048b20, 0x00000000, 0x00000000, 0x0818045a, 0x24ff3008,
    0x00000000, 0x00000000, 0x8b20075a, 0x0000a7ff, 0x00000000, 0x00000000, 0x0818045a, 0x24ff3008,
    0x00000000, 0x00000000, 0x0a18065a, 0x000026ff, 0x00000000, 0x00000000, 0x0b18075a, 0x000027ff,
    0x00000000, 0x00000000, 0x8a18065a, 0x0000a6ff, 0x00000000, 0x00000000, 0x8b20075a, 0x0000a7ff,
    0x00000000, 0x00000000, 0x402a8000, 0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x11b3dc40
    ],

    # Relative function addresses
    'pc_init': 0x20200129,
    'pc_unInit': 0x20200165,
    'pc_program_page': 0x202001a9,
    'pc_erase_sector': 0x20200183,
    'pc_eraseAll': 0x2020016b,

    'static_base' : 0x20200000 + 0x00000020 + 0x000052ac,
    'begin_stack' : 0x20205700,
    'begin_data' : 0x20206000 + 0x1000,
    'page_size' : 0x100,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20001000, 0x20001100],   # Enable double buffering
    'min_program_length' : 0x100,
}

class MIMXRT1015xxxxx(IMXRT):

    VENDOR = "NXP"

    # Note: by default there are 64 KB ITCM, 64 KB DTCM and 128 KB OCRAM available for MIMXRT1015.
    # And it also has 256 KB FlexRAM that can be enabled and configured by GPR17, customers can 
    # allocate this 256 KB FlexRAM to ITCM/DTCM/OCRAM, but the FlexRAM is not available by default.
    MEMORY_MAP = MemoryMap(
        RamRegion(name="itcm",      start=0x00000000, length=0x10000), # 64 KB
        RomRegion(name="romcp",     start=0x00200000, length=0x18000), # 96 KB
        RamRegion(name="dtcm",      start=0x20000000, length=0x10000), # 64 KB
        RamRegion(name="ocram",     start=0x20200000, length=0x20000), # 128 KB
        FlashRegion(name="flexspi", start=0x60000000, end=0x60ffffff, blocksize=0x1000, is_boot_memory=True,
            algo=FLASH_ALGO_QUADSPI, page_size=0x100),
        )

    def __init__(self, session):
        super(MIMXRT1015xxxxx, self).__init__(session, self.MEMORY_MAP)
        self._svd_location = SVDFile.from_builtin("MIMXRT1015.xml")
