"""
Place Photo Module for Google Places API

This module contains the Place Photo functionality for Google Places API (New).
It implements the photo media endpoint for retrieving high-quality place photos.

Photo names must be obtained from:
- Text Search (field mask: places.photos)
- Place Details (field mask: photos)
- Nearby Search (field mask: places.photos)

This module is completely independent and can be used standalone.
"""

import os
import requests
from typing import Dict, List, Optional
from ..config import get_api_key


class PlacePhoto:
    """
    Place Photo functionality for Google Places API (New)
    
    Retrieves high-quality photos from the Places database.
    """
    
    # Constants
    BASE_URL = "https://places.googleapis.com/v1"
    MEDIA_ENDPOINT = "/media"
    
    def __init__(self, api_key: str = None):
        """
        Initialize the Place Photo client
        
        Args:
            api_key: Google Places API key (optional, uses global config if not provided)
        """
        # Get API key from parameter or global config
        if api_key:
            self.api_key = api_key
        else:
            self.api_key = get_api_key()
            if not self.api_key:
                raise ValueError(
                    "API key not provided and not set globally. "
                    "Please set the API key using google_maps_platform.set_api_key('your_key') "
                    "or pass it directly to the module constructor."
                )
    
    def validate_photo_name(self, photo_name: str) -> None:
        """
        Validate photo name format
        
        Args:
            photo_name: Photo resource name
            
        Raises:
            ValueError: If photo name is invalid
            
        Example:
            Valid format: "places/ChIJ.../photos/ATKogp..."
        """
        if not photo_name or not isinstance(photo_name, str):
            raise ValueError("photo_name must be a non-empty string")
        
        if not photo_name.strip():
            raise ValueError("photo_name cannot be empty or whitespace")
        
        # Check format: places/{PLACE_ID}/photos/{PHOTO_RESOURCE}
        if not photo_name.startswith("places/"):
            raise ValueError(
                f"Invalid photo name format. Expected format: 'places/{{PLACE_ID}}/photos/{{PHOTO_RESOURCE}}'\n"
                f"Got: {photo_name}"
            )
        
        if "/photos/" not in photo_name:
            raise ValueError(
                f"Invalid photo name format. Missing '/photos/' segment.\n"
                f"Expected format: 'places/{{PLACE_ID}}/photos/{{PHOTO_RESOURCE}}'\n"
                f"Got: {photo_name}"
            )
    
    def validate_dimensions(self, max_width_px: Optional[int], max_height_px: Optional[int]) -> None:
        """
        Validate dimension parameters
        
        Args:
            max_width_px: Maximum width in pixels
            max_height_px: Maximum height in pixels
            
        Raises:
            ValueError: If neither dimension is provided or if values are invalid
        """
        if max_width_px is None and max_height_px is None:
            raise ValueError(
                "At least one dimension parameter is required: max_width_px or max_height_px"
            )
        
        if max_width_px is not None and max_width_px <= 0:
            raise ValueError(f"max_width_px must be greater than 0, got: {max_width_px}")
        
        if max_height_px is not None and max_height_px <= 0:
            raise ValueError(f"max_height_px must be greater than 0, got: {max_height_px}")
    
    def get_photo(
        self,
        photo_name: str,
        max_width_px: Optional[int] = None,
        max_height_px: Optional[int] = None,
        skip_http_redirect: bool = False
    ) -> bytes:
        """
        Get photo binary data
        
        Args:
            photo_name: Photo resource name from Place Details/Search response
            max_width_px: Maximum width in pixels (at least one dimension required)
            max_height_px: Maximum height in pixels (at least one dimension required)
            skip_http_redirect: If True, returns JSON with photoUri instead of image
            
        Returns:
            Image binary data (JPEG, PNG, or GIF format)
            
        Raises:
            ValueError: If photo_name is invalid or dimensions are not provided
            requests.HTTPError: If the request fails
            
        Example:
            # Get photo with specific dimensions
            photo = PlacePhoto()
            image_data = photo.get_photo(
                photo_name="places/ChIJ.../photos/ATKogp...",
                max_width_px=400,
                max_height_px=400
            )
            
            # Save to file
            with open('photo.jpg', 'wb') as f:
                f.write(image_data)
        """
        # Validate inputs
        self.validate_photo_name(photo_name)
        self.validate_dimensions(max_width_px, max_height_px)
        
        # Build URL
        url = f"{self.BASE_URL}/{photo_name}{self.MEDIA_ENDPOINT}"
        
        # Build parameters
        params = {"key": self.api_key}
        
        if max_width_px is not None:
            params["maxWidthPx"] = max_width_px
        
        if max_height_px is not None:
            params["maxHeightPx"] = max_height_px
        
        if skip_http_redirect:
            params["skipHttpRedirect"] = "true"
        
        # Make request
        try:
            response = requests.get(url, params=params)
            response.raise_for_status()
            return response.content
        except requests.exceptions.HTTPError as e:
            if response.status_code == 403:
                raise requests.HTTPError(
                    "Quota exceeded (403). Please check your API quota.",
                    response=response
                )
            elif response.status_code == 404:
                raise requests.HTTPError(
                    f"Photo not found (404). The photo name may have expired or is invalid.\n"
                    f"Please obtain a fresh photo name from Place Details, Text Search, or Nearby Search.",
                    response=response
                )
            elif response.status_code == 429:
                raise requests.HTTPError(
                    "Too many requests (429). Please reduce the rate of photo requests.",
                    response=response
                )
            else:
                raise
    
    def get_photo_uri(
        self,
        photo_name: str,
        max_width_px: Optional[int] = None,
        max_height_px: Optional[int] = None
    ) -> str:
        """
        Get photo URI without downloading the image
        
        Args:
            photo_name: Photo resource name
            max_width_px: Maximum width in pixels
            max_height_px: Maximum height in pixels
            
        Returns:
            JSON response with photoUri field
            
        Example:
            photo = PlacePhoto()
            uri_response = photo.get_photo_uri(
                photo_name="places/ChIJ.../photos/ATKogp...",
                max_width_px=400
            )
            print(uri_response)  # {'photoUri': 'https://...'}
        """
        # Validate inputs
        self.validate_photo_name(photo_name)
        self.validate_dimensions(max_width_px, max_height_px)
        
        # Build URL
        url = f"{self.BASE_URL}/{photo_name}{self.MEDIA_ENDPOINT}"
        
        # Build parameters with skipHttpRedirect
        params = {
            "key": self.api_key,
            "skipHttpRedirect": "true"
        }
        
        if max_width_px is not None:
            params["maxWidthPx"] = max_width_px
        
        if max_height_px is not None:
            params["maxHeightPx"] = max_height_px
        
        # Make request
        try:
            response = requests.get(url, params=params)
            response.raise_for_status()
            return response.json()
        except requests.exceptions.HTTPError as e:
            raise
    
    def save_photo(
        self,
        photo_name: str,
        output_path: str,
        max_width_px: Optional[int] = None,
        max_height_px: Optional[int] = None
    ) -> bool:
        """
        Download photo and save to file
        
        Args:
            photo_name: Photo resource name
            output_path: Path to save the image (e.g., 'photo.jpg', '/path/to/image.png')
            max_width_px: Maximum width in pixels
            max_height_px: Maximum height in pixels
            
        Returns:
            True if successful, False otherwise
            
        Example:
            photo = PlacePhoto()
            success = photo.save_photo(
                photo_name="places/ChIJ.../photos/ATKogp...",
                output_path="restaurant.jpg",
                max_width_px=800
            )
        """
        try:
            # Get photo data
            image_data = self.get_photo(
                photo_name=photo_name,
                max_width_px=max_width_px,
                max_height_px=max_height_px
            )
            
            # Create directory if it doesn't exist
            output_dir = os.path.dirname(output_path)
            if output_dir and not os.path.exists(output_dir):
                os.makedirs(output_dir)
            
            # Save to file
            with open(output_path, 'wb') as f:
                f.write(image_data)
            
            return True
        except Exception as e:
            print(f"Error saving photo: {e}")
            return False
    
    def extract_photo_names_from_response(self, response: Dict) -> List[Dict]:
        """
        Extract photo information from Place Details/Search response
        
        Args:
            response: Response from Place Details, Text Search, or Nearby Search
            
        Returns:
            List of photo dictionaries with name, width, height, and attributions
            
        Example:
            place_details = PlaceDetails()
            result = place_details.get_details(
                place_id="ChIJ...",
                field_mask="id,displayName,photos"
            )
            
            photo = PlacePhoto()
            photos = photo.extract_photo_names_from_response(result)
            
            for photo_info in photos:
                print(f"Photo: {photo_info['name']}")
                print(f"Size: {photo_info['widthPx']}x{photo_info['heightPx']}")
                if photo_info['attributions']:
                    print(f"Attribution: {photo_info['attributions']}")
        """
        photos = []
        
        # Handle Place Details response format
        if "photos" in response:
            for photo in response["photos"]:
                photo_info = {
                    "name": photo.get("name"),
                    "widthPx": photo.get("widthPx"),
                    "heightPx": photo.get("heightPx"),
                    "attributions": photo.get("authorAttributions", [])
                }
                photos.append(photo_info)
        
        # Handle Text Search / Nearby Search response format (places array)
        elif "places" in response:
            for place in response["places"]:
                if "photos" in place:
                    for photo in place["photos"]:
                        photo_info = {
                            "name": photo.get("name"),
                            "widthPx": photo.get("widthPx"),
                            "heightPx": photo.get("heightPx"),
                            "attributions": photo.get("authorAttributions", [])
                        }
                        photos.append(photo_info)
        
        return photos
    
    def get_attribution_text(self, photo_info: Dict) -> Optional[str]:
        """
        Extract attribution text from photo information
        
        Args:
            photo_info: Photo information dictionary (from extract_photo_names_from_response)
            
        Returns:
            Attribution text if available, None otherwise
            
        Example:
            photo = PlacePhoto()
            photos = photo.extract_photo_names_from_response(result)
            
            for photo_info in photos:
                attribution = photo.get_attribution_text(photo_info)
                if attribution:
                    print(f"Photo by: {attribution}")
        """
        attributions = photo_info.get("attributions", [])
        
        if not attributions:
            return None
        
        # Get first attribution
        author = attributions[0]
        display_name = author.get("displayName", "Unknown")
        
        return f"Photo by {display_name}"
