"""
Place Details Module for Google Places API

This module contains the Place Details functionality for Google Places API (New).
It implements the places/{place_id} endpoint for getting detailed information about specific places.

Place IDs can be obtained from:
- Text Search / Nearby Search
- Geocoding API
- Routes API
- Address Validation API
- Autocomplete

This module is completely independent and can be used standalone.
"""

from typing import Dict, List, Optional
from .shared_utils import APIRequestHandler, FieldMaskHelper


class PlaceDetails:
    """
    Place Details functionality for Google Places API (New)
    
    Retrieves comprehensive information about a specific place using its Place ID.
    """
    
    # Constants
    BASE_URL_SUFFIX = "/places/"
    
    # All available field masks for place details (from official Google Places API documentation)
    AVAILABLE_FIELD_MASKS = [
        "attributions",
        "id",
        "name",
        "photos",
        "addressComponents",
        "addressDescriptor",
        "adrFormatAddress",
        "formattedAddress",
        "location",
        "plusCode",
        "postalAddress",
        "shortFormattedAddress",
        "types",
        "viewport",
        "accessibilityOptions",
        "businessStatus",
        "containingPlaces",
        "displayName",
        "googleMapsLinks",
        "googleMapsUri",
        "iconBackgroundColor",
        "iconMaskBaseUri",
        "primaryType",
        "primaryTypeDisplayName",
        "pureServiceAreaBusiness",
        "subDestinations",
        "utcOffsetMinutes",
        "currentOpeningHours",
        "currentSecondaryOpeningHours",
        "internationalPhoneNumber",
        "nationalPhoneNumber",
        "priceLevel",
        "priceRange",
        "rating",
        "regularOpeningHours",
        "regularSecondaryOpeningHours",
        "userRatingCount",
        "websiteUri",
        "allowsDogs",
        "curbsidePickup",
        "delivery",
        "dineIn",
        "editorialSummary",
        "evChargeAmenitySummary",
        "evChargeOptions",
        "fuelOptions",
        "generativeSummary",
        "goodForChildren",
        "goodForGroups",
        "goodForWatchingSports",
        "liveMusic",
        "menuForChildren",
        "neighborhoodSummary",
        "parkingOptions",
        "paymentOptions",
        "outdoorSeating",
        "reservable",
        "restroom",
        "reviews",
        "reviewSummary",
        "routingSummaries",
        "servesBeer",
        "servesBreakfast",
        "servesBrunch",
        "servesCocktails",
        "servesCoffee",
        "servesDessert",
        "servesDinner",
        "servesLunch",
        "servesVegetarianFood",
        "servesWine",
        "takeout"
    ]
    
    def __init__(self, api_key: str = None):
        """
        Initialize the Place Details client
        
        Args:
            api_key: Google Places API key (optional, uses global config if not provided)
        """
        self.api_handler = APIRequestHandler(api_key)
        self.field_helper = FieldMaskHelper()
        self.details_base_url = f"{self.api_handler.base_url}{self.BASE_URL_SUFFIX}"
    
    def list_field_masks(self) -> None:
        """
        Print all available field masks for place details
        """
        print("Available field masks for place details:")
        print("=" * 50)
        for i, field_mask in enumerate(self.AVAILABLE_FIELD_MASKS, 1):
            print(f"{i:2d}. {field_mask}")
        print(f"\nTotal: {len(self.AVAILABLE_FIELD_MASKS)} field masks")
    
    def validate_field_mask(self, field_mask: str) -> str:
        """
        Validate field mask and return cleaned version
        
        Args:
            field_mask: Field mask to validate (spaces around commas will be removed)
            
        Returns:
            Cleaned and validated field mask (no spaces around commas)
            
        Raises:
            ValueError: If field mask contains invalid fields
        """
        if not field_mask:
            return ""
        
        # Split by comma and clean each field
        fields = [field.strip() for field in field_mask.split(',')]
        valid_fields = []
        invalid_fields = []
        
        for field in fields:
            if field in self.AVAILABLE_FIELD_MASKS:
                valid_fields.append(field)
            else:
                invalid_fields.append(field)
        
        if invalid_fields:
            raise ValueError(
                f"Invalid field masks: {', '.join(invalid_fields)}\n"
                f"Available field masks: {', '.join(self.AVAILABLE_FIELD_MASKS[:10])}..."
            )
        
        # Return without spaces around commas (Google API requirement)
        return ','.join(valid_fields)
    
    def validate_place_id(self, place_id: str) -> None:
        """
        Validate place ID
        
        Args:
            place_id: Place ID to validate
            
        Raises:
            ValueError: If place_id is invalid or empty
        """
        if not place_id or not isinstance(place_id, str):
            raise ValueError("place_id must be a non-empty string")
        
        if not place_id.strip():
            raise ValueError("place_id cannot be empty or whitespace")
        
        # Basic format validation (Google Place IDs typically start with 'ChIJ' or similar)
        if len(place_id) < 10:
            raise ValueError(f"place_id appears to be invalid (too short): {place_id}")
    
    def get_details(
        self, 
        place_id: str, 
        field_mask: str,
        language_code: Optional[str] = None,
        region_code: Optional[str] = None,
        session_token: Optional[str] = None
    ) -> Optional[Dict]:
        """
        Get detailed information about a specific place
        
        Args:
            place_id: Google Places ID of the place (required)
            field_mask: Fields to return (required)
            language_code: Language for results (optional, e.g., "en", "es")
            region_code: Region code for results (optional, e.g., "US", "GB")
            session_token: Session token for billing optimization (optional)
            
        Returns:
            Dictionary containing place details or None if failed
            
        Example:
            # Basic place details
            place_details = PlaceDetails()
            result = place_details.get_details(
                place_id="ChIJk35bizx-j4AREil6UPp7Jn4",
                field_mask="id,displayName,formattedAddress,rating"
            )
            
            # Comprehensive details
            result = place_details.get_details(
                place_id="ChIJk35bizx-j4AREil6UPp7Jn4",
                field_mask="id,displayName,formattedAddress,rating,priceLevel,reviews,photos"
            )
            
            # With language and region
            result = place_details.get_details(
                place_id="ChIJk35bizx-j4AREil6UPp7Jn4",
                field_mask="id,displayName,formattedAddress",
                language_code="es",
                region_code="ES"
            )
            
            # With session token (for cost optimization)
            result = place_details.get_details(
                place_id="ChIJk35bizx-j4AREil6UPp7Jn4",
                field_mask="id,displayName",
                session_token="your_session_token"
            )
        """
        # Validate place_id
        self.validate_place_id(place_id)
        
        # Validate and clean field mask
        field_mask = self.validate_field_mask(field_mask)
        
        # Build headers with field mask
        headers = self.api_handler.build_headers_with_field_mask(
            field_mask=field_mask,
            language_code=language_code,
            region_code=region_code
        )
        
        # Add session token if provided
        if session_token:
            headers["X-Goog-Session-Token"] = session_token
        
        # Make the GET request
        url = f"{self.details_base_url}{place_id}"
        return self.api_handler.make_request(url=url, headers=headers)
    
    def get_multiple_details(
        self, 
        place_ids: List[str], 
        field_mask: str,
        **kwargs
    ) -> Dict[str, Optional[Dict]]:
        """
        Get details for multiple places
        
        Args:
            place_ids: List of Google Places IDs
            field_mask: Fields to return for all places (required)
            **kwargs: Additional parameters (language_code, region_code, session_token)
            
        Returns:
            Dictionary mapping place_id to details (or None if failed)
            
        Example:
            place_details = PlaceDetails()
            place_ids = [
                "ChIJk35bizx-j4AREil6UPp7Jn4",
                "ChIJj61dQgK6j4AR4GeTYWZsKWw"
            ]
            results = place_details.get_multiple_details(
                place_ids=place_ids,
                field_mask="id,displayName,rating"
            )
        """
        results = {}
        
        for place_id in place_ids:
            try:
                details = self.get_details(
                    place_id=place_id,
                    field_mask=field_mask,
                    **kwargs
                )
                results[place_id] = details
            except Exception as e:
                # Store error info instead of None
                results[place_id] = {"error": str(e)}
        
        return results
