#!/usr/bin/python
#
# Copyright 2015 John Kendrick
#
# This file is part of PDielec
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the MIT License
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# You should have received a copy of the MIT License
# along with this program, if not see https://opensource.org/licenses/MIT
#
"""Define some useful constants"""
import math

# define some useful basic constants
hartree2ev = 27.2114
PI = 2.0 * math.acos(0.0)
inv4pi = 1.0 / (4.0 * PI)
speed_light_si = 299792458.0
mu_0_si = 4.0*PI*1e-7
epsilon_0_si = 1.0/(mu_0_si*speed_light_si*speed_light_si)
planck_si = 6.62606896e-34
elementary_charge_si = 1.602176487e-19
electron_mass_si = 9.10938215e-31
proton_mass_si = 1.672621637e-27
electron_gyromagnetic_ratio_si = 1.76085977e11
avogadro_si = 6.02214179e23
molar_gas_si = 8.314472
hbar_si = planck_si/(2.0*PI)
fine_structure_si = elementary_charge_si*elementary_charge_si / (4*PI*epsilon_0_si*hbar_si*speed_light_si)
boltzmann_si = molar_gas_si/avogadro_si
amu_si = 1.0E-3 / avogadro_si
# Derived units
coulomb = 1.0 / elementary_charge_si
metre = electron_mass_si*speed_light_si*fine_structure_si/hbar_si
angstrom = 1.E-10 * metre
amu = 1.E-3 / avogadro_si / electron_mass_si
debye_si = 1.0E-21 / speed_light_si
debye = debye_si * coulomb * metre
d2byamuang2 = debye*debye / (angstrom * angstrom * amu)
joule = 1.0 / (fine_structure_si*fine_structure_si * electron_mass_si * speed_light_si*speed_light_si)
hertz = planck_si*joule
wavenumber = hertz*speed_light_si*100.0
strength_conv = 1.0 / d2byamuang2
angs2bohr = 1.0 / 0.5291772083
# J. R. de Laeter, J. K. Boehlke, P. De Bievre, H. Hidaka, H. S. Peiser, K. J. R. Rosman and P. D. P. Taylor (2003).
# "Atomic weights of the elements. Review 2000 (IUPAC Technical Report)"
isotopic_masses={
    "H":  [[1, 1.0078250319, 0.999885], [2, 2.0141017779, 0.000115]],
    "He": [[3, 3.0160293094, 0.00000134], [4, 4.0026032497, 0.99999866]],
    "Li": [[6, 6.0151223, 0.0759], [7, 7.0160041, 0.9241]],
    "Be": [[9, 9.0121822, 1.0000]],
    "B":  [[10, 10.0129371, 0.199], [11, 11.0093055, 0.801]],
    "C":  [[12, 12, 0.9893], [13, 13.003354838, 0.0107]],
    "N":  [[14, 14.0030740074, 0.99636], [15, 15.000108973, 0.00364]],
    "O":  [[16, 15.9949146223, 0.99757], [17, 16.99913150, 0.00038], [18, 17.9991604, 0.00205]],
    "F":  [[19, 18.99840320, 1.0000]],
    "Ne": [[20, 19.992440176, 0.9048], [21, 20.99384674, 0.0027], [22, 21.99138550, 0.0925]],
    "Na": [[23, 22.98976966, 1.0000]],
    "Mg": [[24, 23.98504187, 0.7899], [25, 24.98583700, 0.1000], [26, 25.98259300, 0.1101]],
    "Al": [[27, 26.98153841, 1.0000]],
    "Si": [[28, 27.97692649, 0.92223], [29, 28.97649468, 0.04685], [30, 29.97377018, 0.03092]],
    "P": [[31, 30.97376149, 1.0000]],
    "S": [[32, 31.97207073, 0.9499], [33, 32.97145854, 0.0075], [34, 33.96786687, 0.0425], [36, 35.96708088, 0.0001]],
    "Cl": [[35, 34.96885271, 0.7576], [37, 36.96590260, 0.2424]],
    "Ar": [[36, 35.96754626, 0.003365], [38, 37.9627322, 0.000632], [40, 39.962383124, 0.996003]],
    "K": [[39, 38.96370, 0.932581], [40, 39.96399867, 0.000117], [41, 40.96182597, 0.067302]],
    "Ca": [[40, 39.9625912, 0.96941], [42, 41.9586183, 0.00647], [43, 42.9587668, 0.00135], [44, 43.9554811, 0.02086], [46, 45.9536927, 0.00004], [48, 47.952533, 0.00187]],
    "Sc": [[45, 44.9559102, 1.0000]],
    "Ti": [[46, 45.9526295, 0.0825], [47, 46.9517637, 0.0744], [48, 47.9479470, 0.7372], [49, 48.9478707, 0.0541], [50, 49.9447920, 0.0518]],
    "V": [[50, 49.9471627, 0.00250], [51, 50.9439635, 0.99750]],
    "Cr": [[50, 49.9460495, 0.04345], [52, 51.9405115, 0.83789], [53, 52.9406534, 0.09501], [54, 53.9388846, 0.02365]],
    "Mn": [[55, 54.9380493, 1.0000]],
    "Fe": [[54, 53.9396147, 0.05845], [56, 55.9349418, 0.91754], [57, 56.9353983, 0.02119], [58, 57.9332801, 0.00282]],
    "Co": [[59, 58.9331999, 1.0000]],
    "Ni": [[58, 57.9353477, 0.680769], [60, 59.9307903, 0.262231], [61, 60.9310601, 0.011399], [62, 61.9283484, 0.036345], [64, 63.9279692, 0.009256]],
    "Cu": [[63, 62.9296007, 0.6915], [65, 64.9277938, 0.3085]],
    "Zn": [[64, 63.9291461, 0.48268], [66, 65.9260364, 0.27975], [67, 66.9271305, 0.04102], [68, 67.9248473, 0.19024], [70, 69.925325, 0.00631]],
    "Ga": [[69, 68.925581, 0.60108], [71, 70.9247073, 0.39892]],
    "Ge": [[70, 69.9242500, 0.2038], [72, 71.9220763, 0.2731], [73, 72.9234595, 0.0776], [74, 73.9211784, 0.3672], [76, 75.921402, 0.0783]],
    "As": [[75, 74.9215966, 1.0000]],
    "Se": [[74, 73.9224767, 0.0089], [76, 75.9192143, 0.0937], [77, 76.9199148, 0.0763], [78, 77.9173097, 0.2377], [80, 79.9165221, 0.4961], [82, 81.9167003, 0.0873]],
    "Br": [[79, 78.9183379, 0.5069], [81, 80.916291, 0.4931]],
    "Kr": [[78, 77.920388, 0.00355], [80, 79.916379, 0.02286], [82, 81.9134850, 0.11593], [83, 82.914137, 0.11500], [84, 83.911508, 0.56987], [86, 85.910615, 0.17279]],
    "Rb": [[85, 84.9117924, 0.7217], [87, 86.9091858, 0.2783]],
    "Sr": [[84, 83.913426, 0.0056], [86, 85.9092647, 0.0986], [87, 86.9088816, 0.0700], [88, 87.9056167, 0.8258]],
    "Y": [[89, 88.9058485, 1.0000]],
    "Zr": [[90, 89.9047022, 0.5145], [91, 90.9056434, 0.1122], [92, 91.9050386, 0.1715], [94, 93.9063144, 0.1738], [96, 95.908275, 0.0280]],
    "Nb": [[93, 92.9063762, 1.0000]],
    "Mo": [[92, 91.906810, 0.1477], [94, 93.9050867, 0.0923], [95, 94.9058406, 0.1590], [96, 95.9046780, 0.1668], [97, 96.9060201, 0.0956], [98, 97.905406, 0.2419], [100, 99.907476, 0.0967]],
    "Tc": None,
    "Ru": [[96, 95.907604, 0.0554], [98, 97.905287, 0.0187], [99, 98.9059385, 0.1276], [100, 99.9042189, 0.1260], [101, 100.9055815, 0.1706], [102, 101.9043488, 0.3155], [104, 103.905430, 0.1862]],
    "Rh": [[103, 102.905504, 1.0000]],
    "Pd": [[102, 101.905607, 0.0102], [104, 103.904034, 0.1114], [105, 104.905083, 0.2233], [106, 105.903484, 0.2733], [108, 107.903895, 0.2646], [110, 109.905153, 0.1172]],
    "Ag": [[107, 106.905093, 0.51839], [109, 108.904756, 0.48161]],
    "Cd": [[106, 105.906458, 0.0125], [108, 107.904183, 0.0089], [110, 109.903006, 0.1249], [111, 110.904182, 0.1280], [112, 111.9027577, 0.2413], [113, 112.9044014, 0.1222], [114, 113.9033586, 0.2873], [116, 115.904756, 0.0749]],
    "In": [[113, 112.904062, 0.0429], [115, 114.903879, 0.9571]],
    "Sn": [[112, 111.904822, 0.0097], [114, 113.902783, 0.0066], [115, 114.903347, 0.0034], [116, 115.901745, 0.1454], [117, 116.902955, 0.0768], [118, 117.901608, 0.2422], [119, 118.903311, 0.0859], [120, 119.9021985, 0.3258], [122, 121.9034411, 0.0463], [124, 123.9052745, 0.0579]],
    "Sb": [[121, 120.9038222, 0.5721], [123, 122.9042160, 0.4279]],
    "Te": [[120, 119.904026, 0.0009], [122, 121.9030558, 0.0255], [123, 122.9042711, 0.0089], [124, 123.9028188, 0.0474], [125, 124.9044241, 0.0707], [126, 125.9033049, 0.1884], [128, 127.9044615, 0.3174], [130, 129.9062229, 0.3408]],
    "I": [[127, 126.904468, 1.0000]],
    "Xe": [[124, 123.9058954, 0.000952], [126, 125.904268, 0.000890], [128, 127.9035305, 0.019102], [129, 128.9047799, 0.264006], [130, 129.9035089, 0.040710], [131, 130.9050828, 0.212324], [132, 131.9041546, 0.269086], [134, 133.9053945, 0.104357], [136, 135.907220, 0.088573]],
    "Cs": [[133, 132.905447, 1.0000]],
    "Ba": [[130, 129.906311, 0.00106], [132, 131.905056, 0.00101], [134, 133.904504, 0.02417], [135, 134.905684, 0.06592], [136, 135.904571, 0.07854], [137, 136.905822, 0.11232], [138, 137.905242, 0.71698]],
    "La": [[138, 137.907108, 0.00090], [139, 138.906349, 0.99910]],
    "Ce": [[136, 135.907140, 0.00185], [138, 137.905986, 0.00251], [140, 139.905435, 0.88450], [142, 141.909241, 0.11114]],
    "Pr": [[141, 140.907648, 1.0000]],
    "Nd": [[142, 141.907719, 0.272], [143, 142.909810, 0.122], [144, 143.910083, 0.238], [145, 144.912569, 0.083], [146, 145.913113, 0.172], [148, 147.916889, 0.057], [150, 149.920887, 0.056]],
    "Pm": None,
    "Sm": [[144, 143.911996, 0.0307], [147, 146.914894, 0.1499], [148, 147.914818, 0.1124], [149, 148.917180, 0.1382], [150, 149.917272, 0.0738], [152, 151.919729, 0.2675], [154, 153.922206, 0.2275]],
    "Eu": [[151, 150.919846, 0.4781], [153, 152.921227, 0.5219]],
    "Gd": [[152, 151.919789, 0.0020], [154, 153.920862, 0.0218], [155, 154.922619, 0.1480], [156, 155.922120, 0.2047], [157, 156.923957, 0.1565], [158, 157.924101, 0.2484], [160, 159.927051, 0.2186]],
    "Tb": [[159, 158.925343, 1.0000]],
    "Dy": [[156, 155.924278, 0.00056], [158, 157.924405, 0.00095], [160, 159.925194, 0.02329], [161, 160.926930, 0.18889], [162, 161.926795, 0.25475], [163, 162.928728, 0.24896], [164, 163.929171, 0.28260]],
    "Ho": [[165, 164.930319, 1.0000]],
    "Er": [[162, 161.928775, 0.00139], [164, 163.929197, 0.01601], [166, 165.930290, 0.33503], [167, 166.932046, 0.22869], [168, 167.932368, 0.26978], [170, 169.935461, 0.14910]],
    "Tm": [[169, 168.934211, 1.0000]],
    "Yb": [[168, 167.933895, 0.0013], [170, 169.934759, 0.0304], [171, 170.936323, 0.1428], [172, 171.936378, 0.2183], [173, 172.938207, 0.1613], [174, 173.938858, 0.3183], [176, 175.942569, 0.1276]],
    "Lu": [[175, 174.9407682, 0.9741], [176, 175.9426827, 0.0259]],
    "Hf": [[174, 173.940042, 0.0016], [176, 175.941403, 0.0526], [177, 176.9432204, 0.1860], [178, 177.9436981, 0.2728], [179, 178.9458154, 0.1362], [180, 179.9465488, 0.3508]],
    "Ta": [[180, 179.947466, 0.00012], [181, 180.947996, 0.99988]],
    "W": [[180, 179.946706, 0.0012], [182, 181.948205, 0.2650], [183, 182.9502242, 0.1431], [184, 183.9509323, 0.3064], [186, 185.95436, 0.2843]],
    "Re": [[185, 184.952955, 0.3740], [187, 186.9557505, 0.6260]],
    "Os": [[184, 183.952491, 0.0002], [186, 185.953838, 0.0159], [187, 186.9557476, 0.0196], [188, 187.9558357, 0.1324], [189, 188.958145, 0.1615], [190, 189.958445, 0.2626], [192, 191.961479, 0.4078]],
    "Ir": [[191, 190.960591, 0.373], [193, 192.962923, 0.627]],
    "Pt": [[190, 189.959930, 0.00014], [192, 191.961035, 0.00782], [194, 193.962663, 0.32967], [195, 194.964774, 0.33832], [196, 195.964934, 0.25242], [198, 197.967875, 0.07163]],
    "Au": [[197, 196.966551, 1.0000]],
    "Hg": [[196, 195.965814, 0.0015], [198, 197.966752, 0.0997], [199, 198.968262, 0.1687], [200, 199.968309, 0.2310], [201, 200.970285, 0.1318], [202, 201.970625, 0.2986], [204, 203.973475, 0.0687]],
    "Tl": [[203, 202.972329, 0.2952], [205, 204.974412, 0.7048]],
    "Pb": [[204, 203.973028, 0.014], [206, 205.974449, 0.241], [207, 206.975880, 0.221], [208, 207.976636, 0.524]],
    "Bi": [[209, 208.980384, 1.0000]],
    "Po": None,
    "At": None,
    "Rn": None,
    "Fr": None,
    "Ra": None,
    "Ac": None,
    "Th": [[232, 232.0380495, 1.0000]],
    "Pa": [[231, 231.03588, 1.0000]],
    "U": [[234, 234.0409447, 0.000054], [235, 235.0439222, 0.007204], [238, 238.0507835, 0.992742]]
}

# "Atomic weights of the elements. Review 2000 (IUPAC Technical Report)"
isotope_masses={
    "H":  1.0078250319,
    "He": 4.0026032497,
    "Li": 7.0160041,
    "Be": 9.0121822,
    "B":  11.0093055,
    "C":  12.0,
    "N":  14.0030740074,
    "O":  15.9949146223,
    "F":  18.99840320,
    "Ne": 19.992440176,
    "Na": 22.98976966,
    "Mg": 23.98504187,
    "Al": 26.98153841,
    "Si": 27.97692649,
    "P":  30.97376149,
    "S":  31.97207073,
    "Cl": 34.96885271,
    "Ar": 39.962383124,
    "K":  38.96370,
    "Ca": 39.9625912,
    "Sc": 44.9559102,
    "Ti": 47.9479470,
    "V":  50.9439635,
    "Cr": 51.9405115,
    "Mn": 54.9380493,
    "Fe": 55.9349418,
    "Co": 58.9331999,
    "Ni": 57.9353477,
    "Cu": 62.9296007,
    "Zn": 63.9291461,
    "Ga": 68.925581,
    "Ge": 73.9211784,
    "As": 74.9215966,
    "Se": 79.9165221,
    "Br": 78.9183379,
    "Kr": 83.911508,
    "Rb": 84.9117924,
    "Sr": 87.9056167,
    "Y":  88.9058485,
    "Zr": 89.9047022,
    "Nb": 92.9063762,
    "Mo": 97.905406,
    "Tc": None,
    "Ru": 101.9043488,
    "Rh": 102.905504,
    "Pd": 105.903484,
    "Ag": 106.905093,
    "Cd": 113.9033586,
    "In": 114.903879,
    "Sn": 119.9021985,
    "Sb": 120.9038222,
    "Te": 129.9062229,
    "I":  126.904468,
    "Xe": 131.9041546,
    "Cs": 132.905447,
    "Ba": 137.905242,
    "La": 138.906349,
    "Ce": 139.905435,
    "Pr": 140.907648,
    "Nd": 141.907719,
    "Pm": None,
    "Sm": 151.919729,
    "Eu": 152.921227,
    "Gd": 157.924101,
    "Tb": 158.925343,
    "Dy": 163.929171,
    "Ho": 164.930319,
    "Er": 167.932368,
    "Tm": 168.934211,
    "Yb": 173.938858,
    "Lu": 174.9407682,
    "Hf": 179.9465488,
    "Ta": 180.947996,
    "W":  183.9509323,
    "Re": 186.9557505,
    "Os": 191.961479,
    "Ir": 192.962923,
    "Pt": 194.964774,
    "Au": 196.966551,
    "Hg": 201.970625,
    "Tl": 204.974412,
    "Pb": 207.976636,
    "Bi": 208.980384,
    "Po": None,
    "At": None,
    "Rn": None,
    "Fr": None,
    "Ra": None,
    "Ac": None,
    "Th": 232.0380495,
    "Pa": 231.03588,
    "U":  238.0507835
}

average_masses = {
    "X": None,
    "H": 1.00794,
    "He": 4.002602,
    "Li": 6.941,
    "Be": 9.012182,
    "B": 10.811,
    "C": 12.0107,
    "N": 14.0067,
    "O": 15.9994,
    "F": 18.9984032,
    "Ne": 20.1797,
    "Na": 22.98976928,
    "Mg": 24.3050,
    "Al": 26.9815386,
    "Si": 28.0855,
    "P": 30.973762,
    "S": 32.065,
    "Cl": 35.453,
    "Ar": 39.948,
    "K": 39.0983,
    "Ca": 40.078,
    "Sc": 44.955912,
    "Ti": 47.867,
    "V": 50.9415,
    "Cr": 51.9961,
    "Mn": 54.938045,
    "Fe": 55.845,
    "Co": 58.933195,
    "Ni": 58.6934,
    "Cu": 63.546,
    "Zn": 65.38,
    "Ga": 69.723,
    "Ge": 72.64,
    "As": 74.92160,
    "Se": 78.96,
    "Br": 79.904,
    "Kr": 83.798,
    "Rb": 85.4678,
    "Sr": 87.62,
    "Y": 88.90585,
    "Zr": 91.224,
    "Nb": 92.90638,
    "Mo": 95.96,
    "Tc": None,
    "Ru": 101.07,
    "Rh": 102.90550,
    "Pd": 106.42,
    "Ag": 107.8682,
    "Cd": 112.411,
    "In": 114.818,
    "Sn": 118.710,
    "Sb": 121.760,
    "Te": 127.60,
    "I": 126.90447,
    "Xe": 131.293,
    "Cs": 132.9054519,
    "Ba": 137.327,
    "La": 138.90547,
    "Ce": 140.116,
    "Pr": 140.90765,
    "Nd": 144.242,
    "Pm": None,
    "Sm": 150.36,
    "Eu": 151.964,
    "Gd": 157.25,
    "Tb": 158.92535,
    "Dy": 162.500,
    "Ho": 164.93032,
    "Er": 167.259,
    "Tm": 168.93421,
    "Yb": 173.054,
    "Lu": 174.9668,
    "Hf": 178.49,
    "Ta": 180.94788,
    "W": 183.84,
    "Re": 186.207,
    "Os": 190.23,
    "Ir": 192.217,
    "Pt": 195.084,
    "Au": 196.966569,
    "Hg": 200.59,
    "Tl": 204.3833,
    "Pb": 207.2,
    "Bi": 208.98040,
    "Po": None,
    "At": None,
    "Rn": None,
    "Fr": None,
    "Ra": None,
    "Ac": None,
    "Th": 232.03806,
    "Pa": 231.03588,
    "U": 238.02891,
    "Np": None,
    "Pu": None,
    "Am": None,
    "Cm": None,
    "Bk": None,
    "Cf": None,
    "Es": None,
    "Fm": None,
    "Md": None,
    "No": None,
    "Lr": None,
    "Rf": None,
    "Db": None,
    "Sg": None,
    "Bh": None,
    "Hs": None,
    "Mt": None,
    "Ds": None,
    "Rg": None,
    "Cn": None,
    "Uut": None,
    "Uuq": None,
    "Uup": None,
    "Uuh": None,
    "Uus": None,
    "Uuo": None
}

#
# Covalent radii from;
# Beatriz Cordero, Veronica Gomex, Ana E. Platero-Prats, Marc Reves, Jorge Echeverria,
# Eduard Cremades, Flavia Barragan and Santiago Alvarez (2008)
# Covalent Radii Revisited - Dalton Trans. (21) 2832-2838 doiL10.1039/b801115j
#
covalent_radii = {
    "X":  0.00,
    "H":  0.31,
    "He": 0.28,
    "Li": 1.28,
    "Be": 0.96,
    "B":  0.84,
    "C":  0.76,
    "N":  0.71,
    "O":  0.66,
    "F":  0.57,
    "Ne": 0.58,
    "Na": 1.66,
    "Mg": 1.41,
    "Al": 1.21,
    "Si": 1.11,
    "P":  1.07,
    "S":  1.05,
    "Cl": 1.02,
    "Ar": 1.06,
    "K":  2.03,
    "Ca": 1.76,
    "Sc": 1.70,
    "Ti": 1.60,
    "V":  1.53,
    "Cr": 1.39,
    "Mn": 1.61,
    "Fe": 1.52,
    "Co": 1.50,
    "Ni": 1.24,
    "Cu": 1.32,
    "Zn": 1.22,
    "Ga": 1.22,
    "Ge": 1.20,
    "As": 1.19,
    "Se": 1.20,
    "Br": 1.20,
    "Kr": 1.16,
    "Rb": 2.20,
    "Sr": 1.95,
    "Y":  1.90,
    "Zr": 1.75,
    "Nb": 1.64,
    "Mo": 1.54,
    "Tc": 1.47,
    "Ru": 1.46,
    "Rh": 1.42,
    "Pd": 1.39,
    "Ag": 1.45,
    "Cd": 1.44,
    "In": 1.42,
    "Sn": 1.39,
    "Sb": 1.39,
    "Te": 1.38,
    "I":  1.39,
    "Xe": 1.40,
    "Cs": 2.44,
    "Ba": 2.15,
    "La": 2.07,
    "Ce": 2.04,
    "Pr": 2.02,
    "Nd": 2.01,
    "Pm": 1.99,
    "Sm": 1.98,
    "Eu": 1.98,
    "Gd": 1.96,
    "Tb": 1.94,
    "Dy": 1.92,
    "Ho": 1.92,
    "Er": 1.89,
    "Tm": 1.90,
    "Yb": 1.87,
    "Lu": 1.87,
    "Hf": 1.75,
    "Ta": 1.70,
    "W":  1.62,
    "Re": 1.51,
    "Os": 1.44,
    "Ir": 1.41,
    "Pt": 1.36,
    "Au": 1.36,
    "Hg": 1.32,
    "Tl": 1.45,
    "Pb": 1.46,
    "Bi": 1.48,
    "Po": 1.40,
    "At": 1.50,
    "Rn": 1.50,
    "Fr": 2.60,
    "Ra": 2.21,
    "Ac": 2.15,
    "Th": 2.06,
    "Pa": 2.00,
    "U":  1.97,
    "Np": 1.90,
    "Pu": 1.87,
    "Am": 1.80,
    "Cm": 1.69,
    "Bk": 1.70,
    "Cf": 1.70,
    "Es": 1.70,
    "Fm": 1.70,
    "Md": 1.70,
    "No": 1.70,
    "Lr": 1.70,
    "Rf": 1.70,
    "Db": 1.70,
    "Sg": 1.70,
    "Bh": 1.70,
    "Hs": 1.70,
    "Mt": 1.70,
    "Ds": 1.70,
    "Rg": 1.70,
    "Cn": 1.70,
    "Uut": 1.70,
    "Uuq": 1.70,
    "Uup": 1.70,
    "Uuh": 1.70,
    "Uus": 1.70,
    "Uuo": 1.70
}

# Elemental colours
elemental_colours = {
    "X":  [  0,   0,   0, 255],
    "H":  [240, 240, 240, 255],
    "He": [255, 170, 170, 255],
    "Li": [255, 170, 170, 255],
    "Be": [255, 170, 170, 255],
    "B":  [  0, 255,   0, 255],
    "C":  [100, 100, 100, 255],
    "N":  [  0,   0, 255, 255],
    "O":  [255,   0,   0, 255],
    "F":  [  0, 255,   0, 255],
    "Ne": [255, 170, 170, 255],
    "Na": [255, 255,   0, 255],
    "Mg": [255, 255,   0, 255],
    "Al": [255, 255,   0, 255],
    "Si": [130,  60, 255, 255],
    "P":  [255,   0, 255, 255],
    "S":  [255, 255,   0, 255],
    "Cl": [  0, 255,   0, 255],
    "Ar": [255, 170, 170, 255],
    "K":  [255, 170, 170, 255],
    "Ca": [255, 170, 170, 255],
    "Sc": [255, 130, 170, 255],
    "Ti": [255, 130, 170, 255],
    "V":  [255, 130, 170, 255],
    "Cr": [255, 130, 170, 255],
    "Mn": [255, 130, 170, 255],
    "Fe": [255, 130, 170, 255],
    "Co": [255, 130, 170, 255],
    "Ni": [255, 130, 170, 255],
    "Cu": [255, 130, 170, 255],
    "Zn": [255, 130, 170, 255],
    "Ga": [255, 130, 170, 255],
    "Ge": [255, 130, 170, 255],
    "As": [255, 130, 170, 255],
    "Se": [255, 130, 170, 255],
    "Br": [255, 130, 170, 255],
    "Kr": [255, 130, 170, 255],
    "Rb": [255, 130, 170, 255],
    "Sr": [255, 130, 170, 255],
    "Y":  [255, 130, 170, 255],
    "Zr": [255, 130, 170, 255],
    "Nb": [255, 130, 170, 255],
    "Mo": [255, 130, 170, 255],
    "Tc": [255, 130, 170, 255],
    "Ru": [255, 130, 170, 255],
    "Rh": [255, 130, 170, 255],
    "Pd": [255, 130, 170, 255],
    "Ag": [255, 130, 170, 255],
    "Cd": [255, 130, 170, 255],
    "In": [255, 130, 170, 255],
    "Sn": [255, 130, 170, 255],
    "Sb": [255, 130, 170, 255],
    "Te": [255, 130, 170, 255],
    "I":  [255, 130, 170, 255],
    "Xe": [255, 130, 170, 255],
    "Cs": [255, 130, 170, 255],
    "Ba": [255, 130, 170, 255],
    "La": [255, 130, 170, 255],
    "Ce": [255, 130, 170, 255],
    "Pr": [255, 130, 170, 255],
    "Nd": [255, 130, 170, 255],
    "Pm": [255, 130, 170, 255],
    "Sm": [255, 130, 170, 255],
    "Eu": [255, 130, 170, 255],
    "Gd": [255, 130, 170, 255],
    "Tb": [255, 130, 170, 255],
    "Dy": [255, 130, 170, 255],
    "Ho": [255, 130, 170, 255],
    "Er": [255, 130, 170, 255],
    "Tm": [255, 130, 170, 255],
    "Yb": [255, 130, 170, 255],
    "Lu": [255, 130, 170, 255],
    "Hf": [255, 130, 170, 255],
    "Ta": [255, 130, 170, 255],
    "W":  [255, 130, 170, 255],
    "Re": [255, 130, 170, 255],
    "Os": [255, 130, 170, 255],
    "Ir": [255, 130, 170, 255],
    "Pt": [255, 130, 170, 255],
    "Au": [255, 130, 170, 255],
    "Hg": [255, 130, 170, 255],
    "Tl": [255, 130, 170, 255],
    "Pb": [255, 130, 170, 255],
    "Bi": [255, 130, 170, 255],
    "Po": [255, 130, 170, 255],
    "At": [255, 130, 170, 255],
    "Rn": [255, 130, 170, 255],
    "Fr": [255, 130, 170, 255],
    "Ra": [255, 130, 170, 255],
    "Ac": [255, 130, 170, 255],
    "Th": [255, 130, 170, 255],
    "Pa": [255, 130, 170, 255],
    "U":  [255, 130, 170, 255],
    "Np": [255, 130, 170, 255],
    "Pu": [255, 130, 170, 255],
    "Am": [255, 130, 170, 255],
    "Cm": [255, 130, 170, 255],
    "Bk": [255, 130, 170, 255],
    "Cf": [255, 130, 170, 255],
    "Es": [255, 130, 170, 255],
    "Fm": [255, 130, 170, 255],
    "Md": [255, 130, 170, 255],
    "No": [255, 130, 170, 255],
    "Lr": [255, 130, 170, 255],
    "Rf": [255, 130, 170, 255],
    "Db": [255, 130, 170, 255],
    "Sg": [255, 130, 170, 255],
    "Bh": [255, 130, 170, 255],
    "Hs": [255, 130, 170, 255],
    "Mt": [255, 130, 170, 255],
    "Ds": [255, 130, 170, 255],
    "Rg": [255, 130, 170, 255],
    "Cn": [255, 130, 170, 255],
    "Uut": [255, 130, 170, 255],
    "Uuq": [255, 130, 170, 255],
    "Uup": [255, 130, 170, 255],
    "Uuh": [255, 130, 170, 255],
    "Uus": [255, 130, 170, 255],
    "Uuo": [255, 130, 170, 255]
}

atomic_number_to_element = {
    0: "X",
    1: "H",
    2: "He",
    3: "Li",
    4: "Be",
    5: "B",
    6: "C",
    7: "N",
    8: "O",
    9: "F",
    10: "Ne",
    11: "Na",
    12: "Mg",
    13: "Al",
    14: "Si",
    15: "P",
    16: "S",
    17: "Cl",
    18: "Ar",
    19: "K",
    20: "Ca",
    21: "Sc",
    22: "Ti",
    23: "V",
    24: "Cr",
    25: "Mn",
    26: "Fe",
    27: "Co",
    28: "Ni",
    29: "Cu",
    30: "Zn",
    31: "Ga",
    32: "Ge",
    33: "As",
    34: "Se",
    35: "Br",
    36: "Kr",
    37: "Rb",
    38: "Sr",
    39: "Y",
    40: "Zr",
    41: "Nb",
    42: "Mo",
    43: "Tc",
    44: "Ru",
    45: "Rh",
    46: "Pd",
    47: "Ag",
    48: "Cd",
    49: "In",
    50: "Sn",
    51: "Sb",
    52: "Te",
    53: "I",
    54: "Xe",
    55: "Cs",
    56: "Ba",
    57: "La",
    58: "Ce",
    59: "Pr",
    60: "Nd",
    61: "Pm",
    62: "Sm",
    63: "Eu",
    64: "Gd",
    65: "Tb",
    66: "Dy",
    67: "Ho",
    68: "Er",
    69: "Tm",
    70: "Yb",
    71: "Lu",
    72: "Hf",
    73: "Ta",
    74: "W",
    75: "Re",
    76: "Os",
    77: "Ir",
    78: "Pt",
    79: "Au",
    80: "Hg",
    81: "Tl",
    82: "Pb",
    83: "Bi",
    84: "Po",
    85: "At",
    86: "Rn",
    87: "Fr",
    88: "Ra",
    89: "Ac",
    90: "Th",
    91: "Pa",
    92: "U",
    93: "Np",
    94: "Pu",
    95: "Am",
    96: "Cm",
    97: "Bk",
    98: "Cf",
    99: "Es",
    100: "Fm",
    101: "Md",
    102: "No",
    103: "Lr",
    104: "Rf",
    105: "Db",
    106: "Sg",
    107: "Bh",
    108: "Hs",
    109: "Mt",
    110: "Ds",
    111: "Rg",
    112: "Cn",
    113: "Uut",
    114: "Uuq",
    115: "Uup",
    116: "Uuh",
    117: "Uus",
    118: "Uuo"
}

# Define support matrix materials as (density, permittivity)
support_matrix_db           = {}
support_matrix_db["air"]    = (1.225e-3, 1.000)
support_matrix_db["vacuum"] = (1.000e-15, 1.000)
support_matrix_db["ptfe"]   = (2.200, 2.000)
support_matrix_db["kbr"]    = (2.750, 2.250)
support_matrix_db["nujol"]  = (0.838, 2.155)
support_matrix_db["ldpe"]   = (0.925, 2.250)
support_matrix_db["mdpe"]   = (0.933, 2.250)
