"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancedServiceBase = exports.NetworkLoadBalancedServiceRecordType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Describes the type of DNS record the service should create.
 *
 * @stability stable
 */
var NetworkLoadBalancedServiceRecordType;
(function (NetworkLoadBalancedServiceRecordType) {
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["ALIAS"] = 0] = "ALIAS";
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["CNAME"] = 1] = "CNAME";
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["NONE"] = 2] = "NONE";
})(NetworkLoadBalancedServiceRecordType = exports.NetworkLoadBalancedServiceRecordType || (exports.NetworkLoadBalancedServiceRecordType = {}));
/**
 * The base class for NetworkLoadBalancedEc2Service and NetworkLoadBalancedFargateService services.
 *
 * @stability stable
 */
class NetworkLoadBalancedServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the NetworkLoadBalancedServiceBase class.
     *
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e;
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        this.internalDesiredCount = props.desiredCount;
        const internetFacing = (_b = props.publicLoadBalancer) !== null && _b !== void 0 ? _b : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = (_c = props.loadBalancer) !== null && _c !== void 0 ? _c : new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, 'LB', lbProps);
        const listenerPort = (_d = props.listenerPort) !== null && _d !== void 0 ? _d : 80;
        const targetProps = {
            port: 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', { port: listenerPort });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            switch ((_e = props.recordType) !== null && _e !== void 0 ? _e : NetworkLoadBalancedServiceRecordType.ALIAS) {
                case NetworkLoadBalancedServiceRecordType.ALIAS:
                    new aws_route53_1.ARecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.CNAME:
                    new aws_route53_1.CnameRecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        domainName: loadBalancer.loadBalancerDnsName,
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.NONE:
                    // Do not create a DNS record
                    break;
            }
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.NetworkLoadBalancer) {
            this._networkLoadBalancer = loadBalancer;
        }
        if (props.loadBalancer === undefined) {
            new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * The Network Load Balancer for the service.
     *
     * @stability stable
     */
    get loadBalancer() {
        if (!this._networkLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._networkLoadBalancer;
    }
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     *
     * @stability stable
     */
    addServiceAsTarget(service) {
        this.targetGroup.addTarget(service);
    }
    /**
     * @stability stable
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.NetworkLoadBalancedServiceBase = NetworkLoadBalancedServiceBase;
_a = JSII_RTTI_SYMBOL_1;
NetworkLoadBalancedServiceBase[_a] = { fqn: "@aws-cdk/aws-ecs-patterns.NetworkLoadBalancedServiceBase", version: "1.104.0" };
//# sourceMappingURL=data:application/json;base64,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