import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { IVpc } from '@aws-cdk/aws-ec2';
import { AwsLogDriver, BaseService, CloudMapOptions, Cluster, ContainerImage, DeploymentController, DeploymentCircuitBreaker, ICluster, LogDriver, PropagatedTagSource, Secret } from '@aws-cdk/aws-ecs';
import { ApplicationListener, ApplicationLoadBalancer, ApplicationProtocol, ApplicationProtocolVersion, ApplicationTargetGroup, IApplicationLoadBalancer } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IRole } from '@aws-cdk/aws-iam';
import { IHostedZone } from '@aws-cdk/aws-route53';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * Describes the type of DNS record the service should create.
 *
 * @stability stable
 */
export declare enum ApplicationLoadBalancedServiceRecordType {
    /**
     * Create Route53 A Alias record.
     *
     * @stability stable
     */
    ALIAS = 0,
    /**
     * Create a CNAME record.
     *
     * @stability stable
     */
    CNAME = 1,
    /**
     * Do not create any DNS records.
     *
     * @stability stable
     */
    NONE = 2
}
/**
 * The properties for the base ApplicationLoadBalancedEc2Service or ApplicationLoadBalancedFargateService service.
 *
 * @stability stable
 */
export interface ApplicationLoadBalancedServiceBaseProps {
    /**
     * The name of the cluster that hosts the service.
     *
     * If a cluster is specified, the vpc construct should be omitted. Alternatively, you can omit both cluster and vpc.
     *
     * @default - create a new cluster; if both cluster and vpc are omitted, a new VPC will be created for you.
     * @stability stable
     */
    readonly cluster?: ICluster;
    /**
     * The VPC where the container instances will be launched or the elastic network interfaces (ENIs) will be deployed.
     *
     * If a vpc is specified, the cluster construct should be omitted. Alternatively, you can omit both vpc and cluster.
     *
     * @default - uses the VPC defined in the cluster or creates a new VPC.
     * @stability stable
     */
    readonly vpc?: IVpc;
    /**
     * The properties required to create a new task definition.
     *
     * TaskDefinition or TaskImageOptions must be specified, but not both.
     *
     * @default none
     * @stability stable
     */
    readonly taskImageOptions?: ApplicationLoadBalancedTaskImageOptions;
    /**
     * Determines whether the Load Balancer will be internet-facing.
     *
     * @default true
     * @stability stable
     */
    readonly publicLoadBalancer?: boolean;
    /**
     * Determines whether or not the Security Group for the Load Balancer's Listener will be open to all traffic by default.
     *
     * @default true -- The security group allows ingress from all IP addresses.
     * @stability stable
     */
    readonly openListener?: boolean;
    /**
     * The desired number of instantiations of the task definition to keep running on the service.
     *
     * The minimum value is 1
     *
     * @default - If the feature flag, ECS_REMOVE_DEFAULT_DESIRED_COUNT is false, the default is 1;
     * if true, the default is 1 for all new services and uses the existing services desired count
     * when updating an existing service.
     * @stability stable
     */
    readonly desiredCount?: number;
    /**
     * Certificate Manager certificate to associate with the load balancer.
     *
     * Setting this option will set the load balancer protocol to HTTPS.
     *
     * @default - No certificate associated with the load balancer, if using
     * the HTTP protocol. For HTTPS, a DNS-validated certificate will be
     * created for the load balancer's specified domain name.
     * @stability stable
     */
    readonly certificate?: ICertificate;
    /**
     * The protocol for connections from the load balancer to the ECS tasks.
     *
     * The default target port is determined from the protocol (port 80 for
     * HTTP, port 443 for HTTPS).
     *
     * @default HTTP.
     * @stability stable
     */
    readonly targetProtocol?: ApplicationProtocol;
    /**
     * The protocol for connections from clients to the load balancer.
     *
     * The load balancer port is determined from the protocol (port 80 for
     * HTTP, port 443 for HTTPS).  A domain name and zone must be also be
     * specified if using HTTPS.
     *
     * @default HTTP. If a certificate is specified, the protocol will be
     * set by default to HTTPS.
     * @stability stable
     */
    readonly protocol?: ApplicationProtocol;
    /**
     * The protocol version to use.
     *
     * @default ApplicationProtocolVersion.HTTP1
     * @stability stable
     */
    readonly protocolVersion?: ApplicationProtocolVersion;
    /**
     * The domain name for the service, e.g. "api.example.com.".
     *
     * @default - No domain name.
     * @stability stable
     */
    readonly domainName?: string;
    /**
     * The Route53 hosted zone for the domain, e.g. "example.com.".
     *
     * @default - No Route53 hosted domain zone.
     * @stability stable
     */
    readonly domainZone?: IHostedZone;
    /**
     * The name of the service.
     *
     * @default - CloudFormation-generated name.
     * @stability stable
     */
    readonly serviceName?: string;
    /**
     * The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.
     *
     * @default - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
     * @stability stable
     */
    readonly healthCheckGracePeriod?: cdk.Duration;
    /**
     * The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.
     *
     * @default - 100 if daemon, otherwise 200
     * @stability stable
     */
    readonly maxHealthyPercent?: number;
    /**
     * The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.
     *
     * @default - 0 if daemon, otherwise 50
     * @stability stable
     */
    readonly minHealthyPercent?: number;
    /**
     * The application load balancer that will serve traffic to the service.
     *
     * The VPC attribute of a load balancer must be specified for it to be used
     * to create a new service with this pattern.
     *
     * [disable-awslint:ref-via-interface]
     *
     * @default - a new load balancer will be created.
     * @stability stable
     */
    readonly loadBalancer?: IApplicationLoadBalancer;
    /**
     * Listener port of the application load balancer that will serve traffic to the service.
     *
     * @default - The default listener port is determined from the protocol (port 80 for HTTP,
     * port 443 for HTTPS). A domain name and zone must be also be specified if using HTTPS.
     * @stability stable
     */
    readonly listenerPort?: number;
    /**
     * Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.
     *
     * Tags can only be propagated to the tasks within the service during service creation.
     *
     * @default - none
     * @stability stable
     */
    readonly propagateTags?: PropagatedTagSource;
    /**
     * Specifies whether to enable Amazon ECS managed tags for the tasks within the service.
     *
     * For more information, see
     * [Tagging Your Amazon ECS Resources](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html)
     *
     * @default false
     * @stability stable
     */
    readonly enableECSManagedTags?: boolean;
    /**
     * The options for configuring an Amazon ECS service to use service discovery.
     *
     * @default - AWS Cloud Map service discovery is not enabled.
     * @stability stable
     */
    readonly cloudMapOptions?: CloudMapOptions;
    /**
     * Specifies whether the load balancer should redirect traffic on port 80 to port 443 to support HTTP->HTTPS redirects This is only valid if the protocol of the ALB is HTTPS.
     *
     * @default false
     * @stability stable
     */
    readonly redirectHTTP?: boolean;
    /**
     * Specifies whether the Route53 record should be a CNAME, an A record using the Alias feature or no record at all.
     *
     * This is useful if you need to work with DNS systems that do not support alias records.
     *
     * @default ApplicationLoadBalancedServiceRecordType.ALIAS
     * @stability stable
     */
    readonly recordType?: ApplicationLoadBalancedServiceRecordType;
    /**
     * Specifies which deployment controller to use for the service.
     *
     * For more information, see
     * [Amazon ECS Deployment Types](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html)
     *
     * @default - Rolling update (ECS)
     * @stability stable
     */
    readonly deploymentController?: DeploymentController;
    /**
     * Whether to enable the deployment circuit breaker.
     *
     * If this property is defined, circuit breaker will be implicitly
     * enabled.
     *
     * @default - disabled
     * @stability stable
     */
    readonly circuitBreaker?: DeploymentCircuitBreaker;
}
/**
 * @stability stable
 */
export interface ApplicationLoadBalancedTaskImageOptions {
    /**
     * The image used to start a container.
     *
     * Image or taskDefinition must be specified, not both.
     *
     * @default - none
     * @stability stable
     */
    readonly image: ContainerImage;
    /**
     * The environment variables to pass to the container.
     *
     * @default - No environment variables.
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * The secret to expose to the container as an environment variable.
     *
     * @default - No secret environment variables.
     * @stability stable
     */
    readonly secrets?: {
        [key: string]: Secret;
    };
    /**
     * Flag to indicate whether to enable logging.
     *
     * @default true
     * @stability stable
     */
    readonly enableLogging?: boolean;
    /**
     * The log driver to use.
     *
     * @default - AwsLogDriver if enableLogging is true
     * @stability stable
     */
    readonly logDriver?: LogDriver;
    /**
     * The name of the task execution IAM role that grants the Amazon ECS container agent permission to call AWS APIs on your behalf.
     *
     * @default - No value
     * @stability stable
     */
    readonly executionRole?: IRole;
    /**
     * The name of the task IAM role that grants containers in the task permission to call AWS APIs on your behalf.
     *
     * @default - A task role is automatically created for you.
     * @stability stable
     */
    readonly taskRole?: IRole;
    /**
     * The container name value to be specified in the task definition.
     *
     * @default - none
     * @stability stable
     */
    readonly containerName?: string;
    /**
     * The port number on the container that is bound to the user-specified or automatically assigned host port.
     *
     * If you are using containers in a task with the awsvpc or host network mode, exposed ports should be specified using containerPort.
     * If you are using containers in a task with the bridge network mode and you specify a container port and not a host port,
     * your container automatically receives a host port in the ephemeral port range.
     *
     * Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.
     *
     * For more information, see
     * [hostPort](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PortMapping.html#ECS-Type-PortMapping-hostPort).
     *
     * @default 80
     * @stability stable
     */
    readonly containerPort?: number;
    /**
     * The name of a family that this task definition is registered to.
     *
     * A family groups multiple versions of a task definition.
     *
     * @default - Automatically generated name.
     * @stability stable
     */
    readonly family?: string;
}
/**
 * The base class for ApplicationLoadBalancedEc2Service and ApplicationLoadBalancedFargateService services.
 *
 * @stability stable
 */
export declare abstract class ApplicationLoadBalancedServiceBase extends CoreConstruct {
    /**
     * (deprecated) The desired number of instantiations of the task definition to keep running on the service.
     *
     * @deprecated - Use `internalDesiredCount` instead.
     */
    readonly desiredCount: number;
    /**
     * The desired number of instantiations of the task definition to keep running on the service.
     *
     * The default is 1 for all new services and uses the existing services desired count
     * when updating an existing service if one is not provided.
     *
     * @stability stable
     */
    readonly internalDesiredCount?: number;
    /**
     * The Application Load Balancer for the service.
     *
     * @stability stable
     */
    get loadBalancer(): ApplicationLoadBalancer;
    /**
     * The listener for the service.
     *
     * @stability stable
     */
    readonly listener: ApplicationListener;
    /**
     * The redirect listener for the service if redirectHTTP is enabled.
     *
     * @stability stable
     */
    readonly redirectListener?: ApplicationListener;
    /**
     * The target group for the service.
     *
     * @stability stable
     */
    readonly targetGroup: ApplicationTargetGroup;
    /**
     * Certificate Manager certificate to associate with the load balancer.
     *
     * @stability stable
     */
    readonly certificate?: ICertificate;
    /**
     * The cluster that hosts the service.
     *
     * @stability stable
     */
    readonly cluster: ICluster;
    private readonly _applicationLoadBalancer?;
    /**
     * Constructs a new instance of the ApplicationLoadBalancedServiceBase class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: ApplicationLoadBalancedServiceBaseProps);
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    protected getDefaultCluster(scope: CoreConstruct, vpc?: IVpc): Cluster;
    /**
     * Adds service as a target of the target group.
     *
     * @stability stable
     */
    protected addServiceAsTarget(service: BaseService): void;
    /**
     * @stability stable
     */
    protected createAWSLogDriver(prefix: string): AwsLogDriver;
}
