"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
// Batch size that limits how many outgoing S3 calls are made at a time.
// This can be tweaked as needed (increased if we want to squeeze out more
// performance, or decreased if we are getting too throttled).
const BATCH_SIZE = 200;
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const VERSION_TRACKER_BUCKET_NAME = env_lambda_shared_1.requireEnv(constants_1.ENV_VERSION_TRACKER_BUCKET_NAME);
    const VERSION_TRACKER_OBJECT_KEY = env_lambda_shared_1.requireEnv(constants_1.ENV_VERSION_TRACKER_OBJECT_KEY);
    const PACKAGE_DATA_BUCKET_NAME = env_lambda_shared_1.requireEnv(constants_1.ENV_PACKAGE_DATA_BUCKET_NAME);
    const PACKAGE_DATA_KEY_PREFIX = env_lambda_shared_1.requireEnv(constants_1.ENV_PACKAGE_DATA_KEY_PREFIX);
    const PACKAGE_PREFIX_REGEX = new RegExp('^' + PACKAGE_DATA_KEY_PREFIX + '((?:@[^/]+\/)?[^/]+)\/v([^/]+)\/$');
    // Gather a list of all package prefixes.
    const packagePrefixes = await listPackagePrefixes(PACKAGE_DATA_BUCKET_NAME, PACKAGE_DATA_KEY_PREFIX);
    // Collect the list of versions for each package in parallel,
    // batched as needed to limit network throughput.
    const requestBatches = groupIntoBatches(packagePrefixes, BATCH_SIZE);
    const versionMap = new Map();
    for (const [idx, batch] of requestBatches.entries()) {
        console.log(`Batch ${idx} of ${requestBatches.length}`);
        const promises = batch.map(async (packagePrefix) => {
            const versionPrefixes = await listPrefixes(PACKAGE_DATA_BUCKET_NAME, packagePrefix);
            if (versionPrefixes.length === 0)
                return;
            const [, name] = PACKAGE_PREFIX_REGEX.exec(versionPrefixes[0]);
            const versions = new Array();
            for (const versionPrefix of versionPrefixes) {
                const [, , version] = PACKAGE_PREFIX_REGEX.exec(versionPrefix);
                versions.push(version);
            }
            versionMap.set(name, versions);
        });
        await Promise.all(promises);
    }
    const versionJson = {
        packages: Object.fromEntries(versionMap),
        updatedAt: new Date().toISOString(),
    };
    let totalVersions = 0;
    versionMap.forEach((versions) => { totalVersions += versions.length; });
    // Update metrics.
    console.log(`${versionMap.size} package versions have been recorded.`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73
        metrics.setDimensions();
        metrics.setNamespace(constants_1.METRICS_NAMESPACE);
        metrics.putMetric("TrackedPackagesCount" /* TRACKED_PACKAGES_COUNT */, versionMap.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("TrackedVersionsCount" /* TRACKED_VERSIONS_COUNT */, totalVersions, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    const result = await aws.s3().putObject({
        Bucket: VERSION_TRACKER_BUCKET_NAME,
        Key: VERSION_TRACKER_OBJECT_KEY,
        Body: JSON.stringify(versionJson),
        ContentType: 'application/json',
        CacheControl: caching_1.CacheStrategy.default().toString(),
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${versionMap.size}`,
            'Version-Count': `${totalVersions}`,
        },
    }).promise();
    return result;
}
exports.handler = handler;
/**
 * List all prefixes in a bucket given a base prefix.
 *
 * For example, if given "data/" it will return
 * [
 *   "data/@aws-amplify/",
 *   "data/@aws-c2a/",
 *   "data/@aws-cdk/",
 *   ...
 * ]
 */
async function listPrefixes(bucket, prefix) {
    var _a;
    const prefixes = new Array();
    let continuationToken;
    do {
        const listRequest = {
            Bucket: bucket,
            Prefix: prefix,
            Delimiter: '/',
            ContinuationToken: continuationToken,
        };
        const listResponse = await aws.s3().listObjectsV2(listRequest).promise();
        continuationToken = listResponse.NextContinuationToken;
        for (const { Prefix: commonPrefix } of (_a = listResponse.CommonPrefixes) !== null && _a !== void 0 ? _a : []) {
            if (!commonPrefix) {
                continue;
            }
            prefixes.push(commonPrefix);
        }
    } while (continuationToken);
    return prefixes;
}
/**
 * List all package prefixes in a bucket given a base prefix.
 *
 * For example, if given "data/" it will return
 * [
 *   "data/@aws-amplify/cdk-exported-backend",
 *   "data/@aws-c2a/cdk-pipelines-step",
 *   "data/@aws-cdk/alexa-ask",
 *   "data/@aws-cdk/app-delivery",
 *   "data/@aws-cdk/assertions",
 *   ...
 * ]
 */
async function listPackagePrefixes(bucket, prefix) {
    const packagePrefixes = new Array();
    // gather a list of all package scopes and unscoped packages
    const initialPrefixes = await listPrefixes(bucket, prefix);
    const scopedPrefixes = initialPrefixes.filter((p) => p === null || p === void 0 ? void 0 : p.startsWith(`${prefix}@`));
    const unscopedPrefixes = initialPrefixes.filter((p) => !(p === null || p === void 0 ? void 0 : p.startsWith(`${prefix}@`)));
    // scoped packages need to be collected separately, so we
    // group the requests into batches, run them in parallel, and
    // flatten the results to an output list
    const batches = groupIntoBatches(scopedPrefixes, BATCH_SIZE);
    for (const batch of batches) {
        const promises = batch.map(async (scopedPrefix) => listPrefixes(bucket, scopedPrefix));
        const results = await Promise.all(promises);
        packagePrefixes.push(...results.flat());
    }
    packagePrefixes.push(...unscopedPrefixes);
    return packagePrefixes;
}
/**
 * Partition an array into contiguous subsequences.
 */
function groupIntoBatches(arr, batchSize) {
    const batches = [];
    for (let i = 0; i < arr.length; i += batchSize) {
        batches.push(arr.slice(i, i + batchSize));
    }
    return batches;
}
//# sourceMappingURL=data:application/json;base64,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