"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const markdown_render_1 = require("jsii-docgen/lib/docgen/render/markdown-render");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                     ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const deleted = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const uploads = new Map();
        const deletions = new Map();
        let unprocessable = false;
        function markPackage(e, marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const upload = uploadFile(event.bucket, key, event.assembly.versionId, Buffer.from(e.message));
            uploads.set(key, upload);
        }
        async function unmarkPackage(marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const marked = await aws.s3ObjectExists(event.bucket, key);
            if (!marked) {
                return;
            }
            const deletion = deleteFile(event.bucket, key);
            deletions.set(key, deletion);
        }
        console.log(`Generating documentation for ${packageFqn}...`);
        try {
            const docs = await docgen.Documentation.forPackage(tarball, { name: assembly.name });
            // if the package used to not be installabe, remove the marker for it.
            await unmarkPackage(constants.UNINSTALLABLE_PACKAGE_SUFFIX);
            for (const language of language_1.DocumentationLanguage.ALL) {
                if (event.languages && !event.languages[language.toString()]) {
                    console.log(`Skipping language ${language} as it was not requested!`);
                    continue;
                }
                const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                    metrics.setDimensions();
                    metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                    async function renderAndDispatch(submodule) {
                        try {
                            console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                            const json = await docs.toJson({
                                submodule,
                                language: docgen.Language.fromString(lang.name),
                            });
                            const jsonPage = Buffer.from(json.render(null, 2));
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, jsonPage.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: jsonBody, contentEncoding: jsonContentEncoding } = compress_content_lambda_shared_1.compressContent(jsonPage);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, jsonBody.length, aws_embedded_metrics_1.Unit.Bytes);
                            const jsonKey = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'json'));
                            console.log(`Uploading ${jsonKey}`);
                            const jsonUpload = uploadFile(event.bucket, jsonKey, event.assembly.versionId, jsonBody, jsonContentEncoding);
                            uploads.set(jsonKey, jsonUpload);
                            const markdown = markdown_render_1.MarkdownRenderer.fromSchema(json.content, {
                                anchorFormatter,
                                linkFormatter: linkFormatter(lang),
                            });
                            const page = Buffer.from(markdown.render());
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: body, contentEncoding } = compress_content_lambda_shared_1.compressContent(page);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                            const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'md'));
                            console.log(`Uploading ${key}`);
                            const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                            uploads.set(key, upload);
                            // if the package used to have a corrupt assembly, remove the marker for it.
                            await unmarkPackage(constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                        }
                        catch (e) {
                            if (e instanceof docgen.LanguageNotSupportedError) {
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'md'));
                            }
                            else if (e instanceof docgen.CorruptedAssemblyError) {
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                                unprocessable = true;
                            }
                            else {
                                throw e;
                            }
                        }
                    }
                    await renderAndDispatch();
                    for (const submodule of submodules) {
                        await renderAndDispatch(submodule);
                    }
                });
                await generateDocs(language);
            }
        }
        catch (error) {
            if (error instanceof docgen.UnInstallablePackageError) {
                markPackage(error, constants.UNINSTALLABLE_PACKAGE_SUFFIX);
                unprocessable = true;
            }
            else {
                throw error;
            }
        }
        for (const [key, upload] of uploads.entries()) {
            const response = await upload;
            created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
        }
        for (const [key, deletion] of deletions.entries()) {
            const response = await deletion;
            deleted.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished deleting ${key} (Version ID: ${response.VersionId})`);
        }
        if (unprocessable) {
            // the message here doesn't matter, we only use the error name
            // to divert this message away from the DLQ.
            const error = new Error();
            error.name = constants.UNPROCESSABLE_PACKAGE_ERROR_NAME;
        }
        return { created, deleted };
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: caching_1.CacheStrategy.default().toString(),
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
function deleteFile(bucket, key) {
    return aws.s3().deleteObject({
        Bucket: bucket,
        Key: key,
    }).promise();
}
function anchorFormatter(type) {
    const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
    const [base, ...rest] = name.split('.');
    if (rest.length > 0) {
        return sanitize(rest.join('.')); // Initializer.parameter.accessControl
    }
    else {
        return sanitize(base);
    }
}
function linkFormatter(lang) {
    const formatter = (type) => {
        const name = getAssemblyRelativeName(type); // BucketProps.parameter.accessControl
        const [baseName, ...rest] = name.split('.');
        const hash = '#' + rest.join('.'); // #parameter.accessControl
        const langParam = `?lang=${lang.toString()}`;
        const submoduleParam = type.submodule ? `&submodule=${type.submodule}` : '';
        return `<a href="/packages/${type.packageName}/v/${type.packageVersion}/api/${baseName}${langParam}${submoduleParam}${hash}">${type.displayName}</a>`;
    };
    return formatter;
}
/**
 * Converts a type's id to an assembly-relative version, e.g.:
 * `aws-cdk-lib.aws_s3.Bucket.parameter.accessControl` => `Bucket.parameter.accessControl`
 */
function getAssemblyRelativeName(type) {
    const parts = type.id.split('.');
    if (type.packageName) {
        if (parts[0] !== type.packageName) {
            throw new Error(`Expected first part of type id "${type.id}" to be equal to ${type.packageName}.`);
        }
        parts.shift();
        if (type.submodule) {
            if (parts[0] !== type.submodule) {
                throw new Error(`Expected second part of type id "${type.id}" to be equal to ${type.packageName}.`);
            }
            parts.shift();
        }
    }
    return parts.join('.');
}
;
function sanitize(str) {
    // HTML5 allows any characters in IDs except whitespace
    return str.replace(/ /g, '-');
}
//# sourceMappingURL=data:application/json;base64,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