"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        const repository = ((_a = props.vpcEndpoints) === null || _a === void 0 ? void 0 : _a.codeArtifact) && props.vpcEndpoints.codeArtifactApi
            ? (_b = props.codeArtifact) === null || _b === void 0 ? void 0 : _b.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            // Set embedded metrics format environment to "Local", to have a consistent experience.
            AWS_EMF_ENVIRONMENT: 'Local',
        };
        if ((_c = props.vpcEndpoints) === null || _c === void 0 ? void 0 : _c.codeArtifactApi) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        this.logGroup = new aws_logs_1.LogGroup(this, 'LogGroup', { retention: props.logRetention });
        this.containerDefinition = new transliterator_1.Transliterator(this, 'Resource', {
            environment,
            logging: aws_ecs_1.LogDrivers.awsLogs({ logGroup: this.logGroup, streamPrefix: 'transliterator' }),
            taskDefinition: new aws_ecs_1.FargateTaskDefinition(this, 'TaskDefinition', {
                cpu: 4096,
                memoryLimitMiB: 8192,
            }),
        });
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.taskDefinition.taskRole);
        // The task handler reads & writes to this bucket.
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.PACKAGE_KEY_SUFFIX}`);
        bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
        }
        const executionRole = this.taskDefinition.obtainExecutionRole();
        (_d = props.vpcEndpoints) === null || _d === void 0 ? void 0 : _d.ecrApi.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
            ],
            resources: ['*'],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_e = props.vpcEndpoints) === null || _e === void 0 ? void 0 : _e.ecr.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
            // We cannot get the ECR repository info from an asset... So scoping down to same-account repositories instead...
            resources: [core_1.Stack.of(this).formatArn({ service: 'ecr', resource: 'repository', arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME, resourceName: '*' })],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_f = props.vpcEndpoints) === null || _f === void 0 ? void 0 : _f.cloudWatchLogs.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: this.logGroup.logGroupName }),
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: `${this.logGroup.logGroupName}:log-stream:*` }),
            ],
            principals: [executionRole],
            sid: 'Allow-Logging',
        }));
        (_g = props.vpcEndpoints) === null || _g === void 0 ? void 0 : _g.stepFunctions.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
                'states:SendTaskSuccess',
            ],
            resources: ['*'],
            principals: [this.taskDefinition.taskRole],
            sid: 'Allow-StepFunctions-Callbacks',
        }));
    }
    get taskDefinition() {
        return this.containerDefinition.taskDefinition;
    }
    createEcsRunTask(scope, id, opts) {
        return new aws_stepfunctions_tasks_1.EcsRunTask(scope, id, {
            // The container sends heartbeats every minute, but when the runloop will
            // actually get to submitting it is fairly dependent on the async
            // workload's nature... so we don't rely on it all too strongly, and
            // default to a 5 minutes timeout here as a minimal protection. Options
            // may override this liberally if they know better.
            heartbeat: core_1.Duration.minutes(5),
            ...opts,
            containerOverrides: [{
                    containerDefinition: this.containerDefinition,
                    command: aws_stepfunctions_1.JsonPath.listAt('$'),
                    environment: [
                        { name: 'SFN_TASK_TOKEN', value: aws_stepfunctions_1.JsonPath.taskToken },
                    ],
                }],
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            launchTarget: new aws_stepfunctions_tasks_1.EcsFargateLaunchTarget({ platformVersion: aws_ecs_1.FargatePlatformVersion.VERSION1_4 }),
            subnets: opts.vpcSubnets,
            securityGroups: opts.securityGroups,
            taskDefinition: this.taskDefinition,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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