"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inventory = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const canary_1 = require("./canary");
const constants_2 = require("./constants");
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
class Inventory extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rate = (_a = props.scheduleRate) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5);
        this.canary = new canary_1.Canary(this, 'Resource', {
            description: '[ConstructHub/Inventory] A canary that periodically inspects the list of indexed packages',
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: props.bucket.bucketName,
            },
            logRetention: props.logRetention,
            memorySize: 10240,
            timeout: rate,
        });
        const grantRead = props.bucket.grantRead(this.canary);
        const grantWriteMissing = props.bucket.grantWrite(this.canary, constants_1.MISSING_DOCUMENTATION_REPORT_PATTERN);
        const grantWriteCorruptAssembly = props.bucket.grantWrite(this.canary, constants_1.CORRUPT_ASSEMBLY_REPORT_PATTERN);
        const grantWriteUnInstallable = props.bucket.grantWrite(this.canary, constants_1.UNINSTALLABLE_PACKAGES_REPORT);
        const rule = new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(rate),
            targets: [new aws_events_targets_1.LambdaFunction(this.canary)],
        });
        rule.node.addDependency(grantRead, grantWriteMissing);
        rule.node.addDependency(grantRead, grantWriteCorruptAssembly);
        rule.node.addDependency(grantRead, grantWriteUnInstallable);
        props.monitoring.addHighSeverityAlarm('Inventory Canary is not Running', this.canary.metricInvocations({ period: core_1.Duration.minutes(5) }).createAlarm(this, 'Not Running', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The inventory canary is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        props.monitoring.addHighSeverityAlarm('Inventory Canary is failing', this.canary.metricErrors({ period: core_1.Duration.minutes(5) }).createAlarm(this, 'Failures', {
            alarmName: `${this.node.path}/Failures`,
            alarmDescription: [
                'The inventory canary is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
        }));
    }
    get function() {
        return this.canary;
    }
    metricMissingPackageMetadataCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricMissingAssemblyCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricPackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageCount" /* PACKAGE_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricPackageMajorCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricMissingPackageTarballCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricUninstallablePackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UninstallablePackageCount" /* UNINSTALLABLE_PACKAGE_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricSubmoduleCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "SubmoduleCount" /* SUBMODULE_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricUnknownObjectCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages for which all versions are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions for which all versions are missing a
     * documnetation artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that are missing a documnetation
     * artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that have at least one version for which there is
     * available documentation in the provided `DocumentationLanguage`.
     */
    metricSupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that have at least one version for
     * which there is available documentation in the provided
     * `DocumentationLanguage`.
     */
    metricSupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that have available documentation in the
     * provided `DocumentationLanguage`.
     */
    metricSupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that have available documentation
     * in the provided `DocumentationLanguage`.
     */
    metricSupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblyPackageCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblyMajorVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that have a language specific corrupt assembly.
     */
    metricCorruptAssemblySubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblySubmoduleCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Inventory = Inventory;
//# sourceMappingURL=data:application/json;base64,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