"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
const framework_detection_lambda_shared_1 = require("./framework-detection.lambda-shared");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b, _c;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const CONFIG_BUCKET_NAME = env_lambda_shared_1.requireEnv('CONFIG_BUCKET_NAME');
    const CONFIG_FILE_KEY = env_lambda_shared_1.requireEnv('CONFIG_FILE_KEY');
    // Load configuration
    const { packageTags: packageTagsConfig, packageLinks: allowedLinks } = await getConfig(CONFIG_BUCKET_NAME, CONFIG_FILE_KEY);
    const codeArtifactProps = (function () {
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            return undefined;
        }
        const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
        const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
        const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
        return { endpoint, domain, domainOwner, apiEndpoint };
    })();
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let parsedAssembly;
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        let packageReadme;
        try {
            parsedAssembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            // needs `dependencyClosure`
            constructFramework = framework_detection_lambda_shared_1.detectConstructFramework(parsedAssembly);
            const { license, name, version, readme } = parsedAssembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            packageReadme = (_c = readme === null || readme === void 0 ? void 0 : readme.markdown) !== null && _c !== void 0 ? _c : '';
            // Delete some fields not used by the client to reduce the size of the assembly.
            // See https://github.com/cdklabs/construct-hub-webapp/issues/691
            delete parsedAssembly.types;
            delete parsedAssembly.readme;
            delete parsedAssembly.dependencyClosure;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, { pkg: packageJsonObj, readme: packageReadme })) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        if (codeArtifactProps) {
            console.log('Publishing to the internal CodeArtifact...');
            try {
                const { publishConfig } = packageJsonObj;
                if (publishConfig) {
                    console.log('Not publishing to CodeArtifact due to the presence of publishConfig in package.json: ', publishConfig);
                }
                else {
                    await code_artifact_lambda_shared_1.codeArtifactPublishPackage(Buffer.from(tarball.Body), codeArtifactProps);
                }
            }
            catch (err) {
                console.error('Failed publishing to CodeArtifact: ', err);
            }
        }
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: caching_1.CacheStrategy.default().toString(),
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: caching_1.CacheStrategy.default().toString(),
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: Buffer.from(JSON.stringify(parsedAssembly), 'utf-8'),
            CacheControl: caching_1.CacheStrategy.default().toString(),
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
/**
 * Looks for the ingestion configuration file in the passed bucket and parses
 * it. If it is not found or invalid then a default is returned.
 */
async function getConfig(bucket, key) {
    var _a;
    const defaultConfig = {
        packageTags: [],
        packageLinks: [],
    };
    try {
        const req = await aws.s3().getObject({
            Bucket: bucket,
            Key: key,
        }).promise();
        const body = (_a = req === null || req === void 0 ? void 0 : req.Body) === null || _a === void 0 ? void 0 : _a.toString();
        if (body) {
            return JSON.parse(body);
        }
        return defaultConfig;
    }
    catch (e) {
        console.error(e);
        return defaultConfig;
    }
}
//# sourceMappingURL=data:application/json;base64,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