"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogBuilder = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("./catalog-builder");
const constants_1 = require("./constants");
/**
 * Builds or re-builds the `catalog.json` object in the designated bucket.
 */
class CatalogBuilder extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const handler = new catalog_builder_1.CatalogBuilder(this, 'Default', {
            description: `Creates the catalog.json object in ${props.bucket.bucketName}`,
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                AWS_EMF_ENVIRONMENT: 'Local',
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
        });
        this.function = handler;
        // This function may invoke itself in case it needs to continue it's work in
        // a "child" invocation. We must hence allow it to invoke itself. We cannot
        // use grantInvoke as this would (naturally) cause a circular reference
        // (Function -> Role -> Function).
        handler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [core_1.Stack.of(this).formatArn({
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                    service: 'lambda',
                    resource: 'function',
                    resourceName: '*',
                })],
        }));
        // allow the catalog builder to use the client.
        props.denyList.grantRead(handler);
        props.bucket.grantReadWrite(this.function);
        // Monitor the derivative of the catalog size, and alarm if the catalog
        // loses more than 5 items. Catalog elements can disappear if they are added
        // to the deny-list, or if they get un-published from the origin repository.
        // Such cases are normal and shouldn't typically result in a significant
        // contraction of the catalog size.
        const catalogSizeChange = new aws_cloudwatch_1.MathExpression({
            expression: 'DIFF(FILL(m1, REPEAT))',
            period: core_1.Duration.minutes(15),
            usingMetrics: { m1: this.metricRegisteredPackageMajorVersions() },
        });
        const alarmShrinkingCatalog = catalogSizeChange.createAlarm(this, 'ShrinkingCatalogAlarm', {
            alarmName: `${this.node.path}/ShrinkingCatalog`,
            alarmDescription: [
                'The number of packages registered in the catalog.json object has shrunk by more than 5',
                'elements. There might be a mass extinction event going on. This should be investigated',
                'as soon as possible.',
                '',
                `Catalog.json: ${deep_link_1.s3ObjectUrl(props.bucket, 'catalog.json')}`,
                `Catalog Builder: ${deep_link_1.lambdaFunctionUrl(handler)}`,
                '',
                `RUNBOOK: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: -5,
        });
        props.monitoring.addHighSeverityAlarm('Catalog Size Shrunk', alarmShrinkingCatalog);
    }
    metricMissingConstructFrameworkCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(15),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingConstructFrameworkCount" /* MISSING_CONSTRUCT_FRAMEWORK_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingConstructFrameworkVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(15),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingConstructFrameworkVersionCount" /* MISSING_CONSTRUCT_FRAMEWORK_VERSION_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricRegisteredPackageMajorVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(15),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "RegisteredPackagesMajorVersion" /* REGISTERED_PACKAGES_MAJOR_VERSION */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.CatalogBuilder = CatalogBuilder;
//# sourceMappingURL=data:application/json;base64,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