"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogClient = exports.CatalogNotFoundError = void 0;
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
class CatalogNotFoundError extends Error {
    constructor(key) {
        super(`No catalog was found at ${key}`);
    }
}
exports.CatalogNotFoundError = CatalogNotFoundError;
/**
 * A client for working with the catalog.
 */
class CatalogClient {
    constructor() {
        this.bucketName = env_lambda_shared_1.requireEnv('CATALOG_BUCKET_NAME');
        this.objectKey = env_lambda_shared_1.requireEnv('CATALOG_OBJECT_KEY');
        this.s3 = aws_lambda_shared_1.s3();
    }
    /**
     * Creates a new client for accessing the catalog.
     */
    static async newClient() {
        const client = new CatalogClient();
        await client.init();
        return client;
    }
    /**
     * Downloads the catalog object and stores it in memory.
     */
    async init() {
        if (this._packages) {
            throw new Error('init() cannot be called twice');
        }
        this._packages = [];
        const params = {
            Bucket: this.bucketName,
            Key: this.objectKey,
        };
        let body;
        try {
            const data = await this.s3.getObject(params).promise();
            body = data.Body;
        }
        catch (e) {
            throw new CatalogNotFoundError(`${this.bucketName}/${this.objectKey}`);
        }
        if (!body) {
            throw new Error(`Catalog body is empty at ${this.bucketName}/${this.objectKey}`);
        }
        const contents = body.toString('utf-8');
        try {
            const data = JSON.parse(contents);
            if (typeof (data) != 'object') {
                throw new Error('Invalid format in catalog file. Expecting a map');
            }
            this._packages = data.packages;
        }
        catch (e) {
            throw new Error(`Unable to parse catalog file ${this.bucketName}/${this.objectKey}: ${e}`);
        }
    }
    /**
     * Returns a copy of the catalog list.
     */
    get packages() {
        if (!this._packages) {
            throw new Error('CatalogClient must be initialized');
        }
        return [...this._packages];
    }
}
exports.CatalogClient = CatalogClient;
//# sourceMappingURL=data:application/json;base64,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