"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parsePrefixList = exports.createRestrictedSecurityGroups = void 0;
const crypto_1 = require("crypto");
const fs = require("fs");
const path_1 = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const s3_1 = require("./s3");
/**
   * Creates SecurityGroups where "sensitive" operations should be listed,
   * which only allows DNS requests to be issued within the VPC (to the local
   * Route53 resolver), as well as HTTPS (port 443) traffic to:
   * - allow-listed IP ranges
   * - endpoints within the same SecurityGroup.
   *
   * This returns MULTIPLE security groups in order to avoid hitting the maximum
   * count of rules per security group, which is relatively low, and prefix
   * lists count as their expansions.
   *
   * There is also a limit of how many security groups can be bound to a network
   * interface (defaults to 5), so there is only so much we can do here.
   *
   * @param scope the scope in which to attach new constructs.
   * @param vpc the VPC in which a SecurityGroup is to be added.
   */
function createRestrictedSecurityGroups(scope, vpc) {
    const securityGroups = new Array();
    securityGroups.push(createInternalTrafficSecurityGroup(scope, vpc));
    const ALLOW_LIST_DIR = path_1.resolve(__dirname, '..', 'resources', 'vpc-allow-lists');
    for (const file of fs.readdirSync(ALLOW_LIST_DIR)) {
        const matches = /^(.+)-(IPv4|IPv6)\.txt$/.exec(file);
        if (matches == null) {
            throw new Error(`Allow-list file ${file} in ${ALLOW_LIST_DIR} is invalid: file name must end in IPv4.txt or IPv6.txt`);
        }
        const [, namespace, ipLabel] = matches;
        const entries = parsePrefixList(path_1.join(ALLOW_LIST_DIR, file));
        if (entries.length === 0) {
            continue;
        }
        // We use a SHA-1 digest of the list of prefixes to be sure we create a
        // whole new prefix list whenever it changes, so we are never bothered by
        // the maxEntries being what it is.
        const hash = entries.reduce((h, { cidr }) => h.update(cidr).update('\0'), crypto_1.createHash('SHA1')).digest('hex');
        // Note - the `prefixListName` is NOT a physical ID, and so updating it
        // will NOT cause a replacement. Additionally, it needs not be unique in
        // any way.
        const pl = new aws_ec2_1.CfnPrefixList(scope, `${namespace}.${ipLabel}#${hash}`, {
            addressFamily: ipLabel,
            prefixListName: `${namespace}.${ipLabel}`,
            entries,
            maxEntries: entries.length,
        });
        // Note: the CfnPrefixList above uses a `.` separator bewteen namespace and
        // ipLabel, since we use a `-` here, we are not colliding. The has is used
        // here to replace the SG when the PL is updated. Failure to do so may
        // result in the SG having both the old & updated PLs in its rule set until
        // the CloudFormation clean-up phase completes, which might exceed the
        // maximum amount of rules allowed on a SG.
        const descr = `${namespace}-${ipLabel}`;
        const sg = new aws_ec2_1.SecurityGroup(scope, `${descr}#${hash}`, {
            allowAllOutbound: false,
            description: `${scope.node.path}/${descr}`,
            vpc,
        });
        // We intentionally ONLY allow HTTPS though there...
        sg.connections.allowTo(NamedPeer.from(aws_ec2_1.Peer.prefixList(pl.attrPrefixListId), pl.node.path), aws_ec2_1.Port.tcp(443), `to ${namespace} (${ipLabel})`);
        core_1.Tags.of(sg).add('Name', `${namespace}.${ipLabel}`);
        securityGroups.push(sg);
    }
    return securityGroups;
}
exports.createRestrictedSecurityGroups = createRestrictedSecurityGroups;
/**
 * Creates a SecurityGroup that allows traffic to flow freely between
 * endpoints within itself on port 443, to the local Route53 resolver on DNS
 * ports, and to the region's AW S3 prefix list on port 443.
 *
 * @param scope the scope in which to attach the new Security Group.
 * @param vpc the VPC in which the SecurityGroup will be created.
 */
function createInternalTrafficSecurityGroup(scope, vpc) {
    const sg = new aws_ec2_1.SecurityGroup(scope, 'InternalTraffic', {
        allowAllOutbound: false,
        description: `${scope.node.path}/SG`,
        vpc,
    });
    // Allow all traffic within the security group on port 443
    sg.connections.allowInternally(aws_ec2_1.Port.tcp(443), 'Traffic within this SecurityGroup');
    // Allow access to S3. This is needed for the S3 Gateway endpoint to work.
    sg.connections.allowTo(NamedPeer.from(aws_ec2_1.Peer.prefixList(new s3_1.S3PrefixList(scope, 'S3-PrefixList').prefixListId), 'AWS S3'), aws_ec2_1.Port.tcp(443), 'to AWS S3');
    // Allow making DNS requests, there should be a Route53 resolver wihtin the VPC.
    sg.connections.allowTo(aws_ec2_1.Peer.ipv4(vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(53), 'to Route53 DNS resolver');
    sg.connections.allowTo(aws_ec2_1.Peer.ipv4(vpc.vpcCidrBlock), aws_ec2_1.Port.udp(53), 'to Route53 DNS resolver');
    return sg;
}
/**
 * Parses the PrefixList in the designated path.
 *
 * @param filePath the file containing the prefix list.
 */
function parsePrefixList(filePath) {
    return fs.readFileSync(filePath, 'utf8')
        .split(/\n/)
        .map((line) => {
        const match = /^\s*([^\s]+)?\s*(?:#.*)?$/.exec(line);
        if (!match) {
            throw new Error(`Invalid line in allow list ${filePath}: ${line}`);
        }
        const [, cidr] = match;
        return cidr;
    })
        // Remove empty lines.
        .filter((cidr) => !!cidr)
        .sort()
        .map((cidr) => ({ cidr }));
}
exports.parsePrefixList = parsePrefixList;
/**
 * This is to work around an issue where the peer's `uniqueId` is a token for
 * our PrefixList values, and this causes the VPC construct to "de-duplicate"
 * all of them (it considers they are identical).
 *
 * There is a fix in the latest EC2 library, however that fix isn't great
 * either, as it addresses the problem at the wrong location (in the in/egress
 * rule, instead of in the peer).
 *
 * Basically, this ensures the `uniqueId` is some string we control, so we
 * remain faithful to the declaraiton intent.
 */
class NamedPeer {
    constructor(peer, uniqueId) {
        this.peer = peer;
        this.uniqueId = uniqueId;
        this.connections = new aws_ec2_1.Connections({ peer: this });
    }
    static from(peer, name) {
        return new NamedPeer(peer, name);
    }
    get canInlineRule() {
        return this.peer.canInlineRule;
    }
    toIngressRuleConfig() {
        return this.peer.toIngressRuleConfig();
    }
    toEgressRuleConfig() {
        return this.peer.toEgressRuleConfig();
    }
}
//# sourceMappingURL=data:application/json;base64,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