"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const iam = require("@aws-cdk/aws-iam");
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 *
 * @stability stable
 */
class Source {
    /**
     * @stability stable
     */
    constructor(props) {
        /**
         * @stability stable
         */
        this.badgeSupported = false;
        this.identifier = props.identifier;
    }
    /**
     * @stability stable
     */
    static s3(props) {
        return new S3Source(props);
    }
    /**
     * @stability stable
     */
    static codeCommit(props) {
        return new CodeCommitSource(props);
    }
    /**
     * @stability stable
     */
    static gitHub(props) {
        return new GitHubSource(props);
    }
    /**
     * @stability stable
     */
    static gitHubEnterprise(props) {
        return new GitHubEnterpriseSource(props);
    }
    /**
     * @stability stable
     */
    static bitBucket(props) {
        return new BitBucketSource(props);
    }
    /**
     * Called by the project when the source is added so that the source can perform binding operations on the source.
     *
     * For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     *
     * @stability stable
     */
    bind(_scope, _project) {
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
        this.fetchSubmodules = props.fetchSubmodules;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
                gitSubmodulesConfig: this.fetchSubmodules ? {
                    fetchSubmodules: this.fetchSubmodules,
                } : undefined,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 *
 * @stability stable
 */
var EventAction;
(function (EventAction) {
    EventAction["PUSH"] = "PUSH";
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
var WebhookFilterTypes;
(function (WebhookFilterTypes) {
    WebhookFilterTypes["FILE_PATH"] = "FILE_PATH";
    WebhookFilterTypes["COMMIT_MESSAGE"] = "COMMIT_MESSAGE";
    WebhookFilterTypes["HEAD_REF"] = "HEAD_REF";
    WebhookFilterTypes["ACTOR_ACCOUNT_ID"] = "ACTOR_ACCOUNT_ID";
    WebhookFilterTypes["BASE_REF"] = "BASE_REF";
})(WebhookFilterTypes || (WebhookFilterTypes = {}));
/**
 * An object that represents a group of filter conditions for a webhook.
 *
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 *
 * @stability stable
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on.
     * @stability stable
     */
    static inEventOf(...actions) {
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @stability stable
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @stability stable
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression).
     * @stability stable
     */
    andCommitMessageIs(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must not affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression).
     * @stability stable
     */
    andCommitMessageIsNot(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression).
     * @stability stable
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression).
     * @stability stable
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must affect a Git reference (ie., a branch or a tag) that matches the given pattern.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the event must not affect a Git reference (ie., a branch or a tag) that matches the given pattern.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must target the given base branch.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @stability stable
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must not target the given base branch.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @stability stable
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must target the given Git reference.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must not target the given Git reference.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition: the push that is the source of the event must affect a file that matches the given pattern.
     *
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition: the push that is the source of the event must not affect a file that matches the given pattern.
     *
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression.
     * @stability stable
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addCommitMessageFilter(commitMessage, include) {
        return this.addFilter(WebhookFilterTypes.COMMIT_MESSAGE, commitMessage, include);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter(WebhookFilterTypes.HEAD_REF, refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter(WebhookFilterTypes.ACTOR_ACCOUNT_ID, accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter(WebhookFilterTypes.BASE_REF, refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(WebhookFilterTypes.FILE_PATH, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus === undefined ? true : props.reportBuildStatus;
        this.webhookFilters = props.webhookFilters || [];
        this.webhookTriggersBatchBuild = props.webhookTriggersBatchBuild;
    }
    bind(_scope, project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook === undefined ? (anyFilterGroupsProvided ? true : undefined) : this.webhook;
        if (!webhook && anyFilterGroupsProvided) {
            throw new Error('`webhookFilters` cannot be used when `webhook` is `false`');
        }
        if (!webhook && this.webhookTriggersBatchBuild) {
            throw new Error('`webhookTriggersBatchBuild` cannot be used when `webhook` is `false`');
        }
        const superConfig = super.bind(_scope, project);
        if (this.webhookTriggersBatchBuild) {
            project.enableBatchBuilds();
        }
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                buildType: this.webhookTriggersBatchBuild ? 'BUILD_BATCH' : undefined,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        if (this.hasCommitMessageFilterAndPrEvent()) {
            throw new Error('COMMIT_MESSAGE filters cannot be used with GitHub Enterprise Server pull request events');
        }
        if (this.hasFilePathFilterAndPrEvent()) {
            throw new Error('FILE_PATH filters cannot be used with GitHub Enterprise Server pull request events');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    hasCommitMessageFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.COMMIT_MESSAGE) &&
            this.hasPrEvent(fg._actions)));
    }
    hasFilePathFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.FILE_PATH) &&
            this.hasPrEvent(fg._actions)));
    }
    hasPrEvent(actions) {
        return actions.includes(EventAction.PULL_REQUEST_CREATED ||
            EventAction.PULL_REQUEST_MERGED ||
            EventAction.PULL_REQUEST_REOPENED ||
            EventAction.PULL_REQUEST_UPDATED);
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        // they also don't support file path conditions
        if (this.anyWebhookFilterContainsFilePathConditions()) {
            throw new Error('BitBucket sources do not support file path conditions for webhook filters');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
    anyWebhookFilterContainsFilePathConditions() {
        return this.webhookFilters.findIndex(fg => {
            return fg._filters.findIndex(f => f.type === WebhookFilterTypes.FILE_PATH) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic291cmNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic291cmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHdDQUF3QztBQUl4QyxpREFNd0I7Ozs7OztBQWtEeEIsTUFBc0IsTUFBTTs7OztJQXlCMUIsWUFBc0IsS0FBa0I7Ozs7UUFGeEIsbUJBQWMsR0FBWSxLQUFLLENBQUM7UUFHOUMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO0lBQ3JDLENBQUM7Ozs7SUExQk0sTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUFvQjtRQUNuQyxPQUFPLElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzdCLENBQUM7Ozs7SUFFTSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQTRCO1FBQ25ELE9BQU8sSUFBSSxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNyQyxDQUFDOzs7O0lBRU0sTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUF3QjtRQUMzQyxPQUFPLElBQUksWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2pDLENBQUM7Ozs7SUFFTSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsS0FBa0M7UUFDL0QsT0FBTyxJQUFJLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLENBQUM7Ozs7SUFFTSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQTJCO1FBQ2pELE9BQU8sSUFBSSxlQUFlLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDcEMsQ0FBQzs7Ozs7Ozs7O0lBZU0sSUFBSSxDQUFDLE1BQXFCLEVBQUUsUUFBa0I7UUFDbkQsT0FBTztZQUNMLGNBQWMsRUFBRTtnQkFDZCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDakMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2FBQ2hCO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQTFDRCx3QkEwQ0M7QUE4QkQ7O0dBRUc7QUFDSCxNQUFlLFNBQVUsU0FBUSxNQUFNO0lBS3JDLFlBQXNCLEtBQXFCO1FBQ3pDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUViLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO0lBQy9DLENBQUM7SUFFTSxJQUFJLENBQUMsTUFBcUIsRUFBRSxRQUFrQjtRQUNuRCxNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNqRCxPQUFPO1lBQ0wsYUFBYSxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQy9CLGNBQWMsRUFBRTtnQkFDZCxHQUFHLFdBQVcsQ0FBQyxjQUFjO2dCQUM3QixhQUFhLEVBQUUsSUFBSSxDQUFDLFVBQVU7Z0JBQzlCLG1CQUFtQixFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO29CQUMxQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGVBQWU7aUJBQ3RDLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDZDtTQUNGLENBQUM7SUFDSixDQUFDO0NBQ0Y7Ozs7OztBQUtELElBQVksV0EwQlg7QUExQkQsV0FBWSxXQUFXO0lBSXJCLDRCQUFhLENBQUE7SUFLYiw0REFBNkMsQ0FBQTtJQUs3Qyw0REFBNkMsQ0FBQTtJQUs3QywwREFBMkMsQ0FBQTtJQU0zQyw4REFBK0MsQ0FBQTtBQUNqRCxDQUFDLEVBMUJXLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBMEJ0QjtBQUVELElBQUssa0JBTUo7QUFORCxXQUFLLGtCQUFrQjtJQUNyQiw2Q0FBdUIsQ0FBQTtJQUN2Qix1REFBaUMsQ0FBQTtJQUNqQywyQ0FBcUIsQ0FBQTtJQUNyQiwyREFBcUMsQ0FBQTtJQUNyQywyQ0FBcUIsQ0FBQTtBQUN2QixDQUFDLEVBTkksa0JBQWtCLEtBQWxCLGtCQUFrQixRQU10Qjs7Ozs7Ozs7Ozs7OztBQVdELE1BQWEsV0FBVztJQWF0QixZQUFvQixPQUF5QixFQUFFLE9BQTJDO1FBQ3hGLElBQUksT0FBTyxDQUFDLElBQUksS0FBSyxDQUFDLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO1NBQzFFO1FBQ0QsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFDdkIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQzs7Ozs7OztJQWJNLE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxPQUFzQjtRQUMvQyxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9DLENBQUM7Ozs7Ozs7SUFtQk0sV0FBVyxDQUFDLFVBQWtCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNwRCxDQUFDOzs7Ozs7O0lBUU0sY0FBYyxDQUFDLFVBQWtCO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNyRCxDQUFDOzs7Ozs7O0lBUU0sa0JBQWtCLENBQUMsYUFBcUI7UUFDN0MsT0FBTyxJQUFJLENBQUMsc0JBQXNCLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQzFELENBQUM7Ozs7Ozs7SUFRTSxxQkFBcUIsQ0FBQyxhQUFxQjtRQUNoRCxPQUFPLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDM0QsQ0FBQzs7Ozs7OztJQVFNLFFBQVEsQ0FBQyxPQUFlO1FBQzdCLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQztJQUM5QyxDQUFDOzs7Ozs7O0lBUU0sV0FBVyxDQUFDLE9BQWU7UUFDaEMsT0FBTyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQy9DLENBQUM7Ozs7Ozs7SUFTTSxZQUFZLENBQUMsT0FBZTtRQUNqQyxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDOUMsQ0FBQzs7Ozs7OztJQVNNLGVBQWUsQ0FBQyxPQUFlO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvQyxDQUFDOzs7Ozs7O0lBUU0saUJBQWlCLENBQUMsT0FBZTtRQUN0QyxPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDL0MsQ0FBQzs7Ozs7OztJQVFNLG9CQUFvQixDQUFDLE9BQWU7UUFDekMsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2hELENBQUM7Ozs7Ozs7OztJQVNNLGVBQWUsQ0FBQyxVQUFrQjtRQUN2QyxPQUFPLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDcEQsQ0FBQzs7Ozs7Ozs7O0lBU00sa0JBQWtCLENBQUMsVUFBa0I7UUFDMUMsT0FBTyxJQUFJLENBQUMsbUJBQW1CLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3JELENBQUM7Ozs7Ozs7OztJQVNNLFlBQVksQ0FBQyxPQUFlO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQztJQUM5QyxDQUFDOzs7Ozs7Ozs7SUFTTSxlQUFlLENBQUMsT0FBZTtRQUNwQyxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDL0MsQ0FBQzs7Ozs7Ozs7OztJQVVNLGFBQWEsQ0FBQyxPQUFlO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQztJQUMvQyxDQUFDOzs7Ozs7Ozs7O0lBVU0sZ0JBQWdCLENBQUMsT0FBZTtRQUNyQyxPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVELGdCQUFnQjtJQUNoQixJQUFXLFFBQVE7UUFDakIsT0FBTyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRCxnQkFBZ0I7SUFDaEIsSUFBVyxRQUFRO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixDQUFDO0lBRUQsZ0JBQWdCO0lBQ1QsT0FBTztRQUNaLE1BQU0sV0FBVyxHQUFxQztZQUNwRCxJQUFJLEVBQUUsT0FBTztZQUNiLE9BQU8sRUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7U0FDNUMsQ0FBQztRQUNGLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFTyxzQkFBc0IsQ0FBQyxhQUFxQixFQUFFLE9BQWdCO1FBQ3BFLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxjQUFjLEVBQUUsYUFBYSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ25GLENBQUM7SUFFTyxtQkFBbUIsQ0FBQyxVQUFrQixFQUFFLE9BQWdCO1FBQzlELE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsVUFBVSxFQUFFLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDcEUsQ0FBQztJQUVPLGdCQUFnQixDQUFDLE9BQWUsRUFBRSxPQUFnQjtRQUN4RCxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLE9BQU8sRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFTyxnQkFBZ0IsQ0FBQyxPQUFlLEVBQUUsT0FBZ0I7UUFDeEQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDdkUsQ0FBQztJQUVPLGlCQUFpQixDQUFDLFNBQWlCLEVBQUUsT0FBZ0I7UUFDM0QsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRU8sbUJBQW1CLENBQUMsVUFBa0IsRUFBRSxPQUFnQjtRQUM5RCxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLFVBQVUsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFTyxnQkFBZ0IsQ0FBQyxPQUFlLEVBQUUsT0FBZ0I7UUFDeEQsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRkFBb0YsQ0FBQyxDQUFDO1NBQ3ZHO1FBQ0QsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDdkUsQ0FBQztJQUVPLGlCQUFpQixDQUFDLE9BQWUsRUFBRSxPQUFnQjtRQUN6RCxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRU8sU0FBUyxDQUFDLElBQXdCLEVBQUUsT0FBZSxFQUFFLE9BQWdCO1FBQzNFLE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUN4RCxJQUFJO2dCQUNKLE9BQU87Z0JBQ1AscUJBQXFCLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUk7YUFDbEQsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDRjtBQTVQRCxrQ0E0UEM7QUF1Q0Q7O0dBRUc7QUFDSCxNQUFlLG1CQUFvQixTQUFRLFNBQVM7SUFPbEQsWUFBc0IsS0FBK0I7UUFDbkQsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBUEMsbUJBQWMsR0FBWSxJQUFJLENBQUM7UUFTN0MsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsaUJBQWlCLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNoRyxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDO1FBQ2pELElBQUksQ0FBQyx5QkFBeUIsR0FBRyxLQUFLLENBQUMseUJBQXlCLENBQUM7SUFDbkUsQ0FBQztJQUVNLElBQUksQ0FBQyxNQUFxQixFQUFFLE9BQWlCO1FBQ2xELE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBQy9ELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBRXpHLElBQUksQ0FBQyxPQUFPLElBQUksdUJBQXVCLEVBQUU7WUFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsSUFBSSxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMseUJBQXlCLEVBQUU7WUFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO1NBQ3pGO1FBRUQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFaEQsSUFBSSxJQUFJLENBQUMseUJBQXlCLEVBQUU7WUFDbEMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLENBQUM7U0FDN0I7UUFFRCxPQUFPO1lBQ0wsY0FBYyxFQUFFO2dCQUNkLEdBQUcsV0FBVyxDQUFDLGNBQWM7Z0JBQzdCLGlCQUFpQixFQUFFLElBQUksQ0FBQyxpQkFBaUI7YUFDMUM7WUFDRCxhQUFhLEVBQUUsV0FBVyxDQUFDLGFBQWE7WUFDeEMsYUFBYSxFQUFFLE9BQU8sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2pELE9BQU87Z0JBQ1AsU0FBUyxFQUFFLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNyRSxZQUFZLEVBQUUsdUJBQXVCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDaEc7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBU0Q7O0dBRUc7QUFDSCxNQUFNLGdCQUFpQixTQUFRLFNBQVM7SUFLdEMsWUFBWSxLQUE0QjtRQUN0QyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFMQyxtQkFBYyxHQUFHLElBQUksQ0FBQztRQUN0QixTQUFJLEdBQUcscUNBQXNCLENBQUM7UUFLNUMsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO0lBQy9CLENBQUM7SUFFTSxJQUFJLENBQUMsTUFBcUIsRUFBRSxPQUFpQjtRQUNsRCx5RUFBeUU7UUFDekUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDOUMsT0FBTyxFQUFFLENBQUMsb0JBQW9CLENBQUM7WUFDL0IsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7U0FDckMsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxPQUFPO1lBQ0wsY0FBYyxFQUFFO2dCQUNkLEdBQUcsV0FBVyxDQUFDLGNBQWM7Z0JBQzdCLFFBQVEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLHNCQUFzQjthQUMzQztZQUNELGFBQWEsRUFBRSxXQUFXLENBQUMsYUFBYTtTQUN6QyxDQUFDO0lBQ0osQ0FBQztDQUNGO0FBaUJEOztHQUVHO0FBQ0gsTUFBTSxRQUFTLFNBQVEsTUFBTTtJQU0zQixZQUFZLEtBQW9CO1FBQzlCLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQU5DLFNBQUksR0FBRyw2QkFBYyxDQUFDO1FBT3BDLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUMzQixJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDdkIsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO0lBQy9CLENBQUM7SUFFTSxJQUFJLENBQUMsTUFBcUIsRUFBRSxPQUFpQjtRQUNsRCxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUUvQixNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxPQUFPO1lBQ0wsY0FBYyxFQUFFO2dCQUNkLEdBQUcsV0FBVyxDQUFDLGNBQWM7Z0JBQzdCLFFBQVEsRUFBRSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUU7YUFDbkQ7WUFDRCxhQUFhLEVBQUUsSUFBSSxDQUFDLE9BQU87U0FDNUIsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXFCRDs7R0FFRztBQUNILE1BQU0sWUFBYSxTQUFRLG1CQUFtQjtJQUk1QyxZQUFZLEtBQXdCO1FBQ2xDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUpDLFNBQUksR0FBRyxpQ0FBa0IsQ0FBQztRQUt4QyxJQUFJLENBQUMsYUFBYSxHQUFHLHNCQUFzQixLQUFLLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQztJQUM3RSxDQUFDO0lBRU0sSUFBSSxDQUFDLE1BQXFCLEVBQUUsT0FBaUI7UUFDbEQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDaEQsT0FBTztZQUNMLGNBQWMsRUFBRTtnQkFDZCxHQUFHLFdBQVcsQ0FBQyxjQUFjO2dCQUM3QixRQUFRLEVBQUUsSUFBSSxDQUFDLGFBQWE7YUFDN0I7WUFDRCxhQUFhLEVBQUUsV0FBVyxDQUFDLGFBQWE7WUFDeEMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxhQUFhO1NBQ3pDLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFtQkQ7O0dBRUc7QUFDSCxNQUFNLHNCQUF1QixTQUFRLG1CQUFtQjtJQUt0RCxZQUFZLEtBQWtDO1FBQzVDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUxDLFNBQUksR0FBRyw0Q0FBNkIsQ0FBQztRQU1uRCxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFDekMsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO0lBQy9DLENBQUM7SUFFTSxJQUFJLENBQUMsTUFBcUIsRUFBRSxRQUFrQjtRQUNuRCxJQUFJLElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMseUZBQXlGLENBQUMsQ0FBQztTQUM1RztRQUVELElBQUksSUFBSSxDQUFDLDJCQUEyQixFQUFFLEVBQUU7WUFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRkFBb0YsQ0FBQyxDQUFDO1NBQ3ZHO1FBRUQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDakQsT0FBTztZQUNMLGNBQWMsRUFBRTtnQkFDZCxHQUFHLFdBQVcsQ0FBQyxjQUFjO2dCQUM3QixRQUFRLEVBQUUsSUFBSSxDQUFDLGFBQWE7Z0JBQzVCLFdBQVcsRUFBRSxJQUFJLENBQUMsZUFBZTthQUNsQztZQUNELGFBQWEsRUFBRSxXQUFXLENBQUMsYUFBYTtZQUN4QyxhQUFhLEVBQUUsV0FBVyxDQUFDLGFBQWE7U0FDekMsQ0FBQztJQUNKLENBQUM7SUFFTyxnQ0FBZ0M7UUFDdEMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQ3BDLEVBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksS0FBSyxrQkFBa0IsQ0FBQyxjQUFjLENBQUM7WUFDckUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFDTywyQkFBMkI7UUFDakMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQ3BDLEVBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksS0FBSyxrQkFBa0IsQ0FBQyxTQUFTLENBQUM7WUFDaEUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFDTyxVQUFVLENBQUMsT0FBc0I7UUFDdkMsT0FBTyxPQUFPLENBQUMsUUFBUSxDQUNyQixXQUFXLENBQUMsb0JBQW9CO1lBQ2hDLFdBQVcsQ0FBQyxtQkFBbUI7WUFDL0IsV0FBVyxDQUFDLHFCQUFxQjtZQUNqQyxXQUFXLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0NBQ0Y7QUFxQkQ7O0dBRUc7QUFDSCxNQUFNLGVBQWdCLFNBQVEsbUJBQW1CO0lBSS9DLFlBQVksS0FBMkI7UUFDckMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBSkMsU0FBSSxHQUFHLG9DQUFxQixDQUFDO1FBSzNDLElBQUksQ0FBQyxhQUFhLEdBQUcseUJBQXlCLEtBQUssQ0FBQyxLQUFLLElBQUksS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDO0lBQ2hGLENBQUM7SUFFTSxJQUFJLENBQUMsTUFBcUIsRUFBRSxRQUFrQjtRQUNuRCx5RUFBeUU7UUFDekUsSUFBSSxJQUFJLENBQUMsNkNBQTZDLEVBQUUsRUFBRTtZQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLGlGQUFpRixDQUFDLENBQUM7U0FDcEc7UUFFRCwrQ0FBK0M7UUFDL0MsSUFBSSxJQUFJLENBQUMsMENBQTBDLEVBQUUsRUFBRTtZQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLDJFQUEyRSxDQUFDLENBQUM7U0FDOUY7UUFFRCxNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNqRCxPQUFPO1lBQ0wsY0FBYyxFQUFFO2dCQUNkLEdBQUcsV0FBVyxDQUFDLGNBQWM7Z0JBQzdCLFFBQVEsRUFBRSxJQUFJLENBQUMsYUFBYTthQUM3QjtZQUNELGFBQWEsRUFBRSxXQUFXLENBQUMsYUFBYTtZQUN4QyxhQUFhLEVBQUUsV0FBVyxDQUFDLGFBQWE7U0FDekMsQ0FBQztJQUNKLENBQUM7SUFFTyw2Q0FBNkM7UUFDbkQsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsRUFBRTtZQUN4QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLFdBQVcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3BGLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ1osQ0FBQztJQUVPLDBDQUEwQztRQUNoRCxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxFQUFFO1lBQ3hDLE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3BGLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ1osQ0FBQztDQUNGO0FBRUQsU0FBUyxTQUFTLENBQUksR0FBVztJQUMvQixNQUFNLEdBQUcsR0FBUSxFQUFFLENBQUM7SUFDcEIsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNoQyxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlY29tbWl0IGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlY29tbWl0JztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBDZm5Qcm9qZWN0IH0gZnJvbSAnLi9jb2RlYnVpbGQuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElQcm9qZWN0IH0gZnJvbSAnLi9wcm9qZWN0JztcbmltcG9ydCB7XG4gIEJJVEJVQ0tFVF9TT1VSQ0VfVFlQRSxcbiAgQ09ERUNPTU1JVF9TT1VSQ0VfVFlQRSxcbiAgR0lUSFVCX0VOVEVSUFJJU0VfU09VUkNFX1RZUEUsXG4gIEdJVEhVQl9TT1VSQ0VfVFlQRSxcbiAgUzNfU09VUkNFX1RZUEUsXG59IGZyb20gJy4vc291cmNlLXR5cGVzJztcblxuLy8gdjIgLSBrZWVwIHRoaXMgaW1wb3J0IGFzIGEgc2VwYXJhdGUgc2VjdGlvbiB0byByZWR1Y2UgbWVyZ2UgY29uZmxpY3Qgd2hlbiBmb3J3YXJkIG1lcmdpbmcgd2l0aCB0aGUgdjIgYnJhbmNoLlxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lXG5pbXBvcnQgeyBDb25zdHJ1Y3QgYXMgQ29yZUNvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNvdXJjZUNvbmZpZyB7XG4gIHJlYWRvbmx5IHNvdXJjZVByb3BlcnR5OiBDZm5Qcm9qZWN0LlNvdXJjZVByb3BlcnR5O1xuXG4gIHJlYWRvbmx5IGJ1aWxkVHJpZ2dlcnM/OiBDZm5Qcm9qZWN0LlByb2plY3RUcmlnZ2Vyc1Byb3BlcnR5O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzb3VyY2VWZXJzaW9uPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVNvdXJjZSB7XG4gIHJlYWRvbmx5IGlkZW50aWZpZXI/OiBzdHJpbmc7XG5cbiAgcmVhZG9ubHkgdHlwZTogc3RyaW5nO1xuXG4gIHJlYWRvbmx5IGJhZGdlU3VwcG9ydGVkOiBib29sZWFuO1xuXG4gIGJpbmQoc2NvcGU6IENvcmVDb25zdHJ1Y3QsIHByb2plY3Q6IElQcm9qZWN0KTogU291cmNlQ29uZmlnO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU291cmNlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaWRlbnRpZmllcj86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBTb3VyY2UgaW1wbGVtZW50cyBJU291cmNlIHtcbiAgcHVibGljIHN0YXRpYyBzMyhwcm9wczogUzNTb3VyY2VQcm9wcyk6IElTb3VyY2Uge1xuICAgIHJldHVybiBuZXcgUzNTb3VyY2UocHJvcHMpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBjb2RlQ29tbWl0KHByb3BzOiBDb2RlQ29tbWl0U291cmNlUHJvcHMpOiBJU291cmNlIHtcbiAgICByZXR1cm4gbmV3IENvZGVDb21taXRTb3VyY2UocHJvcHMpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBnaXRIdWIocHJvcHM6IEdpdEh1YlNvdXJjZVByb3BzKTogSVNvdXJjZSB7XG4gICAgcmV0dXJuIG5ldyBHaXRIdWJTb3VyY2UocHJvcHMpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBnaXRIdWJFbnRlcnByaXNlKHByb3BzOiBHaXRIdWJFbnRlcnByaXNlU291cmNlUHJvcHMpOiBJU291cmNlIHtcbiAgICByZXR1cm4gbmV3IEdpdEh1YkVudGVycHJpc2VTb3VyY2UocHJvcHMpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBiaXRCdWNrZXQocHJvcHM6IEJpdEJ1Y2tldFNvdXJjZVByb3BzKTogSVNvdXJjZSB7XG4gICAgcmV0dXJuIG5ldyBCaXRCdWNrZXRTb3VyY2UocHJvcHMpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGlkZW50aWZpZXI/OiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB0eXBlOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBiYWRnZVN1cHBvcnRlZDogYm9vbGVhbiA9IGZhbHNlO1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3Rvcihwcm9wczogU291cmNlUHJvcHMpIHtcbiAgICB0aGlzLmlkZW50aWZpZXIgPSBwcm9wcy5pZGVudGlmaWVyO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogQ29yZUNvbnN0cnVjdCwgX3Byb2plY3Q6IElQcm9qZWN0KTogU291cmNlQ29uZmlnIHtcbiAgICByZXR1cm4ge1xuICAgICAgc291cmNlUHJvcGVydHk6IHtcbiAgICAgICAgc291cmNlSWRlbnRpZmllcjogdGhpcy5pZGVudGlmaWVyLFxuICAgICAgICB0eXBlOiB0aGlzLnR5cGUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBUaGUgY29uc3RydWN0aW9uIHByb3BlcnRpZXMgY29tbW9uIHRvIGFsbCBidWlsZCBzb3VyY2VzIHRoYXQgYXJlIGJhY2tlZCBieSBHaXQuXG4gKi9cbmludGVyZmFjZSBHaXRTb3VyY2VQcm9wcyBleHRlbmRzIFNvdXJjZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2xvbmVEZXB0aD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBicmFuY2hPclJlZj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGZldGNoU3VibW9kdWxlcz86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQSBjb21tb24gc3VwZXJjbGFzcyBvZiBhbGwgYnVpbGQgc291cmNlcyB0aGF0IGFyZSBiYWNrZWQgYnkgR2l0LlxuICovXG5hYnN0cmFjdCBjbGFzcyBHaXRTb3VyY2UgZXh0ZW5kcyBTb3VyY2Uge1xuICBwcml2YXRlIHJlYWRvbmx5IGNsb25lRGVwdGg/OiBudW1iZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgYnJhbmNoT3JSZWY/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgZmV0Y2hTdWJtb2R1bGVzPzogYm9vbGVhbjtcblxuICBwcm90ZWN0ZWQgY29uc3RydWN0b3IocHJvcHM6IEdpdFNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuXG4gICAgdGhpcy5jbG9uZURlcHRoID0gcHJvcHMuY2xvbmVEZXB0aDtcbiAgICB0aGlzLmJyYW5jaE9yUmVmID0gcHJvcHMuYnJhbmNoT3JSZWY7XG4gICAgdGhpcy5mZXRjaFN1Ym1vZHVsZXMgPSBwcm9wcy5mZXRjaFN1Ym1vZHVsZXM7XG4gIH1cblxuICBwdWJsaWMgYmluZChfc2NvcGU6IENvcmVDb25zdHJ1Y3QsIF9wcm9qZWN0OiBJUHJvamVjdCk6IFNvdXJjZUNvbmZpZyB7XG4gICAgY29uc3Qgc3VwZXJDb25maWcgPSBzdXBlci5iaW5kKF9zY29wZSwgX3Byb2plY3QpO1xuICAgIHJldHVybiB7XG4gICAgICBzb3VyY2VWZXJzaW9uOiB0aGlzLmJyYW5jaE9yUmVmLFxuICAgICAgc291cmNlUHJvcGVydHk6IHtcbiAgICAgICAgLi4uc3VwZXJDb25maWcuc291cmNlUHJvcGVydHksXG4gICAgICAgIGdpdENsb25lRGVwdGg6IHRoaXMuY2xvbmVEZXB0aCxcbiAgICAgICAgZ2l0U3VibW9kdWxlc0NvbmZpZzogdGhpcy5mZXRjaFN1Ym1vZHVsZXMgPyB7XG4gICAgICAgICAgZmV0Y2hTdWJtb2R1bGVzOiB0aGlzLmZldGNoU3VibW9kdWxlcyxcbiAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBFdmVudEFjdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUFVTSCA9ICdQVVNIJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUFVMTF9SRVFVRVNUX0NSRUFURUQgPSAnUFVMTF9SRVFVRVNUX0NSRUFURUQnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBQVUxMX1JFUVVFU1RfVVBEQVRFRCA9ICdQVUxMX1JFUVVFU1RfVVBEQVRFRCcsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUFVMTF9SRVFVRVNUX01FUkdFRCA9ICdQVUxMX1JFUVVFU1RfTUVSR0VEJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUFVMTF9SRVFVRVNUX1JFT1BFTkVEID0gJ1BVTExfUkVRVUVTVF9SRU9QRU5FRCcsXG59XG5cbmVudW0gV2ViaG9va0ZpbHRlclR5cGVzIHtcbiAgRklMRV9QQVRIID0gJ0ZJTEVfUEFUSCcsXG4gIENPTU1JVF9NRVNTQUdFID0gJ0NPTU1JVF9NRVNTQUdFJyxcbiAgSEVBRF9SRUYgPSAnSEVBRF9SRUYnLFxuICBBQ1RPUl9BQ0NPVU5UX0lEID0gJ0FDVE9SX0FDQ09VTlRfSUQnLFxuICBCQVNFX1JFRiA9ICdCQVNFX1JFRicsXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRmlsdGVyR3JvdXAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGluRXZlbnRPZiguLi5hY3Rpb25zOiBFdmVudEFjdGlvbltdKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiBuZXcgRmlsdGVyR3JvdXAobmV3IFNldChhY3Rpb25zKSwgW10pO1xuICB9XG5cbiAgcHJpdmF0ZSByZWFkb25seSBhY3Rpb25zOiBTZXQ8RXZlbnRBY3Rpb24+O1xuICBwcml2YXRlIHJlYWRvbmx5IGZpbHRlcnM6IENmblByb2plY3QuV2ViaG9va0ZpbHRlclByb3BlcnR5W107XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihhY3Rpb25zOiBTZXQ8RXZlbnRBY3Rpb24+LCBmaWx0ZXJzOiBDZm5Qcm9qZWN0LldlYmhvb2tGaWx0ZXJQcm9wZXJ0eVtdKSB7XG4gICAgaWYgKGFjdGlvbnMuc2l6ZSA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBIGZpbHRlciBncm91cCBtdXN0IGNvbnRhaW4gYXQgbGVhc3Qgb25lIGV2ZW50IGFjdGlvbicpO1xuICAgIH1cbiAgICB0aGlzLmFjdGlvbnMgPSBhY3Rpb25zO1xuICAgIHRoaXMuZmlsdGVycyA9IGZpbHRlcnM7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRCcmFuY2hJcyhicmFuY2hOYW1lOiBzdHJpbmcpOiBGaWx0ZXJHcm91cCB7XG4gICAgcmV0dXJuIHRoaXMuYWRkSGVhZEJyYW5jaEZpbHRlcihicmFuY2hOYW1lLCB0cnVlKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRCcmFuY2hJc05vdChicmFuY2hOYW1lOiBzdHJpbmcpOiBGaWx0ZXJHcm91cCB7XG4gICAgcmV0dXJuIHRoaXMuYWRkSGVhZEJyYW5jaEZpbHRlcihicmFuY2hOYW1lLCBmYWxzZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYW5kQ29tbWl0TWVzc2FnZUlzKGNvbW1pdE1lc3NhZ2U6IHN0cmluZyk6IEZpbHRlckdyb3VwIHtcbiAgICByZXR1cm4gdGhpcy5hZGRDb21taXRNZXNzYWdlRmlsdGVyKGNvbW1pdE1lc3NhZ2UsIHRydWUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRDb21taXRNZXNzYWdlSXNOb3QoY29tbWl0TWVzc2FnZTogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZENvbW1pdE1lc3NhZ2VGaWx0ZXIoY29tbWl0TWVzc2FnZSwgZmFsc2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYW5kVGFnSXModGFnTmFtZTogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEhlYWRUYWdGaWx0ZXIodGFnTmFtZSwgdHJ1ZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYW5kVGFnSXNOb3QodGFnTmFtZTogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEhlYWRUYWdGaWx0ZXIodGFnTmFtZSwgZmFsc2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFuZEhlYWRSZWZJcyhwYXR0ZXJuOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5hZGRIZWFkUmVmRmlsdGVyKHBhdHRlcm4sIHRydWUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRIZWFkUmVmSXNOb3QocGF0dGVybjogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkSGVhZFJlZkZpbHRlcihwYXR0ZXJuLCBmYWxzZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYW5kQWN0b3JBY2NvdW50SXMocGF0dGVybjogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEFjdG9yQWNjb3VudElkKHBhdHRlcm4sIHRydWUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRBY3RvckFjY291bnRJc05vdChwYXR0ZXJuOiBzdHJpbmcpOiBGaWx0ZXJHcm91cCB7XG4gICAgcmV0dXJuIHRoaXMuYWRkQWN0b3JBY2NvdW50SWQocGF0dGVybiwgZmFsc2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRCYXNlQnJhbmNoSXMoYnJhbmNoTmFtZTogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEJhc2VCcmFuY2hGaWx0ZXIoYnJhbmNoTmFtZSwgdHJ1ZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRCYXNlQnJhbmNoSXNOb3QoYnJhbmNoTmFtZTogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEJhc2VCcmFuY2hGaWx0ZXIoYnJhbmNoTmFtZSwgZmFsc2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRCYXNlUmVmSXMocGF0dGVybjogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEJhc2VSZWZGaWx0ZXIocGF0dGVybiwgdHJ1ZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRCYXNlUmVmSXNOb3QocGF0dGVybjogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEJhc2VSZWZGaWx0ZXIocGF0dGVybiwgZmFsc2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFuZEZpbGVQYXRoSXMocGF0dGVybjogc3RyaW5nKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEZpbGVQYXRoRmlsdGVyKHBhdHRlcm4sIHRydWUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbmRGaWxlUGF0aElzTm90KHBhdHRlcm46IHN0cmluZyk6IEZpbHRlckdyb3VwIHtcbiAgICByZXR1cm4gdGhpcy5hZGRGaWxlUGF0aEZpbHRlcihwYXR0ZXJuLCBmYWxzZSk7XG4gIH1cblxuICAvKiogQGludGVybmFsICovXG4gIHB1YmxpYyBnZXQgX2FjdGlvbnMoKTogRXZlbnRBY3Rpb25bXSB7XG4gICAgcmV0dXJuIHNldDJBcnJheSh0aGlzLmFjdGlvbnMpO1xuICB9XG5cbiAgLyoqIEBpbnRlcm5hbCAqL1xuICBwdWJsaWMgZ2V0IF9maWx0ZXJzKCk6IENmblByb2plY3QuV2ViaG9va0ZpbHRlclByb3BlcnR5W10ge1xuICAgIHJldHVybiB0aGlzLmZpbHRlcnMuc2xpY2UoKTtcbiAgfVxuXG4gIC8qKiBAaW50ZXJuYWwgKi9cbiAgcHVibGljIF90b0pzb24oKTogQ2ZuUHJvamVjdC5XZWJob29rRmlsdGVyUHJvcGVydHlbXSB7XG4gICAgY29uc3QgZXZlbnRGaWx0ZXI6IENmblByb2plY3QuV2ViaG9va0ZpbHRlclByb3BlcnR5ID0ge1xuICAgICAgdHlwZTogJ0VWRU5UJyxcbiAgICAgIHBhdHRlcm46IHNldDJBcnJheSh0aGlzLmFjdGlvbnMpLmpvaW4oJywgJyksXG4gICAgfTtcbiAgICByZXR1cm4gW2V2ZW50RmlsdGVyXS5jb25jYXQodGhpcy5maWx0ZXJzKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkQ29tbWl0TWVzc2FnZUZpbHRlcihjb21taXRNZXNzYWdlOiBzdHJpbmcsIGluY2x1ZGU6IGJvb2xlYW4pOiBGaWx0ZXJHcm91cCB7XG4gICAgcmV0dXJuIHRoaXMuYWRkRmlsdGVyKFdlYmhvb2tGaWx0ZXJUeXBlcy5DT01NSVRfTUVTU0FHRSwgY29tbWl0TWVzc2FnZSwgaW5jbHVkZSk7XG4gIH1cblxuICBwcml2YXRlIGFkZEhlYWRCcmFuY2hGaWx0ZXIoYnJhbmNoTmFtZTogc3RyaW5nLCBpbmNsdWRlOiBib29sZWFuKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEhlYWRSZWZGaWx0ZXIoYHJlZnMvaGVhZHMvJHticmFuY2hOYW1lfWAsIGluY2x1ZGUpO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRIZWFkVGFnRmlsdGVyKHRhZ05hbWU6IHN0cmluZywgaW5jbHVkZTogYm9vbGVhbik6IEZpbHRlckdyb3VwIHtcbiAgICByZXR1cm4gdGhpcy5hZGRIZWFkUmVmRmlsdGVyKGByZWZzL3RhZ3MvJHt0YWdOYW1lfWAsIGluY2x1ZGUpO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRIZWFkUmVmRmlsdGVyKHJlZk5hbWU6IHN0cmluZywgaW5jbHVkZTogYm9vbGVhbikge1xuICAgIHJldHVybiB0aGlzLmFkZEZpbHRlcihXZWJob29rRmlsdGVyVHlwZXMuSEVBRF9SRUYsIHJlZk5hbWUsIGluY2x1ZGUpO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRBY3RvckFjY291bnRJZChhY2NvdW50SWQ6IHN0cmluZywgaW5jbHVkZTogYm9vbGVhbikge1xuICAgIHJldHVybiB0aGlzLmFkZEZpbHRlcihXZWJob29rRmlsdGVyVHlwZXMuQUNUT1JfQUNDT1VOVF9JRCwgYWNjb3VudElkLCBpbmNsdWRlKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkQmFzZUJyYW5jaEZpbHRlcihicmFuY2hOYW1lOiBzdHJpbmcsIGluY2x1ZGU6IGJvb2xlYW4pOiBGaWx0ZXJHcm91cCB7XG4gICAgcmV0dXJuIHRoaXMuYWRkQmFzZVJlZkZpbHRlcihgcmVmcy9oZWFkcy8ke2JyYW5jaE5hbWV9YCwgaW5jbHVkZSk7XG4gIH1cblxuICBwcml2YXRlIGFkZEJhc2VSZWZGaWx0ZXIocmVmTmFtZTogc3RyaW5nLCBpbmNsdWRlOiBib29sZWFuKSB7XG4gICAgaWYgKHRoaXMuYWN0aW9ucy5oYXMoRXZlbnRBY3Rpb24uUFVTSCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQSBiYXNlIHJlZmVyZW5jZSBjb25kaXRpb24gY2Fubm90IGJlIGFkZGVkIGlmIGEgR3JvdXAgY29udGFpbnMgYSBQVVNIIGV2ZW50IGFjdGlvbicpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5hZGRGaWx0ZXIoV2ViaG9va0ZpbHRlclR5cGVzLkJBU0VfUkVGLCByZWZOYW1lLCBpbmNsdWRlKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkRmlsZVBhdGhGaWx0ZXIocGF0dGVybjogc3RyaW5nLCBpbmNsdWRlOiBib29sZWFuKTogRmlsdGVyR3JvdXAge1xuICAgIHJldHVybiB0aGlzLmFkZEZpbHRlcihXZWJob29rRmlsdGVyVHlwZXMuRklMRV9QQVRILCBwYXR0ZXJuLCBpbmNsdWRlKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkRmlsdGVyKHR5cGU6IFdlYmhvb2tGaWx0ZXJUeXBlcywgcGF0dGVybjogc3RyaW5nLCBpbmNsdWRlOiBib29sZWFuKSB7XG4gICAgcmV0dXJuIG5ldyBGaWx0ZXJHcm91cCh0aGlzLmFjdGlvbnMsIHRoaXMuZmlsdGVycy5jb25jYXQoW3tcbiAgICAgIHR5cGUsXG4gICAgICBwYXR0ZXJuLFxuICAgICAgZXhjbHVkZU1hdGNoZWRQYXR0ZXJuOiBpbmNsdWRlID8gdW5kZWZpbmVkIDogdHJ1ZSxcbiAgICB9XSkpO1xuICB9XG59XG5cbi8qKlxuICogVGhlIGNvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGNvbW1vbiB0byBhbGwgdGhpcmQtcGFydHkgYnVpbGQgc291cmNlcyB0aGF0IGFyZSBiYWNrZWQgYnkgR2l0LlxuICovXG5pbnRlcmZhY2UgVGhpcmRQYXJ0eUdpdFNvdXJjZVByb3BzIGV4dGVuZHMgR2l0U291cmNlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVwb3J0QnVpbGRTdGF0dXM/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdlYmhvb2s/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgd2ViaG9va1RyaWdnZXJzQmF0Y2hCdWlsZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdlYmhvb2tGaWx0ZXJzPzogRmlsdGVyR3JvdXBbXTtcbn1cblxuLyoqXG4gKiBBIGNvbW1vbiBzdXBlcmNsYXNzIG9mIGFsbCB0aGlyZC1wYXJ0eSBidWlsZCBzb3VyY2VzIHRoYXQgYXJlIGJhY2tlZCBieSBHaXQuXG4gKi9cbmFic3RyYWN0IGNsYXNzIFRoaXJkUGFydHlHaXRTb3VyY2UgZXh0ZW5kcyBHaXRTb3VyY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgYmFkZ2VTdXBwb3J0ZWQ6IGJvb2xlYW4gPSB0cnVlO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgd2ViaG9va0ZpbHRlcnM6IEZpbHRlckdyb3VwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVwb3J0QnVpbGRTdGF0dXM6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgd2ViaG9vaz86IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgd2ViaG9va1RyaWdnZXJzQmF0Y2hCdWlsZD86IGJvb2xlYW47XG5cbiAgcHJvdGVjdGVkIGNvbnN0cnVjdG9yKHByb3BzOiBUaGlyZFBhcnR5R2l0U291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG5cbiAgICB0aGlzLndlYmhvb2sgPSBwcm9wcy53ZWJob29rO1xuICAgIHRoaXMucmVwb3J0QnVpbGRTdGF0dXMgPSBwcm9wcy5yZXBvcnRCdWlsZFN0YXR1cyA9PT0gdW5kZWZpbmVkID8gdHJ1ZSA6IHByb3BzLnJlcG9ydEJ1aWxkU3RhdHVzO1xuICAgIHRoaXMud2ViaG9va0ZpbHRlcnMgPSBwcm9wcy53ZWJob29rRmlsdGVycyB8fCBbXTtcbiAgICB0aGlzLndlYmhvb2tUcmlnZ2Vyc0JhdGNoQnVpbGQgPSBwcm9wcy53ZWJob29rVHJpZ2dlcnNCYXRjaEJ1aWxkO1xuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBDb3JlQ29uc3RydWN0LCBwcm9qZWN0OiBJUHJvamVjdCk6IFNvdXJjZUNvbmZpZyB7XG4gICAgY29uc3QgYW55RmlsdGVyR3JvdXBzUHJvdmlkZWQgPSB0aGlzLndlYmhvb2tGaWx0ZXJzLmxlbmd0aCA+IDA7XG4gICAgY29uc3Qgd2ViaG9vayA9IHRoaXMud2ViaG9vayA9PT0gdW5kZWZpbmVkID8gKGFueUZpbHRlckdyb3Vwc1Byb3ZpZGVkID8gdHJ1ZSA6IHVuZGVmaW5lZCkgOiB0aGlzLndlYmhvb2s7XG5cbiAgICBpZiAoIXdlYmhvb2sgJiYgYW55RmlsdGVyR3JvdXBzUHJvdmlkZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYHdlYmhvb2tGaWx0ZXJzYCBjYW5ub3QgYmUgdXNlZCB3aGVuIGB3ZWJob29rYCBpcyBgZmFsc2VgJyk7XG4gICAgfVxuXG4gICAgaWYgKCF3ZWJob29rICYmIHRoaXMud2ViaG9va1RyaWdnZXJzQmF0Y2hCdWlsZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdgd2ViaG9va1RyaWdnZXJzQmF0Y2hCdWlsZGAgY2Fubm90IGJlIHVzZWQgd2hlbiBgd2ViaG9va2AgaXMgYGZhbHNlYCcpO1xuICAgIH1cblxuICAgIGNvbnN0IHN1cGVyQ29uZmlnID0gc3VwZXIuYmluZChfc2NvcGUsIHByb2plY3QpO1xuXG4gICAgaWYgKHRoaXMud2ViaG9va1RyaWdnZXJzQmF0Y2hCdWlsZCkge1xuICAgICAgcHJvamVjdC5lbmFibGVCYXRjaEJ1aWxkcygpO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBzb3VyY2VQcm9wZXJ0eToge1xuICAgICAgICAuLi5zdXBlckNvbmZpZy5zb3VyY2VQcm9wZXJ0eSxcbiAgICAgICAgcmVwb3J0QnVpbGRTdGF0dXM6IHRoaXMucmVwb3J0QnVpbGRTdGF0dXMsXG4gICAgICB9LFxuICAgICAgc291cmNlVmVyc2lvbjogc3VwZXJDb25maWcuc291cmNlVmVyc2lvbixcbiAgICAgIGJ1aWxkVHJpZ2dlcnM6IHdlYmhvb2sgPT09IHVuZGVmaW5lZCA/IHVuZGVmaW5lZCA6IHtcbiAgICAgICAgd2ViaG9vayxcbiAgICAgICAgYnVpbGRUeXBlOiB0aGlzLndlYmhvb2tUcmlnZ2Vyc0JhdGNoQnVpbGQgPyAnQlVJTERfQkFUQ0gnIDogdW5kZWZpbmVkLFxuICAgICAgICBmaWx0ZXJHcm91cHM6IGFueUZpbHRlckdyb3Vwc1Byb3ZpZGVkID8gdGhpcy53ZWJob29rRmlsdGVycy5tYXAoZmcgPT4gZmcuX3RvSnNvbigpKSA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIENvZGVDb21taXRTb3VyY2VQcm9wcyBleHRlbmRzIEdpdFNvdXJjZVByb3BzIHtcbiAgcmVhZG9ubHkgcmVwb3NpdG9yeTogY29kZWNvbW1pdC5JUmVwb3NpdG9yeTtcbn1cblxuLyoqXG4gKiBDb2RlQ29tbWl0IFNvdXJjZSBkZWZpbml0aW9uIGZvciBhIENvZGVCdWlsZCBwcm9qZWN0LlxuICovXG5jbGFzcyBDb2RlQ29tbWl0U291cmNlIGV4dGVuZHMgR2l0U291cmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IGJhZGdlU3VwcG9ydGVkID0gdHJ1ZTtcbiAgcHVibGljIHJlYWRvbmx5IHR5cGUgPSBDT0RFQ09NTUlUX1NPVVJDRV9UWVBFO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlcG86IGNvZGVjb21taXQuSVJlcG9zaXRvcnk7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IENvZGVDb21taXRTb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgICB0aGlzLnJlcG8gPSBwcm9wcy5yZXBvc2l0b3J5O1xuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBDb3JlQ29uc3RydWN0LCBwcm9qZWN0OiBJUHJvamVjdCk6IFNvdXJjZUNvbmZpZyB7XG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL3NldHRpbmctdXAuaHRtbFxuICAgIHByb2plY3QuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnY29kZWNvbW1pdDpHaXRQdWxsJ10sXG4gICAgICByZXNvdXJjZXM6IFt0aGlzLnJlcG8ucmVwb3NpdG9yeUFybl0sXG4gICAgfSkpO1xuXG4gICAgY29uc3Qgc3VwZXJDb25maWcgPSBzdXBlci5iaW5kKF9zY29wZSwgcHJvamVjdCk7XG4gICAgcmV0dXJuIHtcbiAgICAgIHNvdXJjZVByb3BlcnR5OiB7XG4gICAgICAgIC4uLnN1cGVyQ29uZmlnLnNvdXJjZVByb3BlcnR5LFxuICAgICAgICBsb2NhdGlvbjogdGhpcy5yZXBvLnJlcG9zaXRvcnlDbG9uZVVybEh0dHAsXG4gICAgICB9LFxuICAgICAgc291cmNlVmVyc2lvbjogc3VwZXJDb25maWcuc291cmNlVmVyc2lvbixcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFMzU291cmNlUHJvcHMgZXh0ZW5kcyBTb3VyY2VQcm9wcyB7XG4gIHJlYWRvbmx5IGJ1Y2tldDogczMuSUJ1Y2tldDtcbiAgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZlcnNpb24/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUzMgYnVja2V0IGRlZmluaXRpb24gZm9yIGEgQ29kZUJ1aWxkIHByb2plY3QuXG4gKi9cbmNsYXNzIFMzU291cmNlIGV4dGVuZHMgU291cmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHR5cGUgPSBTM19TT1VSQ0VfVFlQRTtcbiAgcHJpdmF0ZSByZWFkb25seSBidWNrZXQ6IHMzLklCdWNrZXQ7XG4gIHByaXZhdGUgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHZlcnNpb24/OiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFMzU291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gICAgdGhpcy5idWNrZXQgPSBwcm9wcy5idWNrZXQ7XG4gICAgdGhpcy5wYXRoID0gcHJvcHMucGF0aDtcbiAgICB0aGlzLnZlcnNpb24gPSBwcm9wcy52ZXJzaW9uO1xuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBDb3JlQ29uc3RydWN0LCBwcm9qZWN0OiBJUHJvamVjdCk6IFNvdXJjZUNvbmZpZyB7XG4gICAgdGhpcy5idWNrZXQuZ3JhbnRSZWFkKHByb2plY3QpO1xuXG4gICAgY29uc3Qgc3VwZXJDb25maWcgPSBzdXBlci5iaW5kKF9zY29wZSwgcHJvamVjdCk7XG4gICAgcmV0dXJuIHtcbiAgICAgIHNvdXJjZVByb3BlcnR5OiB7XG4gICAgICAgIC4uLnN1cGVyQ29uZmlnLnNvdXJjZVByb3BlcnR5LFxuICAgICAgICBsb2NhdGlvbjogYCR7dGhpcy5idWNrZXQuYnVja2V0TmFtZX0vJHt0aGlzLnBhdGh9YCxcbiAgICAgIH0sXG4gICAgICBzb3VyY2VWZXJzaW9uOiB0aGlzLnZlcnNpb24sXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBHaXRIdWJTb3VyY2VQcm9wcyBleHRlbmRzIFRoaXJkUGFydHlHaXRTb3VyY2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvd25lcjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVwbzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEdpdEh1YiBTb3VyY2UgZGVmaW5pdGlvbiBmb3IgYSBDb2RlQnVpbGQgcHJvamVjdC5cbiAqL1xuY2xhc3MgR2l0SHViU291cmNlIGV4dGVuZHMgVGhpcmRQYXJ0eUdpdFNvdXJjZSB7XG4gIHB1YmxpYyByZWFkb25seSB0eXBlID0gR0lUSFVCX1NPVVJDRV9UWVBFO1xuICBwcml2YXRlIHJlYWRvbmx5IGh0dHBzQ2xvbmVVcmw6IHN0cmluZztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogR2l0SHViU291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gICAgdGhpcy5odHRwc0Nsb25lVXJsID0gYGh0dHBzOi8vZ2l0aHViLmNvbS8ke3Byb3BzLm93bmVyfS8ke3Byb3BzLnJlcG99LmdpdGA7XG4gIH1cblxuICBwdWJsaWMgYmluZChfc2NvcGU6IENvcmVDb25zdHJ1Y3QsIHByb2plY3Q6IElQcm9qZWN0KTogU291cmNlQ29uZmlnIHtcbiAgICBjb25zdCBzdXBlckNvbmZpZyA9IHN1cGVyLmJpbmQoX3Njb3BlLCBwcm9qZWN0KTtcbiAgICByZXR1cm4ge1xuICAgICAgc291cmNlUHJvcGVydHk6IHtcbiAgICAgICAgLi4uc3VwZXJDb25maWcuc291cmNlUHJvcGVydHksXG4gICAgICAgIGxvY2F0aW9uOiB0aGlzLmh0dHBzQ2xvbmVVcmwsXG4gICAgICB9LFxuICAgICAgc291cmNlVmVyc2lvbjogc3VwZXJDb25maWcuc291cmNlVmVyc2lvbixcbiAgICAgIGJ1aWxkVHJpZ2dlcnM6IHN1cGVyQ29uZmlnLmJ1aWxkVHJpZ2dlcnMsXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEdpdEh1YkVudGVycHJpc2VTb3VyY2VQcm9wcyBleHRlbmRzIFRoaXJkUGFydHlHaXRTb3VyY2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGh0dHBzQ2xvbmVVcmw6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGlnbm9yZVNzbEVycm9ycz86IGJvb2xlYW47XG59XG5cbi8qKlxuICogR2l0SHViIEVudGVycHJpc2UgU291cmNlIGRlZmluaXRpb24gZm9yIGEgQ29kZUJ1aWxkIHByb2plY3QuXG4gKi9cbmNsYXNzIEdpdEh1YkVudGVycHJpc2VTb3VyY2UgZXh0ZW5kcyBUaGlyZFBhcnR5R2l0U291cmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHR5cGUgPSBHSVRIVUJfRU5URVJQUklTRV9TT1VSQ0VfVFlQRTtcbiAgcHJpdmF0ZSByZWFkb25seSBodHRwc0Nsb25lVXJsOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgaWdub3JlU3NsRXJyb3JzPzogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogR2l0SHViRW50ZXJwcmlzZVNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICAgIHRoaXMuaHR0cHNDbG9uZVVybCA9IHByb3BzLmh0dHBzQ2xvbmVVcmw7XG4gICAgdGhpcy5pZ25vcmVTc2xFcnJvcnMgPSBwcm9wcy5pZ25vcmVTc2xFcnJvcnM7XG4gIH1cblxuICBwdWJsaWMgYmluZChfc2NvcGU6IENvcmVDb25zdHJ1Y3QsIF9wcm9qZWN0OiBJUHJvamVjdCk6IFNvdXJjZUNvbmZpZyB7XG4gICAgaWYgKHRoaXMuaGFzQ29tbWl0TWVzc2FnZUZpbHRlckFuZFByRXZlbnQoKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDT01NSVRfTUVTU0FHRSBmaWx0ZXJzIGNhbm5vdCBiZSB1c2VkIHdpdGggR2l0SHViIEVudGVycHJpc2UgU2VydmVyIHB1bGwgcmVxdWVzdCBldmVudHMnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5oYXNGaWxlUGF0aEZpbHRlckFuZFByRXZlbnQoKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGSUxFX1BBVEggZmlsdGVycyBjYW5ub3QgYmUgdXNlZCB3aXRoIEdpdEh1YiBFbnRlcnByaXNlIFNlcnZlciBwdWxsIHJlcXVlc3QgZXZlbnRzJyk7XG4gICAgfVxuXG4gICAgY29uc3Qgc3VwZXJDb25maWcgPSBzdXBlci5iaW5kKF9zY29wZSwgX3Byb2plY3QpO1xuICAgIHJldHVybiB7XG4gICAgICBzb3VyY2VQcm9wZXJ0eToge1xuICAgICAgICAuLi5zdXBlckNvbmZpZy5zb3VyY2VQcm9wZXJ0eSxcbiAgICAgICAgbG9jYXRpb246IHRoaXMuaHR0cHNDbG9uZVVybCxcbiAgICAgICAgaW5zZWN1cmVTc2w6IHRoaXMuaWdub3JlU3NsRXJyb3JzLFxuICAgICAgfSxcbiAgICAgIHNvdXJjZVZlcnNpb246IHN1cGVyQ29uZmlnLnNvdXJjZVZlcnNpb24sXG4gICAgICBidWlsZFRyaWdnZXJzOiBzdXBlckNvbmZpZy5idWlsZFRyaWdnZXJzLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGhhc0NvbW1pdE1lc3NhZ2VGaWx0ZXJBbmRQckV2ZW50KCkge1xuICAgIHJldHVybiB0aGlzLndlYmhvb2tGaWx0ZXJzLnNvbWUoZmcgPT4gKFxuICAgICAgZmcuX2ZpbHRlcnMuc29tZShmcCA9PiBmcC50eXBlID09PSBXZWJob29rRmlsdGVyVHlwZXMuQ09NTUlUX01FU1NBR0UpICYmXG4gICAgICB0aGlzLmhhc1ByRXZlbnQoZmcuX2FjdGlvbnMpKSk7XG4gIH1cbiAgcHJpdmF0ZSBoYXNGaWxlUGF0aEZpbHRlckFuZFByRXZlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMud2ViaG9va0ZpbHRlcnMuc29tZShmZyA9PiAoXG4gICAgICBmZy5fZmlsdGVycy5zb21lKGZwID0+IGZwLnR5cGUgPT09IFdlYmhvb2tGaWx0ZXJUeXBlcy5GSUxFX1BBVEgpICYmXG4gICAgICB0aGlzLmhhc1ByRXZlbnQoZmcuX2FjdGlvbnMpKSk7XG4gIH1cbiAgcHJpdmF0ZSBoYXNQckV2ZW50KGFjdGlvbnM6IEV2ZW50QWN0aW9uW10pIHtcbiAgICByZXR1cm4gYWN0aW9ucy5pbmNsdWRlcyhcbiAgICAgIEV2ZW50QWN0aW9uLlBVTExfUkVRVUVTVF9DUkVBVEVEIHx8XG4gICAgICBFdmVudEFjdGlvbi5QVUxMX1JFUVVFU1RfTUVSR0VEIHx8XG4gICAgICBFdmVudEFjdGlvbi5QVUxMX1JFUVVFU1RfUkVPUEVORUQgfHxcbiAgICAgIEV2ZW50QWN0aW9uLlBVTExfUkVRVUVTVF9VUERBVEVEKTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQml0QnVja2V0U291cmNlUHJvcHMgZXh0ZW5kcyBUaGlyZFBhcnR5R2l0U291cmNlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3duZXI6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlcG86IHN0cmluZztcbn1cblxuLyoqXG4gKiBCaXRCdWNrZXQgU291cmNlIGRlZmluaXRpb24gZm9yIGEgQ29kZUJ1aWxkIHByb2plY3QuXG4gKi9cbmNsYXNzIEJpdEJ1Y2tldFNvdXJjZSBleHRlbmRzIFRoaXJkUGFydHlHaXRTb3VyY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgdHlwZSA9IEJJVEJVQ0tFVF9TT1VSQ0VfVFlQRTtcbiAgcHJpdmF0ZSByZWFkb25seSBodHRwc0Nsb25lVXJsOiBhbnk7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IEJpdEJ1Y2tldFNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICAgIHRoaXMuaHR0cHNDbG9uZVVybCA9IGBodHRwczovL2JpdGJ1Y2tldC5vcmcvJHtwcm9wcy5vd25lcn0vJHtwcm9wcy5yZXBvfS5naXRgO1xuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBDb3JlQ29uc3RydWN0LCBfcHJvamVjdDogSVByb2plY3QpOiBTb3VyY2VDb25maWcge1xuICAgIC8vIEJpdEJ1Y2tldCBzb3VyY2VzIGRvbid0IHN1cHBvcnQgdGhlIFBVTExfUkVRVUVTVF9SRU9QRU5FRCBldmVudCBhY3Rpb25cbiAgICBpZiAodGhpcy5hbnlXZWJob29rRmlsdGVyQ29udGFpbnNQclJlb3BlbmVkRXZlbnRBY3Rpb24oKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdCaXRCdWNrZXQgc291cmNlcyBkbyBub3Qgc3VwcG9ydCB0aGUgUFVMTF9SRVFVRVNUX1JFT1BFTkVEIHdlYmhvb2sgZXZlbnQgYWN0aW9uJyk7XG4gICAgfVxuXG4gICAgLy8gdGhleSBhbHNvIGRvbid0IHN1cHBvcnQgZmlsZSBwYXRoIGNvbmRpdGlvbnNcbiAgICBpZiAodGhpcy5hbnlXZWJob29rRmlsdGVyQ29udGFpbnNGaWxlUGF0aENvbmRpdGlvbnMoKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdCaXRCdWNrZXQgc291cmNlcyBkbyBub3Qgc3VwcG9ydCBmaWxlIHBhdGggY29uZGl0aW9ucyBmb3Igd2ViaG9vayBmaWx0ZXJzJyk7XG4gICAgfVxuXG4gICAgY29uc3Qgc3VwZXJDb25maWcgPSBzdXBlci5iaW5kKF9zY29wZSwgX3Byb2plY3QpO1xuICAgIHJldHVybiB7XG4gICAgICBzb3VyY2VQcm9wZXJ0eToge1xuICAgICAgICAuLi5zdXBlckNvbmZpZy5zb3VyY2VQcm9wZXJ0eSxcbiAgICAgICAgbG9jYXRpb246IHRoaXMuaHR0cHNDbG9uZVVybCxcbiAgICAgIH0sXG4gICAgICBzb3VyY2VWZXJzaW9uOiBzdXBlckNvbmZpZy5zb3VyY2VWZXJzaW9uLFxuICAgICAgYnVpbGRUcmlnZ2Vyczogc3VwZXJDb25maWcuYnVpbGRUcmlnZ2VycyxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBhbnlXZWJob29rRmlsdGVyQ29udGFpbnNQclJlb3BlbmVkRXZlbnRBY3Rpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMud2ViaG9va0ZpbHRlcnMuZmluZEluZGV4KGZnID0+IHtcbiAgICAgIHJldHVybiBmZy5fYWN0aW9ucy5maW5kSW5kZXgoYSA9PiBhID09PSBFdmVudEFjdGlvbi5QVUxMX1JFUVVFU1RfUkVPUEVORUQpICE9PSAtMTtcbiAgICB9KSAhPT0gLTE7XG4gIH1cblxuICBwcml2YXRlIGFueVdlYmhvb2tGaWx0ZXJDb250YWluc0ZpbGVQYXRoQ29uZGl0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy53ZWJob29rRmlsdGVycy5maW5kSW5kZXgoZmcgPT4ge1xuICAgICAgcmV0dXJuIGZnLl9maWx0ZXJzLmZpbmRJbmRleChmID0+IGYudHlwZSA9PT0gV2ViaG9va0ZpbHRlclR5cGVzLkZJTEVfUEFUSCkgIT09IC0xO1xuICAgIH0pICE9PSAtMTtcbiAgfVxufVxuXG5mdW5jdGlvbiBzZXQyQXJyYXk8VD4oc2V0OiBTZXQ8VD4pOiBUW10ge1xuICBjb25zdCByZXQ6IFRbXSA9IFtdO1xuICBzZXQuZm9yRWFjaChlbCA9PiByZXQucHVzaChlbCkpO1xuICByZXR1cm4gcmV0O1xufVxuIl19