"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReportGroup = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const codebuild_generated_1 = require("./codebuild.generated");
const report_group_utils_1 = require("./report-group-utils");
class ReportGroupBase extends cdk.Resource {
    /**
     * Grants the given entity permissions to write (that is, upload reports to) this report group.
     *
     * @stability stable
     */
    grantWrite(identity) {
        const ret = iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resourceArns: [this.reportGroupArn],
        });
        if (this.exportBucket) {
            this.exportBucket.grantWrite(identity);
        }
        return ret;
    }
}
/**
 * The ReportGroup resource class.
 *
 * @stability stable
 */
class ReportGroup extends ReportGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.reportGroupName,
        });
        const resource = new codebuild_generated_1.CfnReportGroup(this, 'Resource', {
            type: 'TEST',
            exportConfig: {
                exportConfigType: props.exportBucket ? 'S3' : 'NO_EXPORT',
                s3Destination: props.exportBucket
                    ? {
                        bucket: props.exportBucket.bucketName,
                        encryptionDisabled: props.exportBucket.encryptionKey ? false : undefined,
                        encryptionKey: (_a = props.exportBucket.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
                        packaging: props.zipExport ? 'ZIP' : undefined,
                    }
                    : undefined,
            },
            name: props.reportGroupName,
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: cdk.RemovalPolicy.RETAIN,
        });
        this.reportGroupArn = this.getResourceArnAttribute(resource.attrArn, report_group_utils_1.reportGroupArnComponents(this.physicalName));
        this.reportGroupName = this.getResourceNameAttribute(
        // there is no separate name attribute,
        // so use Fn::Select + Fn::Split to make one
        cdk.Fn.select(1, cdk.Fn.split('/', resource.ref)));
        this.exportBucket = props.exportBucket;
    }
    /**
     * Reference an existing ReportGroup, defined outside of the CDK code, by name.
     *
     * @stability stable
     */
    static fromReportGroupName(scope, id, reportGroupName) {
        class Import extends ReportGroupBase {
            constructor() {
                super(...arguments);
                this.reportGroupName = reportGroupName;
                this.reportGroupArn = report_group_utils_1.renderReportGroupArn(scope, reportGroupName);
                this.exportBucket = undefined;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReportGroup = ReportGroup;
//# sourceMappingURL=data:application/json;base64,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