"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxGpuBuildImage = void 0;
const ecr = require("@aws-cdk/aws-ecr");
const core = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
const mappingName = 'AwsDeepLearningContainersRepositoriesAccounts';
/**
 * A CodeBuild GPU image running Linux.
 *
 * This class has public constants that represent the most popular GPU images from AWS Deep Learning Containers.
 *
 * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
 * @stability stable
 */
class LinuxGpuBuildImage {
    constructor(repositoryName, tag, account) {
        this.repositoryName = repositoryName;
        this.account = account;
        /**
         * The type of build environment.
         *
         * @stability stable
         */
        this.type = 'LINUX_GPU_CONTAINER';
        /**
         * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @stability stable
         */
        this.defaultComputeType = project_1.ComputeType.LARGE;
        /**
         * The type of principal that CodeBuild will use to pull this build Docker image.
         *
         * @stability stable
         */
        this.imagePullPrincipalType = project_1.ImagePullPrincipalType.SERVICE_ROLE;
        this.accountExpression = account !== null && account !== void 0 ? account : core.Fn.findInMap(mappingName, core.Aws.REGION, 'repositoryAccount');
        this.imageId = `${this.accountExpression}.dkr.ecr.${core.Aws.REGION}.${core.Aws.URL_SUFFIX}/${repositoryName}:${tag}`;
    }
    /**
     * Returns a Linux GPU build image from AWS Deep Learning Containers.
     *
     * @param repositoryName the name of the repository, for example "pytorch-inference".
     * @param tag the tag of the image, for example "1.5.0-gpu-py36-cu101-ubuntu16.04".
     * @param account the AWS account ID where the DLC repository for this region is hosted in.
     * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
     * @stability stable
     */
    static awsDeepLearningContainersImage(repositoryName, tag, account) {
        return new LinuxGpuBuildImage(repositoryName, tag, account);
    }
    /**
     * Function that allows the build image access to the construct tree.
     *
     * @stability stable
     */
    bind(scope, project, _options) {
        if (!this.account) {
            const scopeStack = core.Stack.of(scope);
            // Unfortunately, the account IDs of the DLC repositories are not the same in all regions.
            // Because of that, use a (singleton) Mapping to find the correct account
            if (!scopeStack.node.tryFindChild(mappingName)) {
                const mapping = {};
                // get the accounts from the region-info module
                const region2Accounts = region_info_1.RegionInfo.regionMap(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
                for (const [region, account] of Object.entries(region2Accounts)) {
                    mapping[region] = { repositoryAccount: account };
                }
                new core.CfnMapping(scopeStack, mappingName, { mapping });
            }
        }
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'AwsDlcRepositoryCodeBuild', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, this.accountExpression),
        });
        repository.grantPull(project);
        return {};
    }
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @stability stable
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`GPU images only support ComputeType '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxGpuBuildImage = LinuxGpuBuildImage;
/**
 * Tensorflow 1.14.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_14_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.14.0-gpu-py36-cu100-ubuntu16.04');
/**
 * Tensorflow 1.15.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.0-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 1.15.2 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.2-gpu-py37-cu100-ubuntu18.04');
/**
 * Tensorflow 1.15.2 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '1.15.2-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 2.0.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.0-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 2.0.1 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.1-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 2.1.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/**
 * Tensorflow 2.1.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/**
 * Tensorflow 2.2.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_2_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.2.0-gpu-py37-cu101-ubuntu18.04');
/**
 * PyTorch 1.2.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_2_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.2.0-gpu-py36-cu100-ubuntu16.04');
/**
 * PyTorch 1.3.1 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_3_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.3.1-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.4.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.4.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.5.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.5.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/**
 * MXNet 1.4.1 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_MXNET_1_4_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.4.1-gpu-py36-cu100-ubuntu16.04');
/**
 * MXNet 1.6.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_MXNET_1_6_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.6.0-gpu-py36-cu101-ubuntu16.04');
//# sourceMappingURL=data:application/json;base64,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