"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cache = exports.LocalCacheMode = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Local cache modes to enable for the CodeBuild Project.
 *
 * @stability stable
 */
var LocalCacheMode;
(function (LocalCacheMode) {
    LocalCacheMode["SOURCE"] = "LOCAL_SOURCE_CACHE";
    LocalCacheMode["DOCKER_LAYER"] = "LOCAL_DOCKER_LAYER_CACHE";
    LocalCacheMode["CUSTOM"] = "LOCAL_CUSTOM_CACHE";
})(LocalCacheMode = exports.LocalCacheMode || (exports.LocalCacheMode = {}));
/**
 * Cache options for CodeBuild Project.
 *
 * A cache can store reusable pieces of your build environment and use them across multiple builds.
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-caching.html
 * @stability stable
 */
class Cache {
    /**
     * @stability stable
     */
    static none() {
        return { _toCloudFormation: () => undefined, _bind: () => { return; } };
    }
    /**
     * Create a local caching strategy.
     *
     * @param modes the mode(s) to enable for local caching.
     * @stability stable
     */
    static local(...modes) {
        return {
            _toCloudFormation: () => ({
                type: 'LOCAL',
                modes,
            }),
            _bind: () => { return; },
        };
    }
    /**
     * Create an S3 caching strategy.
     *
     * @param bucket the S3 bucket to use for caching.
     * @param options additional options to pass to the S3 caching.
     * @stability stable
     */
    static bucket(bucket, options) {
        return {
            _toCloudFormation: () => ({
                type: 'S3',
                location: core_1.Fn.join('/', [bucket.bucketName, options && options.prefix || core_1.Aws.NO_VALUE]),
            }),
            _bind: (project) => {
                bucket.grantReadWrite(project);
            },
        };
    }
}
exports.Cache = Cache;
//# sourceMappingURL=data:application/json;base64,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