"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeBuildSpecs = exports.BuildSpec = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * BuildSpec for CodeBuild projects.
 *
 * @stability stable
 */
class BuildSpec {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * @stability stable
     */
    static fromObject(value) {
        return new ObjectBuildSpec(value);
    }
    /**
     * Use a file from the source as buildspec.
     *
     * Use this if you want to use a file different from 'buildspec.yml'`
     *
     * @stability stable
     */
    static fromSourceFilename(filename) {
        return new FilenameBuildSpec(filename);
    }
}
exports.BuildSpec = BuildSpec;
/**
 * BuildSpec that just returns the input unchanged
 */
class FilenameBuildSpec extends BuildSpec {
    constructor(filename) {
        super();
        this.filename = filename;
        this.isImmediate = false;
    }
    toBuildSpec() {
        return this.filename;
    }
    toString() {
        return `<buildspec file: ${this.filename}>`;
    }
}
/**
 * BuildSpec that understands about structure
 */
class ObjectBuildSpec extends BuildSpec {
    constructor(spec) {
        super();
        this.spec = spec;
        this.isImmediate = true;
    }
    toBuildSpec() {
        // We have to pretty-print the buildspec, otherwise
        // CodeBuild will not recognize it as an inline buildspec.
        return core_1.Lazy.uncachedString({
            produce: (ctx) => core_1.Stack.of(ctx.scope).toJsonString(this.spec, 2),
        });
    }
}
/**
 * Merge two buildspecs into a new BuildSpec
 *
 * NOTE: will currently only merge commands, not artifact
 * declarations, environment variables, secrets, or any
 * other configuration elements.
 *
 * Internal for now because it's not complete/good enough
 * to expose on the objects directly, but we need to it to
 * keep feature-parity for Project.
 *
 * @internal
 */
function mergeBuildSpecs(lhs, rhs) {
    if (!(lhs instanceof ObjectBuildSpec) || !(rhs instanceof ObjectBuildSpec)) {
        throw new Error('Can only merge buildspecs created using BuildSpec.fromObject()');
    }
    return new ObjectBuildSpec(copyCommands(lhs.spec, rhs.spec));
}
exports.mergeBuildSpecs = mergeBuildSpecs;
/**
 * Extend buildSpec phases with the contents of another one
 */
function copyCommands(buildSpec, extend) {
    if (buildSpec.version === '0.1') {
        throw new Error('Cannot extend buildspec at version "0.1". Set the version to "0.2" or higher instead.');
    }
    const ret = Object.assign({}, buildSpec); // Return a copy
    ret.phases = Object.assign({}, ret.phases);
    for (const phaseName of Object.keys(extend.phases)) {
        const phase = ret.phases[phaseName] = Object.assign({}, ret.phases[phaseName]);
        phase.commands = [...phase.commands || [], ...extend.phases[phaseName].commands];
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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