import sys
import os
import argparse
import logging

from .constants import SUPPORTED_PROGRAMMERS, DEFAULT_PROGRAMMER, PROGRAMMER_JLINK
from .programmer import Programmer
from .jlink_programmer import JLinkProgrammer


def main():
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(
        description="Flash embedded devices and manage programmers",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # List connected programmers
  bmlab-scan
  
  # Flash with auto-detected JLink (first available)
  bmlab-flash firmware.hex
  
  # Flash with specific JLink serial
  bmlab-flash firmware.hex --serial 123456
  
  # Flash via JLink IP address
  bmlab-flash firmware.hex --ip 192.168.1.100
  
  # Flash with specific MCU
  bmlab-flash firmware.hex --mcu STM32F765ZG
  
  # Specify programmer explicitly
  bmlab-flash firmware.hex --programmer jlink --serial 123456
        """
    )
    
    parser.add_argument(
        "firmware_file",
        type=str,
        help="Path to firmware file (.hex or .bin)"
    )
    
    parser.add_argument(
        "--serial", "-s",
        type=int,
        default=None,
        help="Programmer serial number (optional, will use first available if not specified)"
    )
    
    parser.add_argument(
        "--ip",
        type=str,
        default=None,
        help="JLink IP address for network connection (e.g., 192.168.1.100)"
    )
    
    parser.add_argument(
        "--mcu",
        type=str,
        default=None,
        help="MCU name (e.g., STM32F765ZG). If not provided, will auto-detect"
    )
    
    parser.add_argument(
        "--programmer", "-p",
        type=str,
        default=DEFAULT_PROGRAMMER,
        choices=SUPPORTED_PROGRAMMERS,
        help=f"Programmer type (default: {DEFAULT_PROGRAMMER})"
    )
    
    parser.add_argument(
        "--log-level", "-l",
        type=str,
        default="WARNING",
        choices=["DEBUG", "INFO", "WARNING", "ERROR"],
        help="Set logging level (default: WARNING)"
    )
    
    args = parser.parse_args()
    
    # Validate that --serial and --ip are mutually exclusive
    if args.serial and args.ip:
        print("Error: Cannot specify both --serial and --ip")
        sys.exit(1)
    
    try:
        # Convert log level string to logging constant
        log_level = getattr(logging, args.log_level.upper())
        
        # Create programmer instance
        if args.programmer.lower() == PROGRAMMER_JLINK:
            prog = JLinkProgrammer(serial=args.serial, ip_addr=args.ip, log_level=log_level)
        else:
            raise NotImplementedError(f"Programmer '{args.programmer}' is not yet implemented")
        
        # Flash firmware
        fw_file = os.path.abspath(args.firmware_file)
        if not os.path.exists(fw_file):
            print(f"Error: Firmware file not found: {fw_file}")
            print(f"To list connected devices, run: bmlab-scan")
            sys.exit(1)
        
        # Check if programmer is available
        if not prog.probe():
            raise RuntimeError(f"Programmer not found or not accessible")
        
        print(f"Flashing {fw_file}...")
        
        # Flash firmware (will auto-connect, verify, flash, reset, and disconnect)
        if not prog.flash(fw_file, mcu=args.mcu):
            raise RuntimeError("Flash operation failed")
        
        print("\n✓ Flashing completed successfully!")
        
    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main()
