"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_glue_1 = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("..");
// --------------------------------------------------------------
// Test deployment with role creation
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    var _a, _b;
    // Stack
    const stack = new core_1.Stack();
    const _jobRole = new aws_iam_1.Role(stack, 'CustomETLJobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com')
    });
    const cfnJobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: 's3://fakescriptlocation/fakebucket',
        },
        role: _jobRole.roleArn
    }, 'testETLJob', {});
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    const _glueJob = defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect((_a = _glueJob[2]) === null || _a === void 0 ? void 0 : _a[0]).toBeDefined();
    expect((_b = _glueJob[2]) === null || _b === void 0 ? void 0 : _b[0]).toBeInstanceOf(aws_s3_1.Bucket);
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://fakescriptlocation/fakebucket"
            },
            Role: {
                "Fn::GetAtt": [
                    "CustomETLJobRole90A83A66",
                    "Arn"
                ]
            },
            GlueVersion: "2.0",
            NumberOfWorkers: 2,
            SecurityConfiguration: "testETLJob",
            WorkerType: "G.1X"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an existing Glue Job
// --------------------------------------------------------------
test('Create a Glue Job outside the construct', () => {
    // Stack
    const stack = new core_1.Stack();
    const _existingCfnJob = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
        command: {
            name: 'pythonshell',
            pythonVersion: '2',
            scriptLocation: 's3://existingfakelocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        allocatedCapacity: 2,
        maxCapacity: 4,
        numberOfWorkers: 2,
        workerType: 'Standard'
    });
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    const _glueJob = defaults.buildGlueJob(stack, {
        existingCfnJob: _existingCfnJob,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(_glueJob[2]).not.toBeDefined();
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            AllocatedCapacity: 2,
            Command: {
                Name: "pythonshell",
                PythonVersion: "2",
                ScriptLocation: "s3://existingfakelocation/existingScript",
            },
            GlueVersion: "1",
            MaxCapacity: 4,
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            WorkerType: "Standard",
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Provide additional parameters other than default ones
// --------------------------------------------------------------
test('Test custom deployment properties', () => {
    // Stack
    const stack = new core_1.Stack();
    const _commandName = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _commandName,
            pythonVersion: '3',
            scriptLocation: 's3://existingfakelocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        numberOfWorkers: 2,
        workerType: 'Standard'
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // check if Glue Job Resource was created correctly
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://existingfakelocation/existingScript",
            },
            GlueVersion: "1",
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            SecurityConfiguration: "ETLJobSecurityConfig",
            WorkerType: "Standard",
        },
        Type: "AWS::Glue::Job"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the role is created
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazon.com"
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            Description: "Existing role"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the security config is created
    expect(stack).toHaveResourceLike('AWS::Glue::SecurityConfiguration', {
        Properties: {
            EncryptionConfiguration: {
                JobBookmarksEncryption: {
                    JobBookmarksEncryptionMode: "CSE-KMS",
                    KmsKeyArn: {
                        "Fn::Join": [
                            "", [
                                "arn:", {
                                    Ref: "AWS::Partition",
                                },
                                ":kms:", {
                                    Ref: "AWS::Region",
                                },
                                ":", {
                                    Ref: "AWS::AccountId",
                                },
                                ":alias/aws/glue",
                            ],
                        ],
                    },
                },
                S3Encryptions: [{
                        S3EncryptionMode: "SSE-S3",
                    }],
            },
            Name: "ETLJobSecurityConfig",
        },
        Type: "AWS::Glue::SecurityConfiguration",
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Do not supply parameters and error out
// --------------------------------------------------------------
test('Do no supply glueJobProps or existingCfnJob and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' }))
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Allow the construct to create the job role required
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [{
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazonaws.com"
                        }
                    }],
                Version: "2012-10-17"
            },
            Description: "Service role that Glue custom ETL jobs will assume for exeuction"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when output location is provided
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3,
            existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                versioned: false,
                bucketName: 'outputbucket',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: core_1.RemovalPolicy.DESTROY
            })
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        Type: 'AWS::S3::Bucket',
        Properties: {
            BucketEncryption: {
                ServerSideEncryptionConfiguration: [{
                        ServerSideEncryptionByDefault: {
                            SSEAlgorithm: "AES256"
                        }
                    }]
            },
            BucketName: "outputbucket"
        },
        UpdateReplacePolicy: "Delete",
        DeletionPolicy: "Delete"
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when script location not provided - throw error
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    try {
        defaults.buildGlueJob(stack, {
            glueJobProps: cfnJobProps,
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3,
                existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                    versioned: false
                })
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
        expect(error.message).toEqual('Script location has to be provided as an s3 Url location. Script location cannot be empty');
    }
});
// --------------------------------------------------------------
// Dont pass Job Command attributes and it should throw an error
// --------------------------------------------------------------
test('Test for incorrect Job Command property', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {},
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Check for CfnJob.JobCommandProperty type
// --------------------------------------------------------------
test('check for JobCommandProperty type', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {
                command: {
                    fakekey: 'fakevalue'
                }
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Supply maxCapacity with GlueVersion 2.0 and error out
// --------------------------------------------------------------
test('GlueJob configuration with glueVersion 2.0 should not support maxCapacity and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                glueVersion: '2.0',
                maxCapacity: '2'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Fail if setting maxCapacity and WorkerType/ NumberOfWorkers
// --------------------------------------------------------------
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                workerType: 'Standard'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2,
                workerType: 'G1.X'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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