"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const elb = require("@aws-cdk/aws-elasticloadbalancingv2");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('Test ObtainAlb with existing ALB', () => {
    const stack = new core_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, existingLoadBalancer);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "unique-name",
    });
});
test('Test ObtainAlb for new ALB with provided props', () => {
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, undefined, {
        loadBalancerName: 'new-loadbalancer',
        vpc,
        internetFacing: true
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "new-loadbalancer",
        Scheme: "internet-facing",
    });
});
test('Test ObtainAlb for new ALB with default props', () => {
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: "internal",
    });
});
test('Test with custom logging bucket props', () => {
    // Creating ALB logging requires a region and account (but
    // these can be fake in unit tests)
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const testName = 'test-name';
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, true, { bucketName: testName });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        BucketName: testName
    });
});
test('Test with no logging', () => {
    const stack = new core_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, false);
    expect(stack).not.toHaveResourceLike('AWS::S3::Bucket', {});
});
function CreateTestVpc(stack) {
    return defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
}
function CreateTestLoadBalancer(stack, vpc) {
    return new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
}
function GetCert(stack) {
    return acm.Certificate.fromCertificateArn(stack, 'not-really-a-cert', "arn:aws:acm:us-east-1:123456789012:certificate/85c52dc8-1b37-4afd-a7aa-f03aac2db0cc");
}
function CreateTestFunction(stack, id) {
    return new lambda.Function(stack, id, {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
}
function CreateTestListener(stack, id, alb) {
    return new elb.ApplicationListener(stack, id, {
        loadBalancer: alb,
        protocol: elb.ApplicationProtocol.HTTP
    });
}
test('Test add single lambda target group with no customization', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testFunction = CreateTestFunction(stack, 'test-function');
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    // This is the code we're testing
    defaults.AddLambdaTarget(stack, 'test-lambda-target', testListener, testFunction);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        DefaultActions: [
            {
                TargetGroupArn: {
                    Ref: "testlambdatargettg7E5C32F4"
                },
                Type: "forward"
            }
        ],
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda",
    });
});
test('Test add single lambda target group with target group props', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testFunction = CreateTestFunction(stack, 'test-function');
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    const targetGroupName = 'test-group';
    // This is the code we're testing
    defaults.AddLambdaTarget(stack, 'test-lambda-target', testListener, testFunction, undefined, { targetGroupName });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda",
        Name: targetGroupName,
    });
});
test('Test add rule props for second lambda target group', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testFunction = CreateTestFunction(stack, 'test-function');
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    const targetGroupName = 'test-group';
    const pathPattern = '*admin*';
    defaults.AddLambdaTarget(stack, 'test-lambda-target', testListener, testFunction, undefined, { targetGroupName });
    // This is the code we're testing
    const ruleProps = {
        conditions: [elb.ListenerCondition.pathPatterns([pathPattern])],
        priority: 10
    };
    defaults.AddLambdaTarget(stack, 'test-second-lambda-target', testListener, testFunction, ruleProps, { targetGroupName });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        pathPattern
                    ]
                }
            }
        ]
    });
});
test('Test adding a listener with defaults', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testCert = GetCert(stack);
    const listener = defaults.AddListener(stack, testAlb, { certificates: [testCert] });
    //  Need to add a target because a listener is not allowed to exist without a target or action
    defaults.AddLambdaTarget(stack, 'dummy-target', listener, CreateTestFunction(stack, 'dummy-function'));
    // This should create 2 listeners, HTTPS plus redirect of HTTP
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
});
test('Test adding an HTTPS listener with no cert (error)', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const app = () => {
        defaults.AddListener(stack, testAlb, {});
    };
    expect(app).toThrowError('A listener using HTTPS protocol requires a certificate');
});
test('Test adding an HTTP listener with a cert (error)', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testCert = GetCert(stack);
    const app = () => {
        defaults.AddListener(stack, testAlb, { protocol: 'HTTP', certificates: [testCert] });
    };
    expect(app).toThrowError('HTTP listeners cannot use a certificate');
});
test('Test adding a HTTP listener', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const listener = defaults.AddListener(stack, testAlb, { protocol: 'HTTP' });
    //  Need to add a target because a listener is not allowed to exist without a target or action
    defaults.AddLambdaTarget(stack, 'dummy-target', listener, CreateTestFunction(stack, 'dummy-function'));
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::Listener', 1);
});
test('Test sending custom logging bucket props', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const listener = defaults.AddListener(stack, testAlb, { protocol: 'HTTP' });
    //  Need to add a target because a listener is not allowed to exist without a target or action
    defaults.AddLambdaTarget(stack, 'dummy-target', listener, CreateTestFunction(stack, 'dummy-function'));
});
test('Test GetActiveListener with 0 listeners', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const app = () => {
        defaults.GetActiveListener(testAlb.listeners);
    };
    expect(app).toThrowError('There are no listeners in the ALB');
});
test('Test GetActiveListener with 1 listener', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    defaults.AddListener(stack, testAlb, { protocol: 'HTTP' });
    const listener = defaults.GetActiveListener(testAlb.listeners);
    expect(listener.node.defaultChild.protocol).toBe('HTTP');
});
test('Test GetActiveListener with 2 listeners', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = CreateTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testCert = GetCert(stack);
    defaults.AddListener(stack, testAlb, { certificates: [testCert] });
    const listener = defaults.GetActiveListener(testAlb.listeners);
    expect(listener.node.defaultChild.protocol).toBe('HTTPS');
});
//# sourceMappingURL=data:application/json;base64,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