"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("@aws-cdk/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_helper_1 = require("./kms-helper");
function buildQueue(scope, id, props) {
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            const checkFifo = props.queueProps.fifo ? true : undefined;
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), { ...props.queueProps, fifo: checkFifo });
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey) {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                queueProps.encryptionMasterKey = props.encryptionKey;
            }
            else {
                queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return [queue, queue.encryptionMasterKey];
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return [props.existingQueueObj];
    }
}
exports.buildQueue = buildQueue;
function buildDeadLetterQueue(scope, props) {
    if (!props.existingQueueObj && (props.deployDeadLetterQueue || props.deployDeadLetterQueue === undefined)) {
        // Create the Dead Letter Queue
        const [dlq] = buildQueue(scope, 'deadLetterQueue', {
            queueProps: props.deadLetterQueueProps
        });
        const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
        // Setup the Dead Letter Queue interface
        const dlqInterface = {
            maxReceiveCount: mrc,
            queue: dlq
        };
        // Return the dead letter queue interface
        return dlqInterface;
    }
    return;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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