"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.buildS3Bucket = exports.createAlbLoggingBucket = exports.createLoggingBucket = exports.applySecureBucketPolicy = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`,
            `${s3Bucket.bucketArn}`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: _reason
        }
    ]);
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function createAlbLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: "This is a log bucket for an Application Load Balancer"
        }
    ]);
    return loggingBucket;
}
exports.createAlbLoggingBucket = createAlbLoggingBucket;
function buildS3Bucket(scope, props, bucketId) {
    var _a, _b, _c;
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: aws_s3_1.StorageClass.GLACIER,
                    transitionAfter: core_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let customBucketProps;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    // If logging S3 access logs is enabled/undefined and an existing bucket object is not provided
    if (props.logS3AccessLogs !== false && !((_a = props.bucketProps) === null || _a === void 0 ? void 0 : _a.serverAccessLogsBucket)) {
        // Create the Logging Bucket
        let loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
        if (props.loggingBucketProps) {
            // User provided logging bucket props
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, props.loggingBucketProps);
        }
        else if ((_b = props.bucketProps) === null || _b === void 0 ? void 0 : _b.removalPolicy) {
            // If the client explicitly specified a removal policy for the main bucket,
            // then replicate that policy on the logging bucket
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, { removalPolicy: props.bucketProps.removalPolicy });
        }
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, loggingBucketProps);
    }
    // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
    if (((_c = props.bucketProps) === null || _c === void 0 ? void 0 : _c.versioned) === undefined || props.bucketProps.versioned) {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
    }
    else {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    customBucketProps = props.bucketProps ? utils_1.overrideProps(customBucketProps, props.bucketProps) : customBucketProps;
    const s3Bucket = new s3.Bucket(scope, _bucketId, customBucketProps);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
exports.buildS3Bucket = buildS3Bucket;
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    utils_1.addCfnSuppressRules(fnResource, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    utils_1.addCfnSuppressRules(policyResource, [
        {
            id: 'W12',
            reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
        }
    ]);
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtYnVja2V0LWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInMzLWJ1Y2tldC1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOzs7QUFJSCxzQ0FBc0M7QUFFdEMsNkRBQXNEO0FBQ3RELG1DQUE2RDtBQUM3RCw4Q0FBeUU7QUFDekUsNENBQStDO0FBQy9DLHdDQUF5QztBQWF6QyxTQUFnQix1QkFBdUIsQ0FBQyxRQUFtQjtJQUN6RCwrREFBK0Q7SUFDL0QsUUFBUSxDQUFDLG1CQUFtQixDQUMxQixJQUFJLHlCQUFlLENBQUM7UUFDbEIsR0FBRyxFQUFFLFdBQVc7UUFDaEIsU0FBUyxFQUFFO1lBQ1QsR0FBRyxRQUFRLENBQUMsU0FBUyxJQUFJO1lBQ3pCLEdBQUcsUUFBUSxDQUFDLFNBQVMsRUFBRTtTQUN4QjtRQUNELE9BQU8sRUFBRSxDQUFDLEdBQUcsQ0FBQztRQUNkLFVBQVUsRUFBRSxDQUFDLElBQUksc0JBQVksRUFBRSxDQUFDO1FBQ2hDLE1BQU0sRUFBRSxnQkFBTSxDQUFDLElBQUk7UUFDbkIsVUFBVSxFQUNWO1lBQ0UsSUFBSSxFQUFFO2dCQUNKLHFCQUFxQixFQUFFLE9BQU87YUFDL0I7U0FDRjtLQUNGLENBQUMsQ0FDSCxDQUFDO0FBQ0osQ0FBQztBQXBCRCwwREFvQkM7QUFFRCxTQUFnQixtQkFBbUIsQ0FBQyxLQUFnQixFQUNsRCxRQUFnQixFQUNoQixrQkFBa0M7SUFFbEMsNEJBQTRCO0lBQzVCLE1BQU0sYUFBYSxHQUFjLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFFcEYsdUJBQXVCLENBQUMsYUFBYSxDQUFDLENBQUM7SUFFdkMsK0NBQStDO0lBQy9DLE1BQU0scUJBQXFCLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFpQixDQUFDO0lBRXZGLCtFQUErRTtJQUMvRSxxQkFBcUIsQ0FBQyxtQkFBbUIsQ0FBQyxlQUFlLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztJQUUvRSxtRUFBbUU7SUFDbkUscUJBQXFCLENBQUMsMkJBQTJCLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUVsRixJQUFJLE9BQU8sR0FBRyx3RUFBd0UsQ0FBQztJQUV2RixJQUFJLFFBQVEsS0FBSyx5QkFBeUIsRUFBRTtRQUMxQyxPQUFPLEdBQUcsaUZBQWlGLENBQUM7S0FDN0Y7SUFFRCwyQkFBbUIsQ0FBQyxxQkFBcUIsRUFBRTtRQUN6QztZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLE9BQU87U0FDaEI7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPLGFBQWEsQ0FBQztBQUN2QixDQUFDO0FBaENELGtEQWdDQztBQUVELFNBQWdCLHNCQUFzQixDQUFDLEtBQWdCLEVBQ3JELFFBQWdCLEVBQ2hCLGtCQUFrQztJQUVsQyw0QkFBNEI7SUFDNUIsTUFBTSxhQUFhLEdBQWMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztJQUVwRix1QkFBdUIsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUV2QywrQ0FBK0M7SUFDL0MsTUFBTSxxQkFBcUIsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQWlCLENBQUM7SUFFdkYsMkJBQW1CLENBQUMscUJBQXFCLEVBQUU7UUFDekM7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSx1REFBdUQ7U0FDaEU7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPLGFBQWEsQ0FBQztBQUN2QixDQUFDO0FBcEJELHdEQW9CQztBQUVELFNBQWdCLGFBQWEsQ0FBQyxLQUFnQixFQUM1QyxLQUF5QixFQUN6QixRQUFpQjs7SUFFakIsc0ZBQXNGO0lBQ3RGLE1BQU0sY0FBYyxHQUF1QixDQUFDO1lBQzFDLDRCQUE0QixFQUFFLENBQUM7b0JBQzdCLFlBQVksRUFBRSxxQkFBWSxDQUFDLE9BQU87b0JBQ2xDLGVBQWUsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztpQkFDbkMsQ0FBQztTQUNILENBQUMsQ0FBQztJQUVILGdDQUFnQztJQUNoQyxJQUFJLGlCQUFpQyxDQUFDO0lBQ3RDLElBQUksYUFBYSxDQUFDO0lBQ2xCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsUUFBUSxHQUFHLFVBQVUsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDO0lBQ2hFLE1BQU0sZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLENBQUMsQ0FBQyxRQUFRLEdBQUcsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDO0lBRXJGLCtGQUErRjtJQUMvRixJQUFJLEtBQUssQ0FBQyxlQUFlLEtBQUssS0FBSyxJQUFJLENBQUMsT0FBQyxLQUFLLENBQUMsV0FBVywwQ0FBRSxzQkFBc0IsQ0FBQyxFQUFFO1FBQ25GLDRCQUE0QjtRQUM1QixJQUFJLGtCQUFrQixHQUFHLG1DQUFjLEVBQUUsQ0FBQztRQUUxQyxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsRUFBRTtZQUM1QixxQ0FBcUM7WUFDckMsa0JBQWtCLEdBQUcscUJBQWEsQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQztTQUNsRjthQUFNLFVBQUksS0FBSyxDQUFDLFdBQVcsMENBQUUsYUFBYSxFQUFFO1lBQzNDLDJFQUEyRTtZQUMzRSxtREFBbUQ7WUFDbkQsa0JBQWtCLEdBQUcscUJBQWEsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUM7U0FDNUc7UUFFRCxhQUFhLEdBQUcsbUJBQW1CLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLGtCQUFrQixDQUFDLENBQUM7S0FDbEY7SUFDRCx5RUFBeUU7SUFDekUsSUFBSSxPQUFBLEtBQUssQ0FBQyxXQUFXLDBDQUFFLFNBQVMsTUFBSyxTQUFTLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxTQUFTLEVBQUU7UUFDN0UsaUJBQWlCLEdBQUcsbUNBQWMsQ0FBQyxhQUFhLEVBQUUsY0FBYyxDQUFDLENBQUM7S0FDbkU7U0FBTTtRQUNMLGlCQUFpQixHQUFHLG1DQUFjLENBQUMsYUFBYSxDQUFDLENBQUM7S0FDbkQ7SUFFRCxpQkFBaUIsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxxQkFBYSxDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUM7SUFFaEgsTUFBTSxRQUFRLEdBQWMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztJQUUvRSx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUVsQyxPQUFPLENBQUMsUUFBUSxFQUFFLGFBQWEsQ0FBQyxDQUFDO0FBQ25DLENBQUM7QUFoREQsc0NBZ0RDO0FBRUQsU0FBZ0IsNENBQTRDLENBQUMsU0FBb0IsRUFBRSxTQUFpQjtJQUNsRyxNQUFNLDRCQUE0QixHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBb0IsQ0FBQztJQUMvRixNQUFNLG9DQUFvQyxHQUFHLDRCQUE0QixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFhLENBQUM7SUFDN0csTUFBTSxzQ0FBc0MsR0FBRyxvQ0FBb0MsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBZSxDQUFDO0lBRWxJLDRDQUE0QztJQUM1QyxNQUFNLFVBQVUsR0FBRyw0QkFBNEIsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBdUIsQ0FBQztJQUNqRywyQkFBbUIsQ0FBQyxVQUFVLEVBQUU7UUFDOUI7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSxvTUFBb007U0FDN007UUFDRDtZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLGlGQUFpRjtTQUMxRjtRQUNEO1lBQ0UsRUFBRSxFQUFFLEtBQUs7WUFDVCxNQUFNLEVBQUUsaUVBQWlFO1NBQzFFO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsNENBQTRDO0lBQzVDLE1BQU0sY0FBYyxHQUFHLHNDQUFzQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFrQixDQUFDO0lBQzFHLDJCQUFtQixDQUFDLGNBQWMsRUFBRTtRQUNsQztZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLGtHQUFrRztTQUMzRztLQUNGLENBQUMsQ0FBQztBQUNMLENBQUM7QUE5QkQsb0dBOEJDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMjEgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgRGVmYXVsdFMzUHJvcHMgfSBmcm9tICcuL3MzLWJ1Y2tldC1kZWZhdWx0cyc7XG5pbXBvcnQgeyBvdmVycmlkZVByb3BzLCBhZGRDZm5TdXBwcmVzc1J1bGVzIH0gZnJvbSAnLi91dGlscyc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIEVmZmVjdCwgQW55UHJpbmNpcGFsIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBTdG9yYWdlQ2xhc3MgfSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbi8vIE5vdGU6IFRvIGVuc3VyZSBDREt2MiBjb21wYXRpYmlsaXR5LCBrZWVwIHRoZSBpbXBvcnQgc3RhdGVtZW50IGZvciBDb25zdHJ1Y3Qgc2VwYXJhdGVcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkUzNCdWNrZXRQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVja2V0UHJvcHM/OiBzMy5CdWNrZXRQcm9wcztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ2dpbmdCdWNrZXRQcm9wcz86IHMzLkJ1Y2tldFByb3BzO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbG9nUzNBY2Nlc3NMb2dzPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFwcGx5U2VjdXJlQnVja2V0UG9saWN5KHMzQnVja2V0OiBzMy5CdWNrZXQpOiB2b2lkIHtcbiAgLy8gQXBwbHkgYnVja2V0IHBvbGljeSB0byBlbmZvcmNlIGVuY3J5cHRpb24gb2YgZGF0YSBpbiB0cmFuc2l0XG4gIHMzQnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koXG4gICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICBzaWQ6ICdIdHRwc09ubHknLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIGAke3MzQnVja2V0LmJ1Y2tldEFybn0vKmAsXG4gICAgICAgIGAke3MzQnVja2V0LmJ1Y2tldEFybn1gXG4gICAgICBdLFxuICAgICAgYWN0aW9uczogWycqJ10sXG4gICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgIGVmZmVjdDogRWZmZWN0LkRFTlksXG4gICAgICBjb25kaXRpb25zOlxuICAgICAge1xuICAgICAgICBCb29sOiB7XG4gICAgICAgICAgJ2F3czpTZWN1cmVUcmFuc3BvcnQnOiAnZmFsc2UnXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KVxuICApO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlTG9nZ2luZ0J1Y2tldChzY29wZTogQ29uc3RydWN0LFxuICBidWNrZXRJZDogc3RyaW5nLFxuICBsb2dnaW5nQnVja2V0UHJvcHM6IHMzLkJ1Y2tldFByb3BzKTogczMuQnVja2V0IHtcblxuICAvLyBDcmVhdGUgdGhlIExvZ2dpbmcgQnVja2V0XG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXQ6IHMzLkJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc2NvcGUsIGJ1Y2tldElkLCBsb2dnaW5nQnVja2V0UHJvcHMpO1xuXG4gIGFwcGx5U2VjdXJlQnVja2V0UG9saWN5KGxvZ2dpbmdCdWNrZXQpO1xuXG4gIC8vIEV4dHJhY3QgdGhlIENmbkJ1Y2tldCBmcm9tIHRoZSBsb2dnaW5nQnVja2V0XG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXRSZXNvdXJjZSA9IGxvZ2dpbmdCdWNrZXQubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgczMuQ2ZuQnVja2V0O1xuXG4gIC8vIE92ZXJyaWRlIGFjY2Vzc0NvbnRyb2wgY29uZmlndXJhdGlvbiBhbmQgYWRkIG1ldGFkYXRhIGZvciB0aGUgbG9nZ2luZyBidWNrZXRcbiAgbG9nZ2luZ0J1Y2tldFJlc291cmNlLmFkZFByb3BlcnR5T3ZlcnJpZGUoJ0FjY2Vzc0NvbnRyb2wnLCAnTG9nRGVsaXZlcnlXcml0ZScpO1xuXG4gIC8vIFJlbW92ZSB0aGUgZGVmYXVsdCBMaWZlY3ljbGVDb25maWd1cmF0aW9uIGZvciB0aGUgTG9nZ2luZyBCdWNrZXRcbiAgbG9nZ2luZ0J1Y2tldFJlc291cmNlLmFkZFByb3BlcnR5RGVsZXRpb25PdmVycmlkZSgnTGlmZWN5Y2xlQ29uZmlndXJhdGlvbi5SdWxlcycpO1xuXG4gIGxldCBfcmVhc29uID0gXCJUaGlzIFMzIGJ1Y2tldCBpcyB1c2VkIGFzIHRoZSBhY2Nlc3MgbG9nZ2luZyBidWNrZXQgZm9yIGFub3RoZXIgYnVja2V0XCI7XG5cbiAgaWYgKGJ1Y2tldElkID09PSAnQ2xvdWRmcm9udExvZ2dpbmdCdWNrZXQnKSB7XG4gICAgX3JlYXNvbiA9IFwiVGhpcyBTMyBidWNrZXQgaXMgdXNlZCBhcyB0aGUgYWNjZXNzIGxvZ2dpbmcgYnVja2V0IGZvciBDbG91ZEZyb250IERpc3RyaWJ1dGlvblwiO1xuICB9XG5cbiAgYWRkQ2ZuU3VwcHJlc3NSdWxlcyhsb2dnaW5nQnVja2V0UmVzb3VyY2UsIFtcbiAgICB7XG4gICAgICBpZDogJ1czNScsXG4gICAgICByZWFzb246IF9yZWFzb25cbiAgICB9XG4gIF0pO1xuXG4gIHJldHVybiBsb2dnaW5nQnVja2V0O1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlQWxiTG9nZ2luZ0J1Y2tldChzY29wZTogQ29uc3RydWN0LFxuICBidWNrZXRJZDogc3RyaW5nLFxuICBsb2dnaW5nQnVja2V0UHJvcHM6IHMzLkJ1Y2tldFByb3BzKTogczMuQnVja2V0IHtcblxuICAvLyBDcmVhdGUgdGhlIExvZ2dpbmcgQnVja2V0XG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXQ6IHMzLkJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc2NvcGUsIGJ1Y2tldElkLCBsb2dnaW5nQnVja2V0UHJvcHMpO1xuXG4gIGFwcGx5U2VjdXJlQnVja2V0UG9saWN5KGxvZ2dpbmdCdWNrZXQpO1xuXG4gIC8vIEV4dHJhY3QgdGhlIENmbkJ1Y2tldCBmcm9tIHRoZSBsb2dnaW5nQnVja2V0XG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXRSZXNvdXJjZSA9IGxvZ2dpbmdCdWNrZXQubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgczMuQ2ZuQnVja2V0O1xuXG4gIGFkZENmblN1cHByZXNzUnVsZXMobG9nZ2luZ0J1Y2tldFJlc291cmNlLCBbXG4gICAge1xuICAgICAgaWQ6ICdXMzUnLFxuICAgICAgcmVhc29uOiBcIlRoaXMgaXMgYSBsb2cgYnVja2V0IGZvciBhbiBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyXCJcbiAgICB9XG4gIF0pO1xuXG4gIHJldHVybiBsb2dnaW5nQnVja2V0O1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRTM0J1Y2tldChzY29wZTogQ29uc3RydWN0LFxuICBwcm9wczogQnVpbGRTM0J1Y2tldFByb3BzLFxuICBidWNrZXRJZD86IHN0cmluZyk6IFtzMy5CdWNrZXQsIHMzLkJ1Y2tldD9dIHtcblxuICAvKiogRGVmYXVsdCBMaWZlIEN5Y2xlIHBvbGljeSB0byB0cmFuc2l0aW9uIG9sZGVyIHZlcnNpb25zIHRvIEdsYWNpZXIgYWZ0ZXIgOTAgZGF5cyAqL1xuICBjb25zdCBsaWZlY3ljbGVSdWxlczogczMuTGlmZWN5Y2xlUnVsZVtdID0gW3tcbiAgICBub25jdXJyZW50VmVyc2lvblRyYW5zaXRpb25zOiBbe1xuICAgICAgc3RvcmFnZUNsYXNzOiBTdG9yYWdlQ2xhc3MuR0xBQ0lFUixcbiAgICAgIHRyYW5zaXRpb25BZnRlcjogRHVyYXRpb24uZGF5cyg5MClcbiAgICB9XVxuICB9XTtcblxuICAvLyBDcmVhdGUgdGhlIEFwcGxpY2F0aW9uIEJ1Y2tldFxuICBsZXQgY3VzdG9tQnVja2V0UHJvcHM6IHMzLkJ1Y2tldFByb3BzO1xuICBsZXQgbG9nZ2luZ0J1Y2tldDtcbiAgY29uc3QgX2J1Y2tldElkID0gYnVja2V0SWQgPyBidWNrZXRJZCArICdTM0J1Y2tldCcgOiAnUzNCdWNrZXQnO1xuICBjb25zdCBfbG9nZ2luZ0J1Y2tldElkID0gYnVja2V0SWQgPyBidWNrZXRJZCArICdTM0xvZ2dpbmdCdWNrZXQnIDogJ1MzTG9nZ2luZ0J1Y2tldCc7XG5cbiAgLy8gSWYgbG9nZ2luZyBTMyBhY2Nlc3MgbG9ncyBpcyBlbmFibGVkL3VuZGVmaW5lZCBhbmQgYW4gZXhpc3RpbmcgYnVja2V0IG9iamVjdCBpcyBub3QgcHJvdmlkZWRcbiAgaWYgKHByb3BzLmxvZ1MzQWNjZXNzTG9ncyAhPT0gZmFsc2UgJiYgIShwcm9wcy5idWNrZXRQcm9wcz8uc2VydmVyQWNjZXNzTG9nc0J1Y2tldCkpIHtcbiAgICAvLyBDcmVhdGUgdGhlIExvZ2dpbmcgQnVja2V0XG4gICAgbGV0IGxvZ2dpbmdCdWNrZXRQcm9wcyA9IERlZmF1bHRTM1Byb3BzKCk7XG5cbiAgICBpZiAocHJvcHMubG9nZ2luZ0J1Y2tldFByb3BzKSB7XG4gICAgICAvLyBVc2VyIHByb3ZpZGVkIGxvZ2dpbmcgYnVja2V0IHByb3BzXG4gICAgICBsb2dnaW5nQnVja2V0UHJvcHMgPSBvdmVycmlkZVByb3BzKGxvZ2dpbmdCdWNrZXRQcm9wcywgcHJvcHMubG9nZ2luZ0J1Y2tldFByb3BzKTtcbiAgICB9IGVsc2UgaWYgKHByb3BzLmJ1Y2tldFByb3BzPy5yZW1vdmFsUG9saWN5KSB7XG4gICAgICAvLyBJZiB0aGUgY2xpZW50IGV4cGxpY2l0bHkgc3BlY2lmaWVkIGEgcmVtb3ZhbCBwb2xpY3kgZm9yIHRoZSBtYWluIGJ1Y2tldCxcbiAgICAgIC8vIHRoZW4gcmVwbGljYXRlIHRoYXQgcG9saWN5IG9uIHRoZSBsb2dnaW5nIGJ1Y2tldFxuICAgICAgbG9nZ2luZ0J1Y2tldFByb3BzID0gb3ZlcnJpZGVQcm9wcyhsb2dnaW5nQnVja2V0UHJvcHMsIHsgcmVtb3ZhbFBvbGljeTogcHJvcHMuYnVja2V0UHJvcHMucmVtb3ZhbFBvbGljeSB9KTtcbiAgICB9XG5cbiAgICBsb2dnaW5nQnVja2V0ID0gY3JlYXRlTG9nZ2luZ0J1Y2tldChzY29wZSwgX2xvZ2dpbmdCdWNrZXRJZCwgbG9nZ2luZ0J1Y2tldFByb3BzKTtcbiAgfVxuICAvLyBBdHRhY2ggdGhlIERlZmF1bHQgTGlmZSBDeWNsZSBwb2xpY3kgT05MWSBJRiB0aGUgdmVyc2lvbmluZyBpcyBFTkFCTEVEXG4gIGlmIChwcm9wcy5idWNrZXRQcm9wcz8udmVyc2lvbmVkID09PSB1bmRlZmluZWQgfHwgcHJvcHMuYnVja2V0UHJvcHMudmVyc2lvbmVkKSB7XG4gICAgY3VzdG9tQnVja2V0UHJvcHMgPSBEZWZhdWx0UzNQcm9wcyhsb2dnaW5nQnVja2V0LCBsaWZlY3ljbGVSdWxlcyk7XG4gIH0gZWxzZSB7XG4gICAgY3VzdG9tQnVja2V0UHJvcHMgPSBEZWZhdWx0UzNQcm9wcyhsb2dnaW5nQnVja2V0KTtcbiAgfVxuXG4gIGN1c3RvbUJ1Y2tldFByb3BzID0gcHJvcHMuYnVja2V0UHJvcHMgPyBvdmVycmlkZVByb3BzKGN1c3RvbUJ1Y2tldFByb3BzLCBwcm9wcy5idWNrZXRQcm9wcykgOiBjdXN0b21CdWNrZXRQcm9wcztcblxuICBjb25zdCBzM0J1Y2tldDogczMuQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzY29wZSwgX2J1Y2tldElkLCBjdXN0b21CdWNrZXRQcm9wcyk7XG5cbiAgYXBwbHlTZWN1cmVCdWNrZXRQb2xpY3koczNCdWNrZXQpO1xuXG4gIHJldHVybiBbczNCdWNrZXQsIGxvZ2dpbmdCdWNrZXRdO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYWRkQ2ZuTmFnUzNCdWNrZXROb3RpZmljYXRpb25SdWxlc1RvU3VwcHJlc3Moc3RhY2tSb290OiBjZGsuU3RhY2ssIGxvZ2ljYWxJZDogc3RyaW5nKSB7XG4gIGNvbnN0IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXIgPSBzdGFja1Jvb3Qubm9kZS50cnlGaW5kQ2hpbGQobG9naWNhbElkKSBhcyBsYW1iZGEuRnVuY3Rpb247XG4gIGNvbnN0IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXJSb2xlUm9sZSA9IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXIubm9kZS5maW5kQ2hpbGQoJ1JvbGUnKSBhcyBpYW0uUm9sZTtcbiAgY29uc3Qgbm90aWZpY2F0aW9uc1Jlc291cmNlSGFuZGxlclJvbGVQb2xpY3kgPSBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyUm9sZVJvbGUubm9kZS5maW5kQ2hpbGQoJ0RlZmF1bHRQb2xpY3knKSBhcyBpYW0uUG9saWN5O1xuXG4gIC8vIEV4dHJhY3QgdGhlIENmbkZ1bmN0aW9uIGZyb20gdGhlIEZ1bmN0aW9uXG4gIGNvbnN0IGZuUmVzb3VyY2UgPSBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyLm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGxhbWJkYS5DZm5GdW5jdGlvbjtcbiAgYWRkQ2ZuU3VwcHJlc3NSdWxlcyhmblJlc291cmNlLCBbXG4gICAge1xuICAgICAgaWQ6ICdXNTgnLFxuICAgICAgcmVhc29uOiBgTGFtYmRhIGZ1bmN0aW9ucyBoYXMgdGhlIHJlcXVpcmVkIHBlcm1pc3Npb24gdG8gd3JpdGUgQ2xvdWRXYXRjaCBMb2dzLiBJdCB1c2VzIGN1c3RvbSBwb2xpY3kgaW5zdGVhZCBvZiBhcm46YXdzOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIHdpdGggdGlnaHRlciBwZXJtaXNzaW9ucy5gXG4gICAgfSxcbiAgICB7XG4gICAgICBpZDogJ1c4OScsXG4gICAgICByZWFzb246IGBUaGlzIGlzIG5vdCBhIHJ1bGUgZm9yIHRoZSBnZW5lcmFsIGNhc2UsIGp1c3QgZm9yIHNwZWNpZmljIHVzZSBjYXNlcy9pbmR1c3RyaWVzYFxuICAgIH0sXG4gICAge1xuICAgICAgaWQ6ICdXOTInLFxuICAgICAgcmVhc29uOiBgSW1wb3NzaWJsZSBmb3IgdXMgdG8gZGVmaW5lIHRoZSBjb3JyZWN0IGNvbmN1cnJlbmN5IGZvciBjbGllbnRzYFxuICAgIH1cbiAgXSk7XG5cbiAgLy8gRXh0cmFjdCB0aGUgQ2ZuUG9saWN5IGZyb20gdGhlIGlhbS5Qb2xpY3lcbiAgY29uc3QgcG9saWN5UmVzb3VyY2UgPSBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyUm9sZVBvbGljeS5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBpYW0uQ2ZuUG9saWN5O1xuICBhZGRDZm5TdXBwcmVzc1J1bGVzKHBvbGljeVJlc291cmNlLCBbXG4gICAge1xuICAgICAgaWQ6ICdXMTInLFxuICAgICAgcmVhc29uOiBgQnVja2V0IHJlc291cmNlIGlzICcqJyBkdWUgdG8gY2lyY3VsYXIgZGVwZW5kZW5jeSB3aXRoIGJ1Y2tldCBhbmQgcm9sZSBjcmVhdGlvbiBhdCB0aGUgc2FtZSB0aW1lYFxuICAgIH1cbiAgXSk7XG59XG4iXX0=