"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            if (props.glueJobProps.glueVersion === '2.0' && props.glueJobProps.maxCapacity) {
                throw Error('Cannot set "MaxCapacity" with GlueVersion 2.0 or higher. Use "NumberOfWorkers" and "WorkerType". ' +
                    'Refer the API documentation https://docs.aws.amazon.com/glue/latest/webapi/API_Job.html for more details');
            }
            if (props.glueJobProps.maxCapacity && (props.glueJobProps.numberOfWorkers || props.glueJobProps.workerType)) {
                throw Error('Cannot set MaxCapacity and "WorkerType" or  "NumberOfWorkers". If using glueVersion 2.0 or beyond, ' +
                    'it is recommended to use "WorkerType" or  "NumberOfWorkers"');
            }
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${core_1.Aws.PARTITION}:kms:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps, _glueSecurityConfigName, _jobArgumentsList), glueJobProps);
    let _scriptLocation;
    if (isJobCommandProperty(_newGlueJobProps.command)) {
        if (_newGlueJobProps.command.scriptLocation) {
            _scriptLocation = _newGlueJobProps.command.scriptLocation;
        }
        else {
            throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
        }
    }
    const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
    _scriptBucketLocation.grantRead(_jobRole);
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole, _outputLocation];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    const splitString = s3Url.slice('s3://'.length);
    return `arn:${core_1.Aws.PARTITION}:s3:::${splitString}`;
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name ||
        command.pythonVersion ||
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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