"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupCognitoForElasticSearch = exports.buildIdentityPool = exports.buildUserPoolClient = exports.buildUserPool = void 0;
const cognito = require("@aws-cdk/aws-cognito");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cognito_defaults_1 = require("./cognito-defaults");
function buildUserPool(scope, userPoolProps) {
    let cognitoUserPoolProps;
    if (userPoolProps) {
        cognitoUserPoolProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolProps, userPoolProps);
    }
    else {
        cognitoUserPoolProps = cognito_defaults_1.DefaultUserPoolProps;
    }
    const userPool = new cognito.UserPool(scope, 'CognitoUserPool', cognitoUserPoolProps);
    // Set the advancedSecurityMode to ENFORCED
    const cfnUserPool = userPool.node.findChild('Resource');
    cfnUserPool.userPoolAddOns = {
        advancedSecurityMode: 'ENFORCED'
    };
    // Add Cfn Nag suppress for the cognito SMS role policy
    const userPoolSmsRole = userPool.node.tryFindChild('smsRole');
    if (userPoolSmsRole) {
        utils_1.addCfnSuppressRules(userPool, [
            {
                id: 'W11',
                reason: `Allowing * resource on permissions policy since its used by Cognito to send SMS messages via sns:Publish`
            }
        ]);
    }
    return userPool;
}
exports.buildUserPool = buildUserPool;
function buildUserPoolClient(scope, userPool, cognitoUserPoolClientProps) {
    let userPoolClientProps;
    if (cognitoUserPoolClientProps) {
        userPoolClientProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolClientProps(userPool), cognitoUserPoolClientProps);
    }
    else {
        userPoolClientProps = cognito_defaults_1.DefaultUserPoolClientProps(userPool);
    }
    return new cognito.UserPoolClient(scope, 'CognitoUserPoolClient', userPoolClientProps);
}
exports.buildUserPoolClient = buildUserPoolClient;
function buildIdentityPool(scope, userpool, userpoolclient, identityPoolProps) {
    let cognitoIdentityPoolProps = cognito_defaults_1.DefaultIdentityPoolProps(userpoolclient.userPoolClientId, userpool.userPoolProviderName);
    if (identityPoolProps) {
        cognitoIdentityPoolProps = utils_1.overrideProps(cognitoIdentityPoolProps, identityPoolProps);
    }
    const idPool = new cognito.CfnIdentityPool(scope, 'CognitoIdentityPool', cognitoIdentityPoolProps);
    return idPool;
}
exports.buildIdentityPool = buildIdentityPool;
function setupCognitoForElasticSearch(scope, domainName, options) {
    // Create the domain for Cognito UserPool
    const userpooldomain = new cognito.CfnUserPoolDomain(scope, 'UserPoolDomain', {
        domain: domainName,
        userPoolId: options.userpool.userPoolId
    });
    userpooldomain.addDependsOn(options.userpool.node.findChild('Resource'));
    // Setup the IAM Role for Cognito Authorized Users
    const cognitoPrincipal = new iam.FederatedPrincipal('cognito-identity.amazonaws.com', {
        'StringEquals': { 'cognito-identity.amazonaws.com:aud': options.identitypool.ref },
        'ForAnyValue:StringLike': { 'cognito-identity.amazonaws.com:amr': 'authenticated' }
    }, 'sts:AssumeRoleWithWebIdentity');
    const cognitoAuthorizedRole = new iam.Role(scope, 'CognitoAuthorizedRole', {
        assumedBy: cognitoPrincipal,
        inlinePolicies: {
            CognitoAccessPolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'es:ESHttp*'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`]
                    })
                ]
            })
        }
    });
    // Attach the IAM Role for Cognito Authorized Users
    new cognito.CfnIdentityPoolRoleAttachment(scope, 'IdentityPoolRoleMapping', {
        identityPoolId: options.identitypool.ref,
        roles: {
            authenticated: cognitoAuthorizedRole.roleArn
        }
    });
    return cognitoAuthorizedRole;
}
exports.setupCognitoForElasticSearch = setupCognitoForElasticSearch;
//# sourceMappingURL=data:application/json;base64,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