"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetActiveListener = exports.AddLambdaTarget = exports.AddListener = exports.ObtainAlb = void 0;
const elb = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const elbt = require("@aws-cdk/aws-elasticloadbalancingv2-targets");
const utils_1 = require("./utils");
const alb_defaults_1 = require("./alb-defaults");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
//  Returns the correct ALB Load Balancer to use in this construct, either an existing
//  one provided as an argument or create  new one otherwise.
function ObtainAlb(scope, id, vpc, publicApi, existingLoadBalancerObj, loadBalancerProps, logAccessLogs, loggingBucketProps) {
    let loadBalancer;
    if (existingLoadBalancerObj) {
        loadBalancer = existingLoadBalancerObj;
    }
    else {
        const consolidatedProps = loadBalancerProps
            ? utils_1.overrideProps(loadBalancerProps, { vpc, internetFacing: publicApi })
            : { vpc, internetFacing: publicApi };
        loadBalancer = new elb.ApplicationLoadBalancer(scope, `${id}-alb`, consolidatedProps);
        if (logAccessLogs === undefined || logAccessLogs === true) {
            const consolidatedLoggingBucketProps = loggingBucketProps
                ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultLoggingBucketProps(), loggingBucketProps)
                : s3_bucket_defaults_1.DefaultLoggingBucketProps();
            const loggingBucket = s3_bucket_helper_1.createAlbLoggingBucket(scope, id, consolidatedLoggingBucketProps);
            loadBalancer.logAccessLogs(loggingBucket);
        }
    }
    return loadBalancer;
}
exports.ObtainAlb = ObtainAlb;
function AddListener(scope, loadBalancer, listenerProps) {
    var _a;
    let consolidatedListenerProps;
    consolidatedListenerProps = utils_1.overrideProps(alb_defaults_1.DefaultListenerProps(loadBalancer), listenerProps);
    //   create the listener
    const listener = new elb.ApplicationListener(scope, "listener", consolidatedListenerProps);
    loadBalancer.listeners.push(listener);
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTP) {
        // This will use core.printWarning in the actual construct
        utils_1.printWarning("AWS recommends encrypting traffic to an Application Load Balancer using HTTPS.");
        if (((_a = listenerProps.certificates) === null || _a === void 0 ? void 0 : _a.length) > 0) {
            throw new Error("HTTP listeners cannot use a certificate");
        }
    }
    else {
        if (!listenerProps.certificates || listenerProps.certificates.length === 0) {
            throw new Error("A listener using HTTPS protocol requires a certificate");
        }
        listener.addCertificates("listener-cert-add", listenerProps.certificates);
    }
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTPS) {
        const opt = {
            port: "443",
            protocol: "HTTPS",
        };
        const httpListener = new elb.ApplicationListener(scope, "redirect-listener", {
            loadBalancer,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            defaultAction: aws_elasticloadbalancingv2_1.ListenerAction.redirect(opt),
        });
        loadBalancer.listeners.push(httpListener);
    }
    return listener;
}
exports.AddListener = AddListener;
// Creates a Target Group for Lambda functions and adds the
// provided functions as a target to that group. Then adds
// the new Target Group to the provided Listener. The expectaion
// is that Lambda specific code is included here, and next we will
// add AddFargateTarget(), with Fargate specific code isolated in that
// function.
function AddLambdaTarget(scope, id, currentListener, lambdaFunction, ruleProps, targetProps) {
    //  Create the target and assign it to a new target group
    const lambdaTarget = new elbt.LambdaTarget(lambdaFunction);
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, {
        targets: [lambdaTarget],
        targetGroupName: targetProps ? targetProps.targetGroupName : undefined,
        healthCheck: targetProps ? targetProps.healthCheck : undefined
    });
    // AddRuleProps includes conditions and priority, combine that with targetGroups and
    // we can assemble AddApplicationTargetGroupProps
    if (ruleProps) {
        const consolidatedTargetProps = utils_1.overrideProps(ruleProps, { targetGroups: [newTargetGroup] });
        currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    }
    else {
        currentListener.addTargetGroups("targets", {
            targetGroups: [newTargetGroup],
        });
    }
    newTargetGroup.setAttribute('stickiness.enabled', undefined);
    return newTargetGroup;
}
exports.AddLambdaTarget = AddLambdaTarget;
// Looks for the listener associated with Target Groups
// If there is a single listener, this returns it whether it is HTTP or HTTPS
// If there are 2 listeners, it finds the HTTPS listener (we assume the HTTP listener redirects to HTTPS)
function GetActiveListener(listeners) {
    let listener;
    if (listeners.length === 0) {
        throw new Error(`There are no listeners in the ALB`);
    }
    if (listeners.length === 1) {
        listener = listeners[0];
    }
    else {
        listener = listeners.find(i => i.node.children[0].protocol === "HTTPS");
    }
    return listener;
}
exports.GetActiveListener = GetActiveListener;
//# sourceMappingURL=data:application/json;base64,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