# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cloud_init_gen']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0,<7.0']

setup_kwargs = {
    'name': 'cloud-init-gen',
    'version': '1.1.0',
    'description': 'Simplified generation of cloud-init user-data content',
    'long_description': 'cloud-init-gen: Simplified generation of cloud-init user-data documents\n=================================================\n\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n[![Latest release](https://img.shields.io/github/v/release/sammck/cloud-init-gen.svg?style=flat-square&color=b44e88)](https://github.com/sammck/cloud-init-gen/releases)\n\nAn API for simplified generation of [cloud-init](https://cloudinit.readthedocs.io/en/latest/) user-data documents.\n\nTable of contents\n-----------------\n\n* [Introduction](#introduction)\n* [Installation](#installation)\n* [Usage](#usage)\n  * [API](api)\n* [Known issues and limitations](#known-issues-and-limitations)\n* [Getting help](#getting-help)\n* [Contributing](#contributing)\n* [License](#license)\n* [Authors and history](#authors-and-history)\n\n\nIntroduction\n------------\n\n[cloud-init](https://cloudinit.readthedocs.io/en/latest/) is the industry standard multi-distribution method for cross-platform cloud instance initialization. It is supported across all major public cloud providers (AWS EC2, Azure, Google Cloud, etc.), provisioning systems for private cloud infrastructure, and bare-metal installations.\n\nWhen a cloud IAAS user creates a VM instance in the cloud (e.g., an EC2 instance), they can optionally provide a _user-data_ document.\nThis document is passed to [cloud-init](https://cloudinit.readthedocs.io/en/latest/) at VM boot time, which uses the document to,\namong other things, configure the VM for first-time use by installing packages, creating OS users, running custom scripts, etc.\n\nPython package `cloud-init-gen` provides a simple API for generation of a well-formed cloud-init _user-data_ document, and rendering the\ndocument into a format expected by cloud service providers (typically base-64).\n\nSome key features of cloud-init-gen:\n\n* Type inference from #-comment and MIME-style headers, as well as explicit MIME-type.\n* Automatic re-encoding with #-comment or MIME Conent-Type, minimizing rendered document size.\n* Transparent support for multipart cloud-init _user-data_ documents, composed one part at a time.\n* Automatic conversion from JSON-friendly structured data to YAML, typical for _cloud-config_ document parts.\n* Support for custom MIME types and additional MIME headers.\n* Automatic compression with GZIP if necessary to keep rendered binary under 16KB (the maximum _user-data_ size).\n* Rendering to string, binary, or binary encoded as a base-64 string.\n\nInstallation\n------------\n\n### Prerequisites\n\n**Python**: Python 3.7+ is required. See your OS documentation for instructions.\n\n### From PyPi\n\nThe current released version of `cloud-init-gen` can be installed with \n\n```bash\npip3 install cloud-init-gen\n```\n\n### From GitHub\n\n[Poetry](https://python-poetry.org/docs/master/#installing-with-the-official-installer) is required; it can be installed with:\n\n```bash\ncurl -sSL https://install.python-poetry.org | python3 -\n```\n\nClone the repository and install cloud-init-gen into a private virtualenv with:\n\n```bash\ncd <parent-folder>\ngit clone https://github.com/sammck/cloud-init-gen.git\ncd cloud-init-gen\npoetry install\n```\n\nYou can then launch a bash shell with the virtualenv activated using:\n\n```bash\npoetry shell\n```\n\n\nUsage\n=====\n\n\nExample\n-------\n\nLet\'s say you want to use [boto3](https://pypi.org/project/boto3/) to create an EC2 instance on\nAWS, and you want to do a few things with cloud-init\n\n  1. On every boot, before doing anything else, you want to clear `/var/log/per-boot.log` if it exists, and write the\n     boot time to the newly cleared file\n  2. On the first boot, you want to install a few Debian/Ubuntu packages, and you want to install the latest version of Docker.\n  3. You want to set up docker for authentication to AWS ECR in the same region\n\n```python\nimport shlex\nimport json\nimport boto3\nfrom cloud_init_gen import CloudInitDoc\n\nsess = boto3.session.Session()\naws_region = sess.region_name\n\nsts = sess.client(\'sts\')\nresp = sts.get_caller_identity()\naws_account_id = resp[\'Account\']\n\nuser_data = CloudInitDoc()\n\nboot_hook = \'\'\'#boothook\n#!/bin/bash\necho "Booted on $(date)" > /var/log/per-boot.log\n\'\'\'\n\nuser_data.add(boot_hook)  # MIME type text/cloud-boothook is infered from the "#boothook" header.\n\necr_domain: str = f"{aws_account_id}.dkr.ecr.{aws_region}.amazonaws.com"\ndocker_config_obj = {\n    "credHelpers": {\n        "public.ecr.aws": "ecr-login",\n        ecr_domain: "ecr-login"\n      }\n  }\ndocker_config = json.dumps(docker_config_obj, separators=(\',\', \':\'), sort_keys=True)\n\ncloud_cfg = dict(\n    repo_update = True,\n    repo_upgrade = "all",\n    apt = dict(\n        sources = {\n          "docker.list": dict(\n              source = "deb [arch=amd64] https://download.docker.com/linux/ubuntu $RELEASE stable",\n              keyid = "9DC858229FC7DD38854AE2D88D81803C0EBFCD88"\n            ),\n          },\n      ),\n\n    packages = [\n        "jq",\n        "awscli",\n        "collectd",\n        "ca-certificates",\n        "curl",\n        "gnupg",\n        "lsb-release",\n        "docker-ce",\n        "docker-ce-cli",\n        "amazon-ecr-credential-helper",\n      ],\n\n    runcmd = [\n        [ "bash", "-c", f"mkdir -p /root/.docker && chmod 700 /root/.docker && echo {shlex.quote(docker_config)} > /root/.docker/config.json && chmod 600 /root/.docker/config.json" ],\n        [ "bash", "-c", \'echo "it works!"\' ],\n      ],\n  )\n\nuser_data.add(cloud_cfg)  # will be rendered as yaml with implicit MIME type text/cloud-config\n\nprint(f"Final user-data (text):\\n====================\\n{user_data.render()}\\n====================")\nprint(f"Final user-data (base64):\\n====================\\n{user_data.render_base64()}\\n====================")\n\nec2 = sess.client(\'ec2\')\nresp = ec2.run_instances(\n    ...\n    UserData=user_data.render_base64()   # boto3/EC2 expect user-data to be encoded with base-64\n    ...\n  )\n```\n\nKnown issues and limitations\n----------------------------\n\n* Has not been tested for compatibility against cloud service providers other than AWS EC2.\n\nGetting help\n------------\n\nPlease report any problems/issues [here](https://github.com/sammck/cloud-init-gen/issues).\n\nContributing\n------------\n\nPull requests welcome.\n\nLicense\n-------\n\ncloud-init-gen is distributed under the terms of the [MIT License](https://opensource.org/licenses/MIT).  The license applies to this file and other files in the [GitHub repository](http://github.com/sammck/cloud-init-gen) hosting this file.\n\nAuthors and history\n---------------------------\n\nThe author of cloud-init-gen is [Sam McKelvie](https://github.com/sammck).\n',
    'author': 'Sam McKelvie',
    'author_email': 'dev@mckelvie.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/sammck/cloud-init-gen',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
