# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['darglint',
 'darglint.analysis',
 'darglint.docstring',
 'darglint.parse',
 'darglint.parse.grammars']

package_data = \
{'': ['*']}

entry_points = \
{'console_scripts': ['darglint = darglint.driver:main'],
 'flake8.extension': ['DAR = darglint.flake8_entry:DarglintChecker']}

setup_kwargs = {
    'name': 'darglint',
    'version': '1.5.4',
    'description': 'A utility for ensuring Google-style docstrings stay up to date with the source code.',
    'long_description': '[![Build Status](https://travis-ci.com/terrencepreilly/darglint.svg?branch=develop)](https://travis-ci.com/terrencepreilly/darglint)\n\n# Darglint\n\nA functional docstring linter which checks whether a docstring\'s\ndescription matches the actual function/method implementation.\n*Darglint* expects docstrings to be formatted using the\n[Google Python Style Guide](https://google.github.io/styleguide/pyguide.html),\nor [Sphinx Style Guide](https://pythonhosted.org/an_example_pypi_project/sphinx.html#function-definitions),\nor [Numpy Style Guide](https://numpydoc.readthedocs.io/en/latest/format.html).\n\nFeel free to submit an issue/pull request if you spot a problem or\nwould like a feature in *darglint*.\n\n**Table of Contents**:\n\n- [Installation](#installation)\n- [Configuration](#configuration)\n- [Usage](#usage)\n- [Scope](#scope)\n- [Sphinx](#sphinx)\n- [Numpy](#numpy)\n- [Integrations](#integrations)\n- [Flake8](#flake8)\n- [Roadmap](#roadmap)\n- [Contribution](#development-and-contributions)\n\n\n## Installation\n\nTo install *darglint*, use pip.\n\n```bash\npip install darglint\n```\n\nOr, clone the repository, `cd` to the directory, and\n\n```bash\npip install .\n```\n\n## Configuration\n\n*darglint* can be configured using a configuration file.  The configuration\nfile must be named either *.darglint*, *setup.cfg*, or *tox.ini*.  It must\nalso have a section starting with the section header, `[darglint]`.\nFinally, the configuration file must be located either in the directory\n*darglint* is called from, or from a parent directory of that working\ndirectory.\n\nCurrently, the configuration file allows us to ignore errors, to specify\nmessage templates, and to specify the strictness of checks.\n\n### Error Configuration\n\nIf we would like to ignore `ExcessRaiseError`s (because we know that\nan underlying function will raise an exception), then we would add its\nerror code to a file named *.darglint*:\n\n```ini\n[darglint]\nignore=DAR402\n```\n\nWe can ignore multiple errors by using a comma-separated list:\n\n```ini\n[darglint]\nignore=DAR402,DAR103\n```\n\nInstead of specifying error codes to ignore in general one can also specify a\nregex to exclude certain function names from tests. For example, the following \nconfiguration would disable linting on all private methods.\n```ini\n[darglint]\nignore_regex=^_(.*)\n```\n\n### Message Template Configuration\n\nIf we would like to specify a message template, we may do so as\nfollows:\n\n```ini\n[darglint]\nmessage_template={msg_id}@{path}:{line}\n```\n\nWhich will produce a message such as `DAR102@driver.py:72`.\n\nFinally, we can specify the docstring style type using `docstring_style`\n("google" by default):\n\n```ini\n[darglint]\ndocstring_style=sphinx\n```\n\n### Strictness Configuration\n\nStrictness determines how lax darglint will be when checking docstrings.\nThere are three levels of strictness available:\n\n- short: One-line descriptions are acceptable; anything\nmore and the docstring will be fully checked.\n\n- long: One-line descriptions and descriptions without\narguments/returns/yields/etc. sections will be allowed.  Anything more,\nand the docstring will be fully checked.\n\n- full: (Default) Docstrings will be fully checked.\n\nFor example, if we have the following function:\n\n```python\ndef double(x):\n    # <docstring>\n    return x * 2\n```\n\nThen the following table describes which errors will be raised for\neach of the docstrings (rows) when checked against each of the\nconfigurations (columns):\n\n```\n┌──────────────────────────────┬──────────────────┬────────────────┬──────────────────┐\n│ Docstring                    │  short           │  long          │  full            │\n├──────────────────────────────┼──────────────────┼────────────────┼──────────────────┤\n│ """Doubles the argument."""  │ None             │ None           │ Missing argument │\n│                              │                  │                │ Missing return   │\n│                              │                  │                │                  │\n│                              │                  │                │                  │\n├──────────────────────────────┼──────────────────┼────────────────┼──────────────────┤\n│ """Doubles the argument.     │ Missing argument │ None           │ Missing argument │\n│                              │ Missing return   │                │ Missing return   │\n│ Not very pythonic.           │                  │                │                  │\n│                              │                  │                │                  │\n│ """                          │                  │                │                  │\n│                              │                  │                │                  │\n├──────────────────────────────┼──────────────────┼────────────────┼──────────────────┤\n│ """Doubles the argument.     │ Missing return   │ Missing return │ Missing return   │\n│                              │                  │                │                  │\n│ Args:                        │                  │                │                  │\n│     x: The number to double. │                  │                │                  │\n│                              │                  │                │                  │\n│ """                          │                  │                │                  │\n└──────────────────────────────┴──────────────────┴────────────────┴──────────────────┘\n```\n\nIn short, if you want to be able to have single-line docstrings, and check\nall other docstrings against their described parameters, you would specify\n\n```ini\n[darglint]\nstrictness=short\n```\n\nIn your configuration file.\n\n### Logging\n\nWhen *darglint* fails unexpectedly, you can try to gather more\ninformation when submitting a bug by running with logging.\nFor example,\n\n```bash\ndarglint --log-level=INFO unexpected_failures.py\n```\n\n*Darglint* accepts the levels, `DEBUG`, `INFO`, `WARNING`, `ERROR`, and\n`CRITICAL`.\n\n\n## Usage\n\n\n### Command Line use\n\nGiven a python source file, `serializers.py`, you would check the docstrings\nas follows:\n\n```bash\ndarglint serializers.py\n```\n\nYou can give an optional verbosity setting to *darglint*.  For example,\n\n```bash\ndarglint -v 2 *.py\n```\n\nWould give a description of the error along with information as to this\nspecific instance.  The default verbosity is 1, which gives the filename,\nfunction name, line number, error code, and some general hints.\n\nTo use an arbitrary error format, you can pass a message template, which\nis a python format string.  For example, if we pass the message\ntemplate\n\n```bash\ndarglint -m "{path}:{line} -> {msg_id}" darglint/driver.py\n```\n\nThen we would get back error messages like\n\n```\ndarglint/driver.py :61 -> DAR101\n```\n\nThe following attributes can be passed to the format string:\n- *line*: The line number,\n- *msg*: The error message,\n- *msg_id*: The error code,\n- *obj*: The function/method name,\n- *path*: The relative file path.\n\nThe message template can also be specified in the configuration file\nas the value `message_template`.\n\n*darglint* is particularly useful when combined with the utility, `find`.\nThis allows us to check all of the files in our project at once.  For example,\nwhen eating my own dogfood (as I tend to do), I invoke *darglint* as follows:\n\n```bash\nfind . -name "*.py" | xargs darglint\n```\n\nWhere I\'m searching all files ending in ".py" recursively from the\ncurrent directory, and calling *darglint* on each one in turn.\n\n### Ignoring Errors in a Docstring\n\nYou can ignore specific errors in a particular docstring.  The syntax\nis much like that of *pycodestyle*, etc.  It generally takes the from\nof:\n\n```python\n# noqa: <error> <argument>\n```\n\nWhere `<error>` is the particular error to ignore (`DAR402`, or `DAR201`\nfor example), and `<argument>` is what (if anything) the ignore\nstatement refers to (if nothing, then it is not specified).\n\nLet us say that we want to ignore a missing return statement\nin the following docstring:\n\n```python\ndef we_dont_want_a_returns_section():\n  """Return the value, 3.\n\n  # noqa: DAR201\n\n  """\n  return 3\n```\n\nWe put the `noqa` anywhere in the top level of the docstring.\nHowever, this won\'t work if we are missing something more specific,\nlike a parameter.  We may not want to ignore all missing parameters,\neither, just one particular one.  For example, we may be writing a\nfunction that takes a class instance as self. (Say, in a bound *celery*\ntask.) Then we would do something like:\n\n```python\ndef a_bound_function(self, arg1):\n  """Do something interesting.\n\n  Args:\n    arg1: The first argument.\n\n  # noqa: DAR101 arg1\n\n  """\n  arg1.execute(self)\n```\n\nSo, the argument comes to the right of the error.\n\nWe may also want to mark excess documentation as being okay.  For example,\nwe may not want to explicitly catch and raise a `ZeroDivisionError`.  We\ncould do the following:\n\n```python\ndef always_raises_exception(x):\n    """Raise a zero division error or type error.o\n\n    Args:\n      x: The argument which could be a number or could not be.\n\n    Raises:\n      ZeroDivisionError: If x is a number.  # noqa: DAR402\n      TypeError: If x is not a number.  # noqa: DAR402\n\n    """\n    x / 0\n```\n\nSo, in this case, the argument for `noqa` is really all the way to\nthe left.  (Or whatever description we are parsing.)  We could also\nhave put it on its own line, as `# noqa: DAR402 ZeroDivisionError`.\n\n### Type Annotations\n\nDarglint parses type annotations in docstrings, and can, optionally,\ncompare the documented type to the actual type annotation.  This can\nbe useful when migrating a codebase to use type annotations.\n\nIn order to make these comparisons, Darglint only accepts types\naccepted by Python (see [PEP 484](https://www.python.org/dev/peps/pep-0484/).)\nThat is, it does not accept parentheses in type signatures. (If\nparentheses are used in the type signature, Darglint will mark that\nargument as missing.  See Issue #90.)\n\n\n### Error Codes\n\n- *DAR001*: The docstring was not parsed correctly due to a syntax error.\n- *DAR002*: An argument/exception lacks a description\n- *DAR003*: A line is under-indented or over-indented.\n- *DAR004*: The docstring contains an extra newline where it shouldn\'t.\n- *DAR101*: The docstring is missing a parameter in the definition.\n- *DAR102*: The docstring contains a parameter not in function.\n- *DAR103*: The docstring parameter type doesn\'t match function.\n- *DAR104*: (disabled) The docstring parameter has no type specified \n- *DAR201*: The docstring is missing a return from definition.\n- *DAR202*: The docstring has a return not in definition.\n- *DAR203*: The docstring parameter type doesn\'t match function.\n- *DAR301*: The docstring is missing a yield present in definition.\n- *DAR302*: The docstring has a yield not in definition.\n- *DAR401*: The docstring is missing an exception raised.\n- *DAR402*: The docstring describes an exception not explicitly raised.\n- *DAR501*: The docstring describes a variable which is not defined.\n\nThe number in the hundreds narrows the error by location in the docstring:\n\n- 000: Syntax, formatting, and style\n- 100: Args section\n- 200: Returns section\n- 300: Yields section\n- 400: Raises section\n- 500: Variables section\n\nYou can enable disabled-by-default exceptions in the configuration file\nusing the `enable` option.  It accepts a comma-separated list of error\ncodes.\n\n```ini\n[darglint]\nenable=DAR104\n```\n\n## Scope\n\nDarglint\'s primary focus is to identify incorrect and missing documentationd\nof a function\'s signature. Checking style is a stretch goal, and is supported\non a best-effort basis.  Darglint does not check stylistic preferences expressed\nby tools in the Python Code Quality Authority (through tools such as `pydocstyle`).\nSo when using Darglint, it may be a good idea to also use `pydocstyle`, if you\nwant to enforce style.  (For example, `pydocstyle` requires the short summary\nto be separated from other sections by a line break.  Darglint makes no such check.)\n\n## Sphinx\n\nDarglint can handle sphinx-style docstrings, but imposes some restrictions\non top of the Sphinx style. For example, all fields (such as `:returns:`)\nmust be the last items in the docstring.  They must be together, and all\nindents should be four spaces.  These restrictions may be loosened at a\nlater date.\n\nTo analyze Sphinx-style docstrings, pass the style flag to the command:\n\n```bash\ndarglint -s sphinx example.py\ndarglint --docstring-style sphinx example.py\n```\n\nAlternatively, you can specify the style in the configuration file using\nthe setting, "docstring\\_style":\n\n```ini\n[darglint]\ndocstring_style=sphinx\n```\n\n## Numpy\n\nDarglint now has an initial implementation for Numpy-style docstrings.\nSimilarly to Sphinx-style docstrings, you can pass a style flag to the\ncommand:\n\n```bash\ndarglint -s numpy example.py\ndarglint --docstring-style numpy example.py\n```\n\nOr set it in a configuration file:\n\n```init\n[darglint]\ndocstring_style=numpy\n```\n\nThe numpy parser and error reporter are not yet fully stabilized.\nAdd issues or suggestions to the tracking bug, Issue #69.\n\n## Integrations\n\n### Flake8\n\nDarglint can be used in conjunction with Flake8 as a plugin.  The only\nsetup necessary is to install Flake8 and Darglint in the same environment.\nDarglint will pull its configuration from Flake8. So, if you would like to\nlint Sphinx-style comments, then you should have `docstring_style=sphinx` in a\nFlake8 configuration file in the project directory.  The settings would\nbe entered under the flake8 configuration, not a separate configuration\nfor Darglint.  E.g.:\n\n```init\n[flake8]\nstrictness=short\ndocstring_style=sphinx\n```\n\nTo see which options are exposed through Flake8, you can check the Flake8\ntool:\n\n```bash\nflake8 --help | grep --before-context=2 Darglint\n```\n\n### SublimeLinter\n\nA plugin for SublimeLinter can be found [here](https://github.com/raddessi/SublimeLinter-contrib-darglint)\n\n### Pre-commit\n\nDownload [pre-commit](https://pre-commit.com/) and\n[install](https://pre-commit.com/#install) it. Once it is installed, add this\nto `.pre-commit-config.yaml` in your repository:\n\n```yaml\nrepos:\n-   repo: https://github.com/terrencepreilly/darglint\n    rev: master\n    hooks:\n    - id: darglint\n```\n\nThen run `pre-commit install` and you\'re ready to go. Before commiting,\n`darglint` will be run on the staged files. If it finds any errors, the user\nis notified and the commit is aborted. Store necessary configuration (such as\nerror formatting) in `.darglint`, `config.cfg` or `tox.ini`.\n\n\n## Roadmap\n\nBelow are some of the current features or efforts.  Where a milestone or\nissue is associated with the idea, it will be mentioned.  Some of these\nideas are moonshots and may not get implemented.  They are ordered\nroughly according to current priority/feasibility.\n\n- [ ] Expose command-line options through sphinx.\n- [ ] Robust logging for errors caused/encountered by *darglint*.\n- [ ] Check class docstrings (See Issue #25).\n- [ ] Autoformatting docstrings.  (See Milestone #3).\n- [ ] Optional aggressive style checking through command line flag.\n- [ ] ALE support.\n- [ ] Syntastic support. (Syntastic is not accepting new checkers until\ntheir next API stabilizes, so this may take some time.)\n\n\n## Development and Contributions\n\n### Development Setup\n\nInstall `darglint`. First, clone the repository:\n\n```bash\ngit clone https://github.com/terrencepreilly/darglint.git\n```\n\n`cd` into the directory, create a virtual environment (optional), then setup:\n\n```bash\ncd darglint/\nvirtualenv -p python3.6 .env\nsource .env/bin/activate\npip install -e .\n```\n\nYou can run the tests using\n\n```bash\npython setup.py test\n```\n\nOr, install `pytest` manually, `cd` to the project\'s root directory,\nand run\n\n```bash\npytest\n```\n\nThis project tries to conform by the styles imposed by `pycodestyle`\nand `pydocstyle`, as well as by `darglint` itself.\n\n\nA dockerfile exists for testing with Python3.4.  Although it\'s not\nofficially supported (only 3.5+), it\'s nice to try to make minor\nversion numbers support it.  You would build the dockerfile and\ntest using something like\n\n```bash\npushd docker-build\ndocker build -t darglint-34 -f Dockerfile.test34 .\npopd\ndocker run -it --rm -v $(pwd):/code darglint-34 pytest\n```\n\n### Contribution\n\nIf you would like to tackle an issue or feature, email me or comment on the\nissue to make sure it isn\'t already being worked on.  Contributions will\nbe accepted through pull requests.  New features should include unit tests,\nand, of course, properly formatted documentation.\n\nAlso, check out the wiki prior to updating the grammar.  It includes a\ndescription of darglint\'s parsing pipline.\n',
    'author': 'terrencepreilly',
    'author_email': 'terrencepreilly@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'entry_points': entry_points,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
