import numpy as np
import scipy.linalg as linalg
import scipy.stats as stats
from geomstats.learning.frechet_mean import FrechetMean
from geomstats.geometry.euclidean import Euclidean
from sklearn.utils import check_array
from .base import BaseThresholder
from .thresh_utility import normalize, cut, gen_kde


class KARCH(BaseThresholder):
    """KARCH class for Riemannian Center of Mass thresholder.

       Use the Karcher mean (Riemannian Center of Mass) to evaluate a
       non-parametric means to threshold scores generated by the
       decision_scores where outliers are set to any value beyond the
       Karcher mean + one standard deviation of the decision_scores
       
       Paramaters
       ----------

       ndim : int, optional (default=2)

       method : str, optional (default='complex')
       {'simple', 'complex'}

       Attributes
       ----------

       eval_: numpy array of binary labels of the training data. 0 stands
           for inliers and 1 for outliers/anomalies.

    """

    def __init__(self, ndim=2, method='complex'):

        self.ndim = ndim
        self.method = method

    def eval(self, decision):
        """Outlier/inlier evaluation process for decision scores.

        Parameters
        ----------
        decision : np.array or list of shape (n_samples)
                   which are the decision scores from a
                   outlier detection.
        
        Returns
        -------
        outlier_labels : numpy array of shape (n_samples,)
            For each observation, tells whether or not
            it should be considered as an outlier according to the
            fitted model. 0 stands for inliers and 1 for outliers.
        """

        decision = check_array(decision, ensure_2d=False)

        decision = normalize(decision)

        # Create euclidean manifold and find Karcher mean
        manifold = Euclidean(dim=self.ndim)
        estimator = FrechetMean(metric=manifold.metric)
    
        if self.method=='complex':

            # Create kde of scores
            val_data, _ = gen_kde(decision,0,1,len(decision))
            val_data = val_data.reshape(-1,1)
            val_norm = np.sort(decision).reshape(1,-1)
            
            try:
                # find kde and score dot product and solve the 
                vals = np.dot(val_data,val_norm)
                estimator.fit(vals)
                kmean = np.mean(estimator.estimate_)+np.std(decision)
    
            except ValueError:
                kmean = 1.0
        else:
            estimator.fit(decision.reshape(1,-1))
            kmean = np.mean(estimator.estimate_) + np.std(decision)


        # Get the mean of each dimension's Karcher mean 
        limit = kmean

        self.thresh_ = limit

        return cut(decision, limit)
