import numpy as np
from scipy.ndimage import gaussian_filter
from sklearn.utils import check_array
from .base import BaseThresholder
from .thresh_utility import normalize, cut


class GF(BaseThresholder):
    """GF class for Gaussian Filter thresholder.

       Use a Gaussian filter to evaluate a non-parametric means
       to threshold scores generated by the decision_scores where outliers
       are set to any value beyond the maximum value generated after filtering
       
       Paramaters
       ----------

       Attributes
       ----------

       eval_: numpy array of binary labels of the training data. 0 stands
           for inliers and 1 for outliers/anomalies.

    """

    def __init__(self):

        pass

    def eval(self, decision):
        """Outlier/inlier evaluation process for decision scores.

        Parameters
        ----------
        decision : np.array or list of shape (n_samples)
                   which are the decision scores from a
                   outlier detection.
        
        Returns
        -------
        outlier_labels : numpy array of shape (n_samples,)
            For each observation, tells whether or not
            it should be considered as an outlier according to the
            fitted model. 0 stands for inliers and 1 for outliers.
        """

        decision = check_array(decision, ensure_2d=False)

        decision = normalize(decision)

        # Apply a Gaussian filter with proportional sigma
        sigma = len(decision)*np.std(decision)
        val = gaussian_filter(decision, sigma=sigma)

        # Set limit as the max gaussian filtered value
        limit = np.max(val)

        self.thresh_ = limit

        return cut(decision, limit)
