import numpy as np
from sklearn.utils import check_array
from .base import BaseThresholder
from .thresh_utility import normalize, cut, gen_kde


class FGD(BaseThresholder):
    """FGD class for Fixed Gradient Descent thresholder.

       Use the fixed gradient descent to evaluate a non-parametric means
       to threshold scores generated by the decision_scores where outliers
       are set to any value beyond where the first derivative of the kde
       with respect to the decision scores passes the second inflection point
       
       Paramaters
       ----------

       Attributes
       ----------

       eval_: numpy array of binary labels of the training data. 0 stands
           for inliers and 1 for outliers/anomalies.

    """

    def __init__(self):

        pass

    def eval(self, decision):
        """Outlier/inlier evaluation process for decision scores.

        Parameters
        ----------
        decision : np.array or list of shape (n_samples)
                   which are the decision scores from a
                   outlier detection.
        
        Returns
        -------
        outlier_labels : numpy array of shape (n_samples,)
            For each observation, tells whether or not
            it should be considered as an outlier according to the
            fitted model. 0 stands for inliers and 1 for outliers.
        """

        decision = check_array(decision, ensure_2d=False)

        decision = normalize(decision)

        # Generate KDE
        val, dat_range = gen_kde(decision,0,1,len(decision)*3)

        # Calculate the first derivative of the KDE with respect
        # to the data range
        deriv = np.gradient(val, dat_range[1]-dat_range[0])

        count = 0    
        ind = 0

        # Set the limit where the first derivative curve passes through the
        # origin a second time. (the first time occurs at the kde max peak)
        for i in range(len(deriv)-1):

            if (deriv[i]>0)&(deriv[i+1]<=0):
                count+=1
                if count==2:
                    ind = i
                    break

        limit = dat_range[ind]
        self.thresh_ = limit

        return cut(decision, limit)
