# -*- coding: utf-8; -*-
#
# This file is part of Superdesk.
#
# Copyright 2013, 2014 Sourcefabric z.u. and contributors.
#
# For the full copyright and license information, please see the
# AUTHORS and LICENSE files distributed with this source code, or
# at https://www.sourcefabric.org/superdesk/license

import logging

import bcrypt
from flask import current_app as app

from superdesk.services import BaseService
from superdesk.utils import is_hashed, get_hash


logger = logging.getLogger(__name__)


class UsersService(BaseService):
    """
    A service that knows how to perform CRUD operations on the `users`
    collection.

    Serves mainly as a proxy to the data layer.
    """

    def on_create(self, docs):
        super().on_create(docs)
        for doc in docs:
            if doc.get('password', None) and not is_hashed(doc.get('password')):
                doc['password'] = self._get_password_hash(doc['password'])

    def on_update(self, updates, original):
        if 'password' in updates:
            updates['password'] = self._get_password_hash(updates['password'])

    def _get_password_hash(self, password):
        return get_hash(password, app.config.get('BCRYPT_GENSALT_WORK_FACTOR', 12))

    def password_match(self, password, hashed_password):
        """Return true if the given password matches the hashed password
        :param password: plain password
        :param hashed_password: hashed password
        """
        try:
            return hashed_password == bcrypt.hashpw(password, hashed_password)
        except Exception:
            return False
