# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['localizationkit', 'localizationkit.tests']

package_data = \
{'': ['*']}

install_requires = \
['toml==0.10.0']

setup_kwargs = {
    'name': 'localizationkit',
    'version': '0.10.0',
    'description': 'String localization tests',
    'long_description': '# localizationkit\n\n`localizationkit` is a toolkit for ensuring that your localized strings are the best that they can be.\n\nIncluded are tests for various things such as:\n\n* Checking that all strings have comments\n* Checking that the comments don\'t just match the value\n* Check that tokens have position specifiers\n* Check that no invalid tokens are included\n\nwith lots more to come. \n\n## Getting started\n\n### Configuration\n\nTo use the library, first off, create a configuration file that is in the TOML format. Here\'s an example:\n\n```toml\ndefault_language = "en"\n\n[has_comments]\nminimum_comment_length = 25\nminimum_comment_words = 8\n\n[token_matching]\nallow_missing_defaults = true\n\n[token_position_identifiers]\nalways = false\n```\n\nThis configuration file sets that `en` is the default language (so this is the language that will be checked for comments, etc. and all tests will run relative to it). Then it sets various settings for each test. Every instance of `[something_here]` specifies that the following settings are for that test. For example, the test `has_comments` will now make sure that not only are there comments, but that they are at least 25 characters in length and 8 words in length. \n\nYou can now load in your configuration:\n\n```python\nfrom localizationkit import Configuration\n\nconfiguration = Configuration.from_file("/path/to/config.toml")\n```\n\n### Localization Collections\n\nNow we need to prepare the strings that will go in. Here\'s how you can create an individual string:\n\n```python\nfrom localizationkit import LocalizedString\n\nmy_string = LocalizedString("My string\'s key", "My string\'s value", "My string\'s comment", "en")\n```\n\nThis creates a single string with a key, value and comment, with its language code set to `en`. Once you\'ve created some more (usually for different languages too), you can bundle them into a collection:\n\n```python\nfrom localizationkit import LocalizedCollection\n\ncollection = LocalizedCollection(list_of_my_strings)\n```\n\n### Running the tests\n\nAt this point, you are ready to run the tests:\n\n```python\nimport localizationkit\n\nresults = localizationkit.run_tests(configuration, collection)\n\nfor result in results:\n    if not result.succeeded():\n        print("The following test failed:", result.name)\n        print("Failures encountered:")\n        for violation in result.violations:\n            print(violation)\n```\n\n### Not running the tests\n\nSome tests don\'t make sense for everyone. To skip a test you can add the following to your config file at the root:\n\n```toml\nblacklist = ["test_identifier_1", "test_identifier_2"]\n```\n\n# Rule documentation\n\nMost tests have configurable rules. If a rule is not specified, it will use the default instead.\n\nSome tests are opt in only. These will be marked as such.\n\n## Comment Linebreaks\n\nIdentifier: `comment_linebreaks`\nOpt-In: `true`\n\nChecks that comments for strings do not contain linebreaks. Comments which contain linebreaks can interfere with parsing in other tools such as [dotstrings](https://github.com/microsoft/dotstrings).\n\n## Comment Similarity\n\nIdentifier: `comment_similarity`\n\nChecks the similarity between a comment and the string\'s value in the default language. This is achieved via `difflib`\'s `SequenceMatcher`. More details can be found [here](https://docs.python.org/3/library/difflib.html#difflib.SequenceMatcher.ratio)\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `maximum_similarity_ratio` | float | Between 0 and 1 | 0.5 | Set the maximum similarity ratio between the comment and the string value. The higher the value, the more similar they are. The longer the string the more accurate this will be. |\n\n</details>\n\n## Duplicate Keys\n\nIdentifier: `duplicate_keys`\n\nChecks that there are no duplicate keys in the collection.\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `all_languages` | boolean | `true` or `false` | `false` | Set to `true` to check that every language has unique keys, not just the default language. |\n\n</details>\n\n## Has Comments\n\nIdentifier: `has_comments`\n\nChecks that strings have comments.\n\n_Note: Only languages that have Latin style scripts are really supported for the words check due to splitting on spaces to check._\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `minimum_comment_length` | int | Any integer | 30 | Set the minimum allowable length for a comment. Set the value to negative to not check. |\n| `minimum_comment_words` | int | Any integer | 10 | Set the minimum allowable number of words for a comment. Set the value to negative to not check. |\n\n</details>\n\n## Has Value\n\nIdentifier: `has_value`\n\nChecks that strings have values. Since any value is enough for some strings, it simply makes sure that the string isn\'t None/null and isn\'t empty.\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `default_language_only` | boolean | `true` or `false` | `false` | Set to true to only check the default language for missing values. Otherwise all languages will be checked. |\n\n</details>\n\n## Invalid Tokens\n\nIdentifier: `invalid_tokens`\n\nChecks that all format tokens in a string are valid.\n\n_Note: This check is not language specific. It only works very broadly._\n\n## Key Length\n\nIdentifier: `key_length`\n\nChecks the length of the keys.\n\n_Note: By default this test doesn\'t check anything. It needs to have parameters set to positive values to do anything._\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `minimum` | int | Any integer | -1 | Set the minimum allowable length for a key. Set the value to negative to not check. |\n| `maximum` | int | Any integer | -1 | Set the maximum allowable length for a key. Set the value to negative to not check. |\n\n</details>\n\n## Objective-C Alternative Tokens\n\nIdentifier: `objectivec_alternative_tokens`\nOpt-In: `true`\n\nChecks that strings do not contain Objective-C style alternative position tokens.\n\nObjective-C seems to be allows positional tokens of the form `%1@` rather than `%1$@`. While not illegal, it is preferred that all tokens between languages are consistent so that tools don\'t experience unexpected failures, etc.\n\n## Swift Interpolation\n\nIdentifier: `swift_interpolation`\nOpt-In: `true`\n\nChecks that strings do not contain Swift style interpolation values since these cannot be localized.\n\n## Token Matching\n\nIdentifier: `token_matching`\n\nChecks that the tokens in a string match across all languages. e.g. If your English string is "Hello %s" but your French string is "Bonjour", this would flag that there is a missing token in the French string.\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `allow_missing_defaults` | boolean | `true` or `false` | `false` | Due to the way that automated localization works, usually there will be a default language, and then other translations will come in over time. If a translation is deleted, it isn\'t always deleted from all languages immediately. Setting this parameter to true will allow any strings in your non-default language to be ignored if that string is missing from your default language. |\n\n</details>\n\n## Token Position Identifiers\n\nIdentifier: `token_position_identifiers`\n\nCheck that each token has a position specifier with it. e.g. `%s` is not allowed, but `%1$s` is. Tokens can move around in different languages, so position specifiers are extremely important.\n\n<details>\n    <summary>Configuration</summary>\n\n| Parameter | Type | Acceptable Values | Default | Details | \n| --- | --- | --- | --- | --- |\n| `always` | boolean | `true` or `false` | `false` | If a string only has a single token, it doesn\'t need a position specifier. Set this to `true` to require it even in those cases.\n\n</details>\n\n# Contributing\n\nThis project welcomes contributions and suggestions.  Most contributions require you to agree to a\nContributor License Agreement (CLA) declaring that you have the right to, and actually do, grant us\nthe rights to use your contribution. For details, visit https://cla.opensource.microsoft.com.\n\nWhen you submit a pull request, a CLA bot will automatically determine whether you need to provide\na CLA and decorate the PR appropriately (e.g., status check, comment). Simply follow the instructions\nprovided by the bot. You will only need to do this once across all repos using our CLA.\n\nThis project has adopted the [Microsoft Open Source Code of Conduct](https://opensource.microsoft.com/codeofconduct/).\nFor more information see the [Code of Conduct FAQ](https://opensource.microsoft.com/codeofconduct/faq/) or\ncontact [opencode@microsoft.com](mailto:opencode@microsoft.com) with any additional questions or comments.\n',
    'author': 'Dale Myers',
    'author_email': 'dalemy@microsoft.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Microsoft/localizationkit',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
