import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as core from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The interface that represents the AccessPoint resource.
 */
export interface IAccessPoint extends core.IResource {
    /**
     * The ARN of the access point.
     * @attribute
     */
    readonly accessPointArn: string;
    /**
     * The creation data of the access point.
     * @attribute
     */
    readonly accessPointCreationDate: string;
    /**
     * The IPv4 DNS name of the access point.
     */
    readonly domainName: string;
    /**
     * The regional domain name of the access point.
     */
    readonly regionalDomainName: string;
    /**
     * The virtual hosted-style URL of an S3 object through this access point.
     * Specify `regional: false` at the options for non-regional URL.
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     */
    virtualHostedUrlForObject(key?: string, options?: s3.VirtualHostedStyleUrlOptions): string;
}
/**
  * The S3 object lambda access point configuration.
  */
export interface AccessPointProps {
    /**
     * The bucket to which this access point belongs.
     */
    readonly bucket: s3.IBucket;
    /**
     * The Lambda function used to transform objects.
     */
    readonly handler: lambda.IFunction;
    /**
     * The name of the S3 object lambda access point.
     *
     * @default a unique name will be generated
     */
    readonly accessPointName?: string;
    /**
     * Whether CloudWatch metrics are enabled for the access point.
     *
     * @default false
     */
    readonly cloudWatchMetricsEnabled?: boolean;
    /**
     * Whether the Lambda function can process `GetObject-Range` requests.
     *
     * @default false
     */
    readonly supportsGetObjectRange?: boolean;
    /**
     * Whether the Lambda function can process `GetObject-PartNumber` requests.
     *
     * @default false
     */
    readonly supportsGetObjectPartNumber?: boolean;
    /**
     * Additional JSON that provides supplemental data passed to the
     * Lambda function on every request.
     *
     * @default - No data.
     */
    readonly payload?: {
        [key: string]: any;
    };
}
declare abstract class AccessPointBase extends core.Resource implements IAccessPoint {
    abstract readonly accessPointArn: string;
    abstract readonly accessPointCreationDate: string;
    abstract readonly accessPointName: string;
    /** Implement the {@link IAccessPoint.domainName} field. */
    get domainName(): string;
    /** Implement the {@link IAccessPoint.regionalDomainName} field. */
    get regionalDomainName(): string;
    /** Implement the {@link IAccessPoint.virtualHostedUrlForObject} method. */
    virtualHostedUrlForObject(key?: string, options?: s3.VirtualHostedStyleUrlOptions): string;
}
/**
 * The access point resource attributes.
 */
export interface AccessPointAttributes {
    /**
     * The ARN of the access point.
     */
    readonly accessPointArn: string;
    /**
     * The creation data of the access point.
     */
    readonly accessPointCreationDate: string;
}
/**
  * An S3 object lambda access point for intercepting and
  * transforming `GetObject` requests.
  */
export declare class AccessPoint extends AccessPointBase {
    /**
     * Reference an existing AccessPoint defined outside of the CDK code.
     */
    static fromAccessPointAttributes(scope: Construct, id: string, attrs: AccessPointAttributes): IAccessPoint;
    /**
     * The ARN of the access point.
     */
    readonly accessPointName: string;
    /**
     * The ARN of the access point.
     * @attribute
     */
    readonly accessPointArn: string;
    /**
     * The creation data of the access point.
     * @attribute
     */
    readonly accessPointCreationDate: string;
    constructor(scope: Construct, id: string, props: AccessPointProps);
}
export {};
