# This file is part of the Data Cleaning Library (openclean).
#
# Copyright (C) 2018-2021 New York University.
#
# openclean is released under the Revised BSD License. See file LICENSE for
# full license details.

"""Abstract base class for pattern discovery operators."""

from abc import ABCMeta, abstractmethod

from openclean.profiling.base import DistinctSetProfiler


class Pattern(metaclass=ABCMeta):
    """Interface for objects representing patterns, e.g., a regular expression,
    that was discovered by a pattern finder. Implementations maintain a
    representation of the pattern itself as well as any additional metadata
    that was generated during the discovery process.
    """
    def __repr__(self):
        """Get a string representation of the pattern for display purposes.

        Returns
        -------
        string
        """
        return self.pattern()

    @abstractmethod
    def compile(self, negate=False, generator=None):
        """Get an instance of a value function that is predicate which can be
        used to test whether an given value is accepted by the pattern or not.

        Parameters
        ----------
        negate: bool, default=False
            If the negate flag is True, the returned predicate should return
            True for values that are not accepeted by the pattern and False for
            those that are accepeted.
        generator: PatternFinder (optional)
            The patternfinder used to generate the original pattern. required
            to recreate the tokenization and type detection on the new values

        Returns
        -------
        openclean.function.value.base.ValueFunction
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def metadata(self):
        """Return a dictionary containing optional metadata associated with the
        pattern. This can for example be statistics generated by the pattern
        discovery algorithm providing additional information or evidence for
        the confidence that the algorithm has in the pattern or the relevance
        of the pattern.

        The structure of the dictionary is implementation-dependent. If no
        additional metadata was generated an empty dictionary should be
        returned.

        Returns
        -------
        dict
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def pattern(self):
        """Get a string representation of the pattern for display purposes.

        Returns
        -------
        string
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def to_dict(self):
        """Returns a dictionary serialization of the pattern. This is an
        external representation that is used when the results of a pattern
        finder are included in the result generated by a data profiler.

        Returns
        -------
        dict
        """
        raise NotImplementedError()  # pragma: no cover


class PatternFinder(DistinctSetProfiler, metaclass=ABCMeta):
    """Interface for generic regular expression discovery. Each implementation
    should take an interable of (distinct) values (e.g., from a column in a
    data frame) as their input. The result is a (list of) string(s) that each
    represent a regular expression.
    """
    def exec(self, values):
        """This method is executed when the pattern finder is used as part of a
        data profiler. It returns a list with dictionary serializations for the
        patterns that are discovered by the find method.

        Parameters
        ----------
        values: list
            List of scalar values or tuples of scalar values.

        Returns
        -------
        list
        """
        result = list()
        for pattern in self.find(values):
            result.add(pattern.to_dict())
        return result

    @abstractmethod
    def find(self, values):
        """Discover patterns like regular expressions in a given sequence of
        (distinct) values. Returns a list of objects representing the
        discovered patterns.

        Parameters
        ----------
        values: list
            List of scalar values or tuples of scalar values.

        Returns
        -------
        list(openclean.profiling.pattern.base.Pattern)
        """
        raise NotImplementedError()  # pragma: no cover
