# This file is part of the Data Cleaning Library (openclean).
#
# Copyright (C) 2018-2021 New York University.
#
# openclean is released under the Revised BSD License. See file LICENSE for
# full license details.

"""Generic conditional outlier detector. Identify values as outliers if they
satisfy a given outlier predicate.
"""

from abc import ABCMeta, abstractmethod
from collections import Counter
from typing import Any, List, Optional, Type

from openclean.data.types import Value
from openclean.profiling.anomalies.base import AnomalyDetector


class ConditionalOutliers(AnomalyDetector, metaclass=ABCMeta):
    """Detect outliers in a given value sequence by testing for each value
    whether they satisfy an implementation-specific outlier condition.
    """
    def __init__(self, resultcls: Optional[Type] = list):
        """Initialize the type of the result class. Instances of this class
        are generated at the start of the process method to collect the outlier
        results.

        Parameters
        ----------
        resultclass: (subcleass of) type list
            Class that is instantiated to collect results. The should be a
            subclass of list (or at least support the append method).
        """
        self.resultcls = resultcls

    def process(self, values: Counter) -> List:
        """Identify values in a given set of values that satisfy the
        outlier condition. This method is called if the outlier detector is
        part of a data profiler configuration. The result is a list containing
        either the oulier values or dictionaries containing the outlier value
        (associated with the key 'value') and additional information that the
        outlier detector provided as supporting evidence (associated with the
        key 'metadata').

        Parameters
        ----------
        values: collections.Counter
            Set of distinct scalar values or tuples of scalar values that are
            mapped to their respective frequency count.

        Returns
        -------
        list
        """
        result = self.resultcls()
        for value in values:
            outlier = self.outlier(value)
            if outlier is not None:
                result.append(outlier)
        return result

    @abstractmethod
    def outlier(self, value: Value, count: int) -> Any:
        """Implementation specific outlier condition. If the given value is
        classified as an outlier, the result is a dictionary object containing
        the outlier value and additional optional provenance information that
        was generated by the outlier detector. If the value is not an outlier,
        the result is None.

        Parameters
        ----------
        value: scalar or tuple
            Value that is being classified as an outlier.

        Returns
        -------
        any
        """
        raise NotImplementedError()
