# AUTOGENERATED! DO NOT EDIT! File to edit: modules.ipynb (unless otherwise specified).

__all__ = ['tensors2parameters', 'Inversible', 'Inverse', 'assert_inversible', 'Translation', 'Rotation',
           'EulerRotation', 'InversibleSequential', 'Rigid', 'Rigids', 'Normalize', 'Augment', 'NoDistortion',
           'Heikkila97Distortion', 'Wang08Distortion', 'a2A', 'Cam', 'CamSF']

# Cell
import torch
import torch.nn as nn

from .utils import *

# Cell
def tensors2parameters(args): return args_loop(args, lambda arg: nn.Parameter(arg))

# Cell
class Inversible(nn.Module):
    def __init__(self):
        super().__init__()

    def forward_param(self): raise NotImplementedError('Please implement forward_param() method')
    def inverse_param(self): raise NotImplementedError('Please implement inverse_param() method')

    # Call this method to get a copy of "inverse aware" reconstructed parameters
    def get_param(self):
        with torch.no_grad():
            return self.forward_param().detach().clone()

    def inverse(self): raise NotImplementedError('Please implement inverse() method')

# Cell
class Inverse(Inversible): # Note that Inverse is itself Inversible...
    def __init__(self, m):
        super().__init__()
        self.m = m

    def __repr__(self):
        return f'{self.__class__.__name__}({self.m.__repr__()})'

    def forward_param(self): return self.m.inverse_param()
    def inverse_param(self): return self.m.forward_param()

    def forward(self, x): return self.m.inverse(x)
    def inverse(self, x): return self.m.forward(x)

# Cell
def assert_inversible(m, x, y, **kwargs):
    assert_allclose(m(x),          y, **kwargs)
    assert_allclose(Inverse(m)(y), x, **kwargs)

# Cell
class Translation(Inversible):
    def __init__(self, t):
        super().__init__()
        self.t = tensors2parameters(t)

    def __repr__(self):
        t_x, t_y, t_z = self.t
        return f'{self.__class__.__name__}(tx:{t_x:.4} ty:{t_y:.4} tz:{t_z:.4})'

    def forward_param(self): return  self.t
    def inverse_param(self): return -self.forward_param()

    def forward(self, ps, inverse=False):
        t = self.forward_param() if not inverse else self.inverse_param()
        return ps + t
    def inverse(self, ps): return self.forward(ps, inverse=True)

# Cell
class Rotation(Inversible):
    def __init__(self):
        super().__init__()

    def r2R(self):
        raise NotImplementedError('Please implement rotation matrix reconstruction')

    def forward_param(self): return self.r2R()
    def inverse_param(self): return self.forward_param().T

    def forward(self, ps, inverse=False):
        R = self.forward_param() if not inverse else self.inverse_param()
        return pmm(ps, R)
    def inverse(self, ps): return self.forward(ps, inverse=True)

# Cell
class EulerRotation(Rotation):
    def __init__(self, R):
        super().__init__()
        self.euler = tensors2parameters(R2euler(R))

    def __repr__(self):
        e_x, e_y, e_z = self.euler
        return f'{self.__class__.__name__}(ex:{e_x:.4} ey:{e_y:.4} ez:{e_z:.4})'

    def r2R(self): return euler2R(self.euler)

# Cell
class InversibleSequential(Inversible):
    def __init__(self, ms):
        super().__init__()
        self.ms = nn.ModuleList(ms)

    def forward(self, x):
        for m in self.ms:
            x = m.forward(x)
        return x

    def inverse(self, x):
        for m in reversed(self.ms):
            x = m.inverse(x)
        return x

# Cell
class Rigid(InversibleSequential):
    def __init__(self, R, t, Rotation=EulerRotation):
        super().__init__([Rotation(R), Translation(t)]) # NOTE: rotation happens first

    def forward_param(self):
        return Rt2M(self.ms[0].forward_param(), # R
                    self.ms[1].forward_param()) # t

    def inverse_param(self): return invert_rigid(self.forward_param())

# Cell
class Rigids(InversibleSequential):
    def __init__(self, rigids):
        super().__init__(rigids)

    def forward_param(self):
        M = self.ms[0].forward_param()
        for m in self.ms[1:]:
            M = mult_rigid(m.forward_param(), M)
        return M

    def inverse_param(self): return invert_rigid(self.forward_param())

# Cell
class Normalize(nn.Module): # Note: Not Inversible
    def __init__(self):
        super().__init__()

    def forward(self, ps): return normalize(ps)

# Cell
class Augment(Inversible):
    def __init__(self):
        super().__init__()

    def forward(self, ps): return augment(ps)
    def inverse(self, ps): return deaugment(ps)

# Cell
NoDistortion = nn.Identity

# Cell
class Heikkila97Distortion(nn.Module):
    def __init__(self, d):
        super().__init__()
        self.d = tensors2parameters(d)

    def __repr__(self):
        k1, k2, p1, p2 = self.d
        return f'{self.__class__.__name__}(k1:{k1:.4} k2:{k2:.4} p1:{p1:.4} p2:{p2:.4})'

    def forward(self, ps):
        k1, k2, p1, p2 = self.d
        xs, ys = ps[:,0], ps[:,1]

        # Radial distortion
        rs = xs**2 + ys**2
        xs_r = xs*(1 + k1*rs + k2*rs**2)
        ys_r = ys*(1 + k1*rs + k2*rs**2)

        # Decentering distortion
        xs_d = xs_r + 2*p1*xs*ys + p2*(3*xs**2 + ys**2)
        ys_d = ys_r + p1*(xs**2 + 3*ys**2) + 2*p2*xs*ys

        return stackify((xs_d, ys_d), dim=1)

# Cell
class Wang08Distortion(nn.Module):
    def __init__(self, d):
        super().__init__()
        self.d = tensors2parameters(d)

    def __repr__(self):
        k1, k2, p, t = self.d
        return f'{self.__class__.__name__}(k1:{k1:.4} k2:{k2:.4} p:{p:.4} t:{t:.4})'

    def forward(self, ps):
        k1, k2, p, t = self.d
        xs, ys = ps[:,0], ps[:,1]

        # Radial distortion
        rs = xs**2 + ys**2
        xs_r = xs*(1 + k1*rs + k2*rs**2)
        ys_r = ys*(1 + k1*rs + k2*rs**2)

        # Image plane (small angle approximation) rotation distortion
        xs_d = xs_r/(-p*xs_r + t*ys_r + 1)
        ys_d = ys_r/(-p*xs_r + t*ys_r + 1)

        return stackify((xs_d, ys_d), dim=1)

# Cell
def a2A(alpha_x, alpha_y, x_o, y_o):
    zero, one = alpha_x.new_tensor(0), alpha_x.new_tensor(1)

    return stackify(((alpha_x,    zero, x_o),
                     (   zero, alpha_y, y_o),
                     (   zero,    zero, one)))

# Cell
class Cam(Inversible):
    def __init__(self):
        super().__init__()

    def a2A(self):
        raise NotImplementedError('Please implement camera matrix reconstruction')

    def forward_param(self): return self.a2A()
    def inverse_param(self): return torch.inverse(self.forward_param()) # Overwrite this in child classes

    def forward(self, ps, inverse=False):
        A = self.forward_param() if not inverse else self.inverse_param()
        return pmm(ps, A, aug=True)
    def inverse(self, ps): return self.forward(ps, inverse=True)

# Cell
class CamSF(Cam):
    def __init__(self, A):
        super().__init__()
        assert_allclose(A[0,0], A[1,1]) # Check that alpha_x and alpha_y are the same
        self.a = tensors2parameters(A[[0,0,1],[0,2,2]])

    def __repr__(self):
        alpha, x_o, y_o = self.a
        return f'{self.__class__.__name__}(alpha:{alpha:.4} x_o:{x_o:.4} y_o:{y_o:.4})'

    def a2A(self):
        alpha, x_o, y_o = self.a
        return a2A(alpha_x=alpha, alpha_y=alpha, x_o=x_o, y_o=y_o)

    def inverse_param(self): # Overwrite parent method
        alpha, x_o, y_o = self.a
        return a2A(alpha_x=1/alpha,
                   alpha_y=1/alpha,
                   x_o=-x_o/alpha,
                   y_o=-y_o/alpha)