import datetime
import gzip
import logging
import os
import sys
from subprocess import check_call, check_output

import pandas as pd
from pathlib import Path

RESULTS_ROOT_DIRECTORY_PATH = "<RESULTS_ROOT_DIRECTORY_PATH>"
TENX_RELATIVE_DIRECTORY_PATH = "<TENX_RELATIVE_DIRECTORY_PATH>"
BD_DEMUX_DIRECTORY_NAME = "bd_demux"
BD_DEMUX_FILE_NAME = "Sample_Tag_Calls.csv"
DEMUXLET_DIRECTORY_NAME = "demuxlet"
DEMUXLET_FILE_NAME = "demuxlet.best"
FEATURES_FILE_NAME = "features.tsv.gz"
BARCODES_FILE_NAME = "barcodes.tsv.gz"
MATRIX_FILE_NAME = "matrix.mtx.gz"
SCRNASEQ_DEMUX_DIRECTORY_NAME = "scrnaseq_demux"
SCRNASEQ_DEMUX_FILE_NAME = "scrnaseq_demux_posterior_probs.csv"


def load_10x_lane(analysis_output_directory_path):
    """

    :param analysis_output_directory_path: Local path to the top level directory of a run of the scrnaseq pipeline
    For example, the folder referred to in the `analysis_output_path` column of the `scrnaseq_analyis_run` table
    :return: Dataframes of the three file types generated by the Cellranger pipeline
    (features.tsv, barcodes.tsv, matrix.mtx)
    """
    analysis_output_directory_path = Path(analysis_output_directory_path)
    tenx_dir = analysis_output_directory_path / TENX_RELATIVE_DIRECTORY_PATH

    features_df = pd.read_csv(
        tenx_dir / FEATURES_FILE_NAME, compression="gzip", header=None, sep="\t", usecols=[0, 1],
    ).rename(columns={0: "ensembl_id", 1: "gene_name"})

    barcodes_df = pd.read_csv(tenx_dir / BARCODES_FILE_NAME, compression="gzip", header=None, sep="\t").rename(
        columns={0: "barcode"}
    )

    mat = pd.read_csv(
        tenx_dir / MATRIX_FILE_NAME, skiprows=3, header=None, sep=" ", names=["feature_idx", "barcode_idx", "count"],
    )

    return features_df, barcodes_df, mat


# def load_10x_lane_scipy(analysis_output_directory_path):
#     analysis_output_directory_path = Path(analysis_output_directory_path)
#     tenx_dir = analysis_output_directory_path / TENX_RELATIVE_DIRECTORY_PATH
#     features_df, barcodes_df, _ = load_10x_lane(analysis_output_directory_path)
#     mat = mmread(str(tenx_dir / MATRIX_FILE_NAME))
#     return features_df, barcodes_df, mat


def load_bd_demux(analysis_output_directory_path):
    """
    Loads the output file from bd_demux into a dataframe. Only keeps rows that describe singlets and creates
    a column for the bd_antibody_id embedded in the cell barcode to use to figure out the sample/donor of each cell
    :param analysis_output_directory_path: Local path to the top level directory of a run of the scrnaseq pipeline
    For example, the folder referred to in the `analysis_output_path` column of the `scrnaseq_analyis_run` table
    (has to be local though and not an S3 path)
    :return: Dataframe containing the results of demultiplexing
    """
    analysis_output_directory_path = Path(analysis_output_directory_path)
    demux_bd = pd.read_csv(analysis_output_directory_path / BD_DEMUX_DIRECTORY_NAME / BD_DEMUX_FILE_NAME)
    singlets = demux_bd.loc[demux_bd["noiseReduction"].str.startswith("SampleTag")].copy()
    singlets.loc[:, "bd_id"] = singlets["noiseReduction"].str.split("_").str[-1].str.split("|").str[0].astype(int)
    singlets = singlets.rename(columns={"cell": "barcode"})[["barcode", "bd_id"]]
    return singlets


def load_demuxlet_demux(analysis_output_directory_path):
    """
    Loads the output file from demuxlet into a dataframe. Only keeps rows that describe singlets and creates
    a column for the <name> in the cell barcode
    :param analysis_output_directory_path: Local path to the top level directory of a run of the scrnaseq pipeline
    For example, the folder referred to in the `analysis_output_path` column of the `scrnaseq_analyis_run` table
    (has to be local though and not an S3 path)
    :return: Dataframe containing the results of demultiplexing
    """
    analysis_output_directory_path = Path(analysis_output_directory_path)
    demuxlet = pd.read_csv(
        analysis_output_directory_path / DEMUXLET_DIRECTORY_NAME / DEMUXLET_FILE_NAME, sep="\t", usecols=[0, 5],
    )
    demuxlet.columns = [col.lower() for col in demuxlet.columns]
    demuxed_cells = demuxlet.loc[demuxlet.loc[:, "best"].str.startswith("SNG"), :].copy()
    source_ids = demuxed_cells.loc[:, "best"].str.split("-").str[1].tolist()
    #

    return demuxed_cells


def load_scrnaseq_demux(analysis_output_directory_path, prob_thresh):
    analysis_output_directory_path = Path(analysis_output_directory_path)
    demux = pd.read_csv(
        analysis_output_directory_path / SCRNASEQ_DEMUX_DIRECTORY_NAME / SCRNASEQ_DEMUX_FILE_NAME, sep="\t",
    )
    demux.rename(columns={"BARCODE": "barcode"}, inplace=True)
    demux["barcode_idx"] = demux.index
    demux.set_index("barcode", inplace=True)
    # demux.columns = [col.lower() for col in demux.columns]
    sng_cols = list(filter(lambda x: "+" not in x, demux.columns))
    demux = demux[sng_cols]
    donor_cols = list(filter(lambda x: x.startswith("D"), demux.columns))
    donor_demux = demux[donor_cols]
    filter_ = donor_demux.max(axis=1) > prob_thresh
    demux = demux[filter_]
    #
    return demux.set_index("barcode_idx")


def check_donor_mixup(good_cells_df, lane_label_df):
    pass


def combine_10x_outputs(
    output_dir,
    sample_metadata_df,
    scrnaseq_demux=True,
    prob_thresh=0.99,
    min_cell_frac=0.01,
    min_cells=100,
    for_mixup_stats=False,
):
    """
    This function will take a dataframe describing samples for which cellranger output exists across multiple
    cellranger runs and will combine those runs into a single set of files as though they were all run together.
    This allows us to keep all of our data in one "dataset"

    output_dir - output directory, no existing 10x output files
    metadata_df - pandas dataframe of metadata for the samples that make up the 10x data that will be combined.
    """
    logger = logging.getLogger("combine_10x_outputs")

    output_dir = Path(output_dir)
    os.makedirs(output_dir, exist_ok=True)
    curr_good_cell_count = 0
    reference_feature_map = None
    all_metadata = []

    # We append to tmp_matrix.mtx.gz and barcodes.tsv.gz below, so if it already exists, this can mess things up.
    # Make the user confirm they really want to do this so we don't blow up existing data.
    tmp_matrix_file_path = output_dir / "tmp_matrix.mtx.gz"
    if os.path.exists(tmp_matrix_file_path):
        sys.exit(f"{tmp_matrix_file_path} already exists. If you really want to regenerate this file, delete it first.")
    barcodes_file_path = output_dir / BARCODES_FILE_NAME
    if os.path.exists(barcodes_file_path):
        sys.exit(f"{barcodes_file_path} already exists. If you really want to regenerate this file, delete it first.")
    # keep only newest run for each sample
    #
    if len(sample_metadata_df) == 0:
        raise ValueError("empty metadata provided")
    grp = sample_metadata_df.groupby(["<grouping_vars>"])
    grouping_vars = None
    for idx, ((grouping_vars), lane_label_df) in enumerate(grp):
        print(f"Loading {grouping_vars}")
        print(datetime.datetime.now())

        #

        features_df, barcodes_df, mat = load_10x_lane(analysis_output_path)
        n_cells_init = len(barcodes_df)
        # because the barcodes and features are 1-indexed in the matrix file
        # Confirmation that MEX files are 1-indexed: https://math.nist.gov/MatrixMarket/formats.html
        barcodes_df["barcode_idx"] = barcodes_df.index + 1
        features_df["feature_idx"] = features_df.index + 1
        if reference_feature_map is None:
            reference_feature_map = features_df.rename(columns={"feature_idx": "reference_feature_idx"})
        print(analysis_output_path)
        if is_bd:
            good_cells_df = load_bd_demux(analysis_output_path)
        elif scrnaseq_demux:
            good_cells_df = load_scrnaseq_demux(analysis_output_path, prob_thresh)
        else:
            good_cells_df = load_demuxlet_demux(analysis_output_path)
        good_cells_df = barcodes_df.merge(good_cells_df, on="barcode")
        if is_bd:
            good_cells_df = good_cells_df.rename(columns={"bd_id": "bd_antibody_id"}).merge(
                lane_label_df, on="bd_antibody_id"
            )
            print("USING BD LANE")
        else:
            #
            good_cells_df = good_cells_df.merge(lane_label_df, on=merge_column_name)
        good_cells_df.loc[:, "barcode"] = (
            good_cells_df.loc[:, "barcode"]
            + "_"
            + good_cells_df.loc[:, "entity_id"].map(str)
            + "_"
            + good_cells_df.loc[:, "lane_label"]
        )

        good_cells_df["new_barcode_idx"] = good_cells_df.index + 1 + curr_good_cell_count
        filtered_mat = mat.merge(good_cells_df, on="barcode_idx").merge(features_df, on="feature_idx")
        curr_good_cell_count += len(good_cells_df)  # set global index for combined matrix
        filtered_mat = filtered_mat.merge(
            reference_feature_map[["reference_feature_idx", "ensembl_id"]], on="ensembl_id",
        )
        with gzip.open(str(tmp_matrix_file_path), mode="at") as f:
            filtered_mat[["reference_feature_idx", "new_barcode_idx", "count"]].to_csv(
                f, index=False, sep=" ", header=None
            )

        with gzip.open(str(barcodes_file_path), mode="at") as f:
            good_cells_df["barcode"].to_csv(f, index=False, header=False)
        all_metadata.append(good_cells_df)
        logger.info(f"keeping {len(good_cells_df)} / {n_cells_init} cells")
    # Remove "best" column that is unique to demuxlet cell dataframes
    to_merge = [df.drop("best", axis=1) if "best" in df.columns else df for df in all_metadata]
    # Reorder all dataframes to have columns in the same order (as the first lane_label_df)
    to_merge = [df[to_merge[0].columns] for df in to_merge]
    merged_metadata = pd.concat(to_merge)

    merged_metadata = merged_metadata.sort_values("new_barcode_idx")
    merged_metadata = merged_metadata.rename(columns={"barcode": "cell_id"})
    merged_metadata.to_csv(output_dir / "cell_metadata.tsv", index=False, sep="\t")

    feature_map_out = reference_feature_map[["ensembl_id", "gene_name"]].applymap(
        lambda x: x.replace("GRCh38_______", "")
    )
    feature_map_out.to_csv(
        output_dir / FEATURES_FILE_NAME, compression="gzip", index=False, header=None, sep="\t",
    )

    final_matrix_path = output_dir / MATRIX_FILE_NAME

    num_entries = int(check_output(f"gunzip -c {tmp_matrix_file_path} | wc -l", shell=True).decode("utf-8").strip())

    header = "%%MatrixMarket matrix coordinate integer general\n"
    header += """%metadata_json: {"format_version": 2, "software_version": "3.0.2"}\n"""
    header += "\t".join(map(str, [len(reference_feature_map), len(merged_metadata), num_entries]))
    header += "\n"

    with gzip.open(final_matrix_path, "wb") as f:
        f.write(header.encode())

    failed = check_call(f"cat {tmp_matrix_file_path} >> {final_matrix_path}", shell=True)
    if not failed:
        os.remove(tmp_matrix_file_path)
