"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatTable = void 0;
const chalk = require("chalk");
const string_width_1 = require("string-width");
const table = require("table");
/**
 * Render a two-dimensional array to a visually attractive table
 *
 * First row is considered the table header.
 */
function formatTable(cells, columns) {
    return table.table(cells, {
        border: TABLE_BORDER_CHARACTERS,
        columns: buildColumnConfig(columns !== undefined ? calculateColumnWidths(cells, columns) : undefined),
        drawHorizontalLine: (line) => {
            // Numbering like this: [line 0] [header = row[0]] [line 1] [row 1] [line 2] [content 2] [line 3]
            return (line < 2 || line === cells.length) || lineBetween(cells[line - 1], cells[line]);
        },
    }).trimRight();
}
exports.formatTable = formatTable;
/**
 * Whether we should draw a line between two rows
 *
 * Draw horizontal line if 2nd column values are different.
 */
function lineBetween(rowA, rowB) {
    return rowA[1] !== rowB[1];
}
function buildColumnConfig(widths) {
    if (widths === undefined) {
        return undefined;
    }
    const ret = {};
    widths.forEach((width, i) => {
        if (width === undefined) {
            return;
        }
        ret[i] = { width };
    });
    return ret;
}
/**
 * Calculate column widths given a terminal width
 *
 * We do this by calculating a fair share for every column. Extra width smaller
 * than the fair share is evenly distributed over all columns that exceed their
 * fair share.
 */
function calculateColumnWidths(rows, terminalWidth) {
    // The terminal is sometimes reported to be 0. Also if the terminal is VERY narrow,
    // just assume a reasonable minimum size.
    terminalWidth = Math.max(terminalWidth, 40);
    // use 'string-width' to not count ANSI chars as actual character width
    const columns = rows[0].map((_, i) => Math.max(...rows.map(row => string_width_1.default(String(row[i])))));
    // If we have no terminal width, do nothing
    const contentWidth = terminalWidth - 2 - columns.length * 3;
    // If we don't exceed the terminal width, do nothing
    if (sum(columns) <= contentWidth) {
        return columns;
    }
    const fairShare = Math.min(contentWidth / columns.length);
    const smallColumns = columns.filter(w => w < fairShare);
    let distributableWidth = contentWidth - sum(smallColumns);
    const fairDistributable = Math.floor(distributableWidth / (columns.length - smallColumns.length));
    const ret = new Array();
    for (const requestedWidth of columns) {
        if (requestedWidth < fairShare) {
            // Small column gets what they want
            ret.push(requestedWidth);
        }
        else {
            // Last column gets all remaining, otherwise get fair redist share
            const width = distributableWidth < 2 * fairDistributable ? distributableWidth : fairDistributable;
            ret.push(width);
            distributableWidth -= width;
        }
    }
    return ret;
}
function sum(xs) {
    let total = 0;
    for (const x of xs) {
        total += x;
    }
    return total;
}
// What color the table is going to be
const tableColor = chalk.gray;
// Unicode table characters with a color
const TABLE_BORDER_CHARACTERS = {
    topBody: tableColor('─'),
    topJoin: tableColor('┬'),
    topLeft: tableColor('┌'),
    topRight: tableColor('┐'),
    bottomBody: tableColor('─'),
    bottomJoin: tableColor('┴'),
    bottomLeft: tableColor('└'),
    bottomRight: tableColor('┘'),
    bodyLeft: tableColor('│'),
    bodyRight: tableColor('│'),
    bodyJoin: tableColor('│'),
    joinBody: tableColor('─'),
    joinLeft: tableColor('├'),
    joinRight: tableColor('┤'),
    joinJoin: tableColor('┼'),
};
//# sourceMappingURL=data:application/json;base64,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