import { aws_ec2 as ec2, aws_ecs as ecs } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * Amazon ECS Capacity Providers for AWS Fargate
 */
export declare enum FargateCapacityProviderType {
    FARGATE = "FARGATE",
    FARGATE_SPOT = "FARGATE_SPOT"
}
/**
 * The Capacity Provider strategy
 */
export interface CapacityProviderStrategyItem {
    readonly base?: number;
    readonly weight: number;
    readonly capacityProvider: FargateCapacityProviderType;
}
/**
 * Properties for the FargateRunner
 */
export interface FargateRunnerProps {
    /**
     * VPC for the fargate
     */
    readonly vpc: ec2.IVpc;
    /**
     * tags for the runner
     */
    readonly tags?: string[];
    /**
     * The security group for Fargate CI task
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * subnet for fargate CI task
     */
    readonly fargateJobSubnet?: ec2.SubnetSelection;
    /**
     * GitLab registration token for the runner
     */
    readonly registrationToken?: string;
    /**
     * gitlab URL prefix
     *
     * @default - 'https://gitlab.com'
     */
    readonly gitlabURL?: string;
    /**
     * Fargate job executor options
     */
    readonly executor?: FargateJobExecutor;
    /**
     * Default capacity provider strategy for the Amazon ECS cluster
     *
     * @default DEFAULT_CLUSTER_CAPACITY_PROVIDER_STRATEGY
     */
    readonly clusterDefaultCapacityProviderStrategy?: CapacityProviderStrategyItem[];
    /**
     * Default capacity provider strategy for the Amazon ECS service
     *
     * @default DEFAULT_SERVICE_CAPACITY_PROVIDER_STRATEGY
     */
    readonly serviceDefaultCapacityProviderStrategy?: CapacityProviderStrategyItem[];
    /**
     * The docker image for the job runner container
     */
    readonly image?: JobRunnerImage;
}
/**
 * The docker image for the job executor
 */
export declare class JobExecutorImage {
    readonly uri: string;
    /**
     * Debian
     * @see https://gitlab.com/tmaczukin-test-projects/fargate-driver-debian
     */
    static readonly DEBIAN: JobExecutorImage;
    /**
     * Node
     * @see https://gitlab.com/aws-fargate-driver-demo/docker-nodejs-gitlab-ci-fargate
     */
    static readonly NODE: JobExecutorImage;
    /**
     * JSII for AWS CDK
     * @see https://gitlab.com/pahud/docker-jsii-cdk-gitlab-ci-fargate
     *
     */
    static readonly JSII: JobExecutorImage;
    /**
     * Custom image
     * @param image custom image registry URI
     */
    static of(image: string): JobExecutorImage;
    /**
     *
     * @param image job executor image URI
     */
    private constructor();
}
/**
 * The docker image for the job runner
 */
export declare class JobRunnerImage {
    readonly uri: string;
    /**
     * Debian
     * @see https://gitlab.com/pahud/docker-gitlab-runner-fargate-driver
     */
    static readonly DEFAULT: JobRunnerImage;
    /**
     * Custom image
     * @param image custom image registry URI
     */
    static of(image: string): JobRunnerImage;
    /**
     *
     * @param image job runner image URI
     */
    private constructor();
}
/**
 * The FargateRunner
 */
export declare class FargateRunner extends Construct {
    readonly vpc: ec2.IVpc;
    constructor(scope: Construct, id: string, props: FargateRunnerProps);
    private synthesizeTags;
    private createSecurityGroup;
}
/**
 * The properties for the FargateJobExecutor
 */
export interface FargateJobExecutorProps {
    /**
     * The docker image for the job executor container
     */
    readonly image?: JobExecutorImage;
    /**
     * AWS region for the job executor
     */
    readonly region?: string;
    readonly cluster?: ecs.ICluster;
    readonly subnet?: ec2.ISubnet;
    readonly securityGroup?: ec2.ISecurityGroup;
}
/**
 * The FargateJobExecutor
 */
export declare class FargateJobExecutor extends Construct {
    /**
     * task definition arn
     */
    readonly taskDefinitionArn: string;
    readonly region: string;
    readonly cluster?: ecs.ICluster;
    readonly subnet?: ec2.ISubnet;
    readonly securityGroup?: ec2.ISecurityGroup;
    constructor(scope: Construct, id: string, props?: FargateJobExecutorProps);
}
