# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hamlpy',
 'hamlpy.parser',
 'hamlpy.template',
 'hamlpy.test',
 'hamlpy.views',
 'hamlpy.views.generic']

package_data = \
{'': ['*'], 'hamlpy.test': ['templates/*']}

install_requires = \
['django>=2.1', 'regex>=2020.1.1']

entry_points = \
{'console_scripts': ['hamlpy-watcher = hamlpy.hamlpy_watcher:watch_folder']}

setup_kwargs = {
    'name': 'django-hamlpy',
    'version': '1.4.4',
    'description': 'HAML like syntax for Django templates',
    'long_description': '# Introduction\n\n[![Build Status](https://github.com/nyaruka/django-hamlpy/workflows/CI/badge.svg)](https://github.com/nyaruka/django-hamlpy/actions?query=workflow%3ACI)\n[![Coverage Status](https://codecov.io/gh/nyaruka/django-hamlpy/branch/main/graph/badge.svg)](https://codecov.io/gh/nyaruka/django-hamlpy)\n[![PyPI Release](https://img.shields.io/pypi/v/django-hamlpy.svg)](https://pypi.python.org/pypi/django-hamlpy/)\n\nWhy type:\n\n```html\n<div class="left" id="banner">\n    Greetings!\n</div>\n```\n\nwhen you can just type:\n\n```haml\n.left#banner\n    Greetings!\n```\n\n... and do something more fun with all the time you save not typing angle brackets and remembering to close tags? \n\nThe syntax above is [Haml](http://www.haml-lang.com) - a templating language used extensively in the Ruby on Rails \ncommunity. This library lets Django developers use a Haml like syntax in their templates. It\'s not a template engine in \nitself, but simply a compiler which will convert "HamlPy" files into templates that Django can understand.\n\nThis project is a fork of the no longer maintained [HamlPy](https://github.com/jessemiller/HamlPy). It introduces \nPython 3 support, support for new Django versions, and a host of new features and bug fixes. Note that the package name \nis now *django-hamlpy*.\n\n## Installing\n\nThe latest stable version can be installed using [pip](http://pypi.python.org/pypi/pip/):\n\n    pip install django-hamlpy\n\nAnd the latest development version can be installed directly from GitHub:\n\n    pip install git+https://github.com/nyaruka/django-hamlpy\n\n**NOTE:** If you run into build errors, then you may need to install [python\'s development package](http://stackoverflow.com/a/21530768/2896976).\n\n## Syntax\n\nAlmost all of the syntax of Haml is preserved.\n\n```haml\n#profile(style="width: 200px")\n    .left.column\n        #date 2010/02/18\n        #address Toronto, ON\n    .right.column<\n        #bio Jesse Miller\n```\n\nturns into:\n\n```htmldjango\n<div id=\'profile\' style="width: 200px">\n    <div class=\'left column\'>\n        <div id=\'date\'>2010/02/18</div>\n        <div id=\'address\'>Toronto, ON</div>\n    </div>\n    <div class=\'right column\'><div id=\'bio\'>Jesse Miller</div></div>\n</div>\n```\n\nThe main difference is instead of interpreting Ruby, or even Python we instead can create Django tags and variables. For \nexample:\n\n```haml\n%ul#athletes\n    - for athlete in athlete_list\n        %li.athlete{\'id\': \'athlete_#{ athlete.pk }\'}= athlete.name\n```\n\nbecomes...\n\n```htmldjango\n<ul id=\'athletes\'>\n    {% for athlete in athlete_list %}\n        <li class=\'athlete\' id=\'athlete_{{ athlete.pk }}\'>{{ athlete.name }}</li>\n    {% endfor %}\n</ul>\n```\n\n## Usage\n\nThere are two different ways to use this library.\n\n### Option 1: Template loaders\n\nThese are Django template loaders which will convert any templates with `.haml` or `.hamlpy` extensions to regular \nDjango templates whenever they are requested by a Django view. To use them, add them to the list of template loaders in \nyour Django settings, e.g.\n\n```python\nTEMPLATES=[\n    {\n        \'BACKEND\': \'django.template.backends.django.DjangoTemplates\',\n        \'DIRS\': [\'./templates\'],\n        \'OPTIONS\': {\n            \'loaders\': (\n                \'hamlpy.template.loaders.HamlPyFilesystemLoader\',\n                \'hamlpy.template.loaders.HamlPyAppDirectoriesLoader\',\n                ...\n            ), \n        }\n    }\n]\n```\n\nEnsure they are listed before the standard Django template loaders or these loaders will try to process your Haml \ntemplates.\n\n#### Template caching\n\nYou can use these loaders with template caching - just add `django.template.loaders.cached.Loader` to your list of \nloaders, e.g.\n\n```python\n\'loaders\': (\n    (\'django.template.loaders.cached.Loader\', (\n        \'hamlpy.template.loaders.HamlPyFilesystemLoader\',\n        \'hamlpy.template.loaders.HamlPyAppDirectoriesLoader\',\n        ...\n    )),\n)\n```\n\n#### Settings\n\nYou can configure the Haml compiler with the following Django settings:\n\n  * `HAMLPY_ATTR_WRAPPER` -- The character that should wrap element attributes. Defaults to `\'` (an apostrophe).\n  * `HAMLPY_DJANGO_INLINE_STYLE` -- Whether to support `={...}` syntax for inline variables in addition to `#{...}`. \n     Defaults to `False`.\n\n### Option 2: Watcher\n\nThe library can also be used as a stand-alone program. There is a watcher script which will monitor Haml files in a \ngiven directory and convert them to HTML as they are edited.\n\n```\nusage: hamlpy_watcher.py [-h] [-v] [-i EXT [EXT ...]] [-ext EXT] [-r S]\n                         [--tag TAG] [--attr-wrapper {",\'}] [--django-inline]\n                         [--jinja] [--once]\n                         input_dir [output_dir]\n\npositional arguments:\n  input_dir             Folder to watch\n  output_dir            Destination folder\n\noptional arguments:\n  -h, --help            show this help message and exit\n  -v, --verbose         Display verbose output\n  -i EXT [EXT ...], --input-extension EXT [EXT ...]\n                        The file extensions to look for.\n  -ext EXT, --extension EXT\n                        The output file extension. Default is .html\n  -r S, --refresh S     Refresh interval for files. Default is 3 seconds.\n                        Ignored if the --once flag is set.\n  --tag TAG             Add self closing tag. eg. --tag macro:endmacro\n  --attr-wrapper {",\'}  The character that should wrap element attributes.\n                        This defaults to \' (an apostrophe).\n  --django-inline       Whether to support ={...} syntax for inline variables\n                        in addition to #{...}\n  --jinja               Makes the necessary changes to be used with Jinja2.\n  --once                Runs the compiler once and exits on completion.\n                        Returns a non-zero exit code if there were any compile\n                        errors.\n```\n\n### Create message files for translation\n\nHamlPy must first be included in Django\'s list of apps, i.e.\n\n```python\nINSTALLED_APPS = [\n  ...\n  \'hamlpy\'\n  ...\n]\n```\n\nThen just include your Haml templates along with all the other files which contain translatable strings, e.g.\n\n```bash\npython manage.py makemessages --extension haml,html,py,txt\n```\n\n## Reference\n\nCheck out the [reference](http://github.com/nyaruka/django-hamlpy/blob/master/REFERENCE.md) file for the complete syntax \nreference and more examples.\n\n## Class Based Views\n\nThis library also provides [the same class based generic views than django](https://docs.djangoproject.com/en/1.10/topics/class-based-views/generic-display/) with the enhancement that they start by looking for templates endings with `*.haml` and `*.hamlpy` in addition to their default templates. Apart from that, they are exactly the same class based generic views. For example:\n\n```python\nfrom hamlpy.views.generic import DetailView, ListView\nfrom my_app.models import SomeModel\n\n# will look for the templates `my_app/somemodel_detail.haml`,\n# `my_app/somemodel_detail.hamlpy` and  `my_app/somemodel_detail.html`\nDetailView.as_view(model=SomeModel)\n\n# will look for the templates `my_app/somemodel_list.haml`,\n# `my_app/somemodel_list.hamlpy` and  `my_app/somemodel_list.html`\nListView.as_view(model=SomeModel)\n```\n\nThe available view classes are:\n\nDisplay views:\n\n* [DetailView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#detailview)\n* [ListView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#listview)\n\nEdit views:\n\n* [CreateView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#createview)\n* [UpdateView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#updateview)\n* [DeleteView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#deleteview)\n\nDate related views:\n\n* [DateDetailView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#datedetailview)\n* [ArchiveIndexView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#archiveindexview)\n* [YearArchiveView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#yeararchiveview)\n* [MonthArchiveView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#montharchiveview)\n* [WeekArchiveView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#weekarchiveview)\n* [DayArchiveView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#dayarchiveview)\n* [TodayArchiveView](https://docs.djangoproject.com/en/1.10/ref/class-based-views/generic-display/#todayarchiveview)\n\nAll views are importable from `hamlpy.views.generic` and are built using the `HamlExtensionTemplateView` mixin which you \ncan use to create your own custom Haml-using views. For example:\n\n```python\nfrom hamlpy.views.generic import HamlExtensionTemplateView\n\nclass MyNewView(HamlExtensionTemplateView, ParentViewType):\n    pass\n```\n\n**Note**: `HamlExtensionTemplateView` *needs* to be first in the inheritance list.\n\n## Contributing\n\nWe\'re always happy to have contributions to this project. To get started you\'ll need to clone the project and install \nthe dependencies:\n\n    poetry install\n\nPlease write tests for any new features and always ensure the current tests pass. To run the tests, use:\n\n    py.test hamlpy  \n    \nTo run the performance test, use:\n\n    python -m hamlpy.test.test_templates\n',
    'author': 'Nyaruka',
    'author_email': 'code@nyaruka.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
