import json
import logging
import re
from typing import Any, List, Type, TypeVar, get_type_hints

import requests


from nemo_library.model.application import Application
from nemo_library.model.attribute_group import AttributeGroup
from nemo_library.model.defined_column import DefinedColumn
from nemo_library.model.dependency_tree import DependencyTree
from nemo_library.model.diagram import Diagram
from nemo_library.model.imported_column import ImportedColumn
from nemo_library.model.metric import Metric
from nemo_library.model.pages import Page
from nemo_library.model.project import Project
from nemo_library.model.report import Report
from nemo_library.model.subprocess import SubProcess
from nemo_library.model.tile import Tile
from nemo_library.utils.config import Config
from nemo_library.utils.utils import FilterType, FilterValue, clean_meta_data, log_error

T = TypeVar("T")


def _deserializeMetaDataObject(value: Any, target_type: Type) -> Any:
    """
    Recursively deserializes JSON data into a nested DataClass structure.
    """
    if isinstance(value, list):
        # Check if we expect a list of DataClasses
        if hasattr(target_type, "__origin__") and target_type.__origin__ is list:
            element_type = target_type.__args__[0]
            return [_deserializeMetaDataObject(v, element_type) for v in value]
        return value  # Regular list without DataClasses
    elif isinstance(value, dict):
        # Check if the target type is a DataClass
        if hasattr(target_type, "__annotations__"):
            field_types = get_type_hints(target_type)
            return target_type(
                **{
                    key: _deserializeMetaDataObject(value[key], field_types[key])
                    for key in value
                    if key in field_types
                }
            )
        return value  # Regular dictionary
    return value  # Primitive values


def _generic_metadata_create_or_update(
    config: Config,
    projectname: str,
    objects: List[T],
    endpoint: str,
    get_existing_func,
) -> None:
    """
    Generic function to create or update metadata entries.

    :param config: Configuration containing connection details
    :param projectname: Name of the project
    :param objects: List of objects to create or update
    :param endpoint: API endpoint (e.g., "Tiles" or "Pages")
    :param get_existing_func: Function to check if an object already exists
    """

    # Initialize request
    headers = config.connection_get_headers()
    project_id = getProjectID(config, projectname)

    for obj in objects:
        logging.info(f"Create/update {endpoint} '{obj.displayName}'")

        obj.tenant = config.get_tenant()
        obj.projectId = project_id

        # Check if the object already exists
        existing_object = get_existing_func(
            config=config,
            projectname=projectname,
            filter=obj.internalName,
            filter_type=FilterType.EQUAL,
            filter_value=FilterValue.INTERNALNAME,
        )

        if len(existing_object) == 1:
            # Update existing object
            obj.id = existing_object[0].id
            response = requests.put(
                f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}/{obj.id}",
                json=obj.to_dict(),
                headers=headers,
            )
            if response.status_code != 200:
                log_error(
                    f"Request failed.\nURL: {f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}/{obj.id}",}\nStatus: {response.status_code}, error: {response.text}"
                )

        else:
            # Create new object
            response = requests.post(
                f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}",
                json=obj.to_dict(),
                headers=headers,
            )
            if response.status_code != 201:
                log_error(
                    f"Request failed.\nURL: {f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}"}\nStatus: {response.status_code}, error: {response.text}"
                )


def _generic_metadata_delete(config: Config, ids: List[str], endpoint: str) -> None:
    """
    Generic function to delete metadata entries.

    :param config: Configuration containing connection details
    :param ids: List of IDs to be deleted
    :param endpoint: API endpoint (e.g., "Metrics" or "Columns")
    """

    # Initialize request
    headers = config.connection_get_headers()

    for obj_id in ids:
        logging.info(f"Deleting {endpoint[:-1]} with ID {obj_id}")

        response = requests.delete(
            f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}/{obj_id}",
            headers=headers,
        )

        if response.status_code != 204:
            log_error(
                f"Request failed.\nURL: {f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}/{obj_id}"}\nStatus: {response.status_code}, error: {response.text}"
            )


def _generic_metadata_get(
    config: Config,
    projectname: str,
    endpoint: str,
    endpoint_postfix: str,
    return_type: Type[T],
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[T]:
    """
    Generic method to fetch and filter metadata for different objects.

    :param config: Configuration containing connection details
    :param projectname: Name of the project
    :param endpoint: API endpoint (e.g., "Tiles" or "Pages")
    :param return_type: The class of the returned object (Tile or Page)
    :param filter: Filter value for searching
    :param filter_type: Type of filter (EQUAL, STARTSWITH, etc.)
    :param filter_value: The attribute to filter on (e.g., DISPLAYNAME)
    :return: A list of objects of the specified return_type
    """

    # Initialize request
    headers = config.connection_get_headers()
    project_id = getProjectID(config, projectname)

    response = requests.get(
        f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}/project/{project_id}{endpoint_postfix}",
        headers=headers,
    )

    if response.status_code != 200:
        log_error(
            f"Request failed.\nURL:{f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/{endpoint}/project/{project_id}{endpoint_postfix}"}\nStatus: {response.status_code}, error: {response.text}"
        )
        return []

    data = json.loads(response.text)

    def match_filter(value: str, filter: str, filter_type: FilterType) -> bool:
        """Applies the given filter to the value."""
        if filter == "*":
            return True
        elif filter_type == FilterType.EQUAL:
            return value == filter
        elif filter_type == FilterType.STARTSWITH:
            return value.startswith(filter)
        elif filter_type == FilterType.ENDSWITH:
            return value.endswith(filter)
        elif filter_type == FilterType.CONTAINS:
            return filter in value
        elif filter_type == FilterType.REGEX:
            return re.search(filter, value) is not None
        return False

    # Apply filter to the data
    filtered_data = [
        item
        for item in data
        if match_filter(item.get(filter_value.value, ""), filter, filter_type)
    ]

    # Clean metadata and return the list of objects
    cleaned_data = clean_meta_data(filtered_data)
    return [_deserializeMetaDataObject(item, return_type) for item in cleaned_data]


def getProjectID(
    config: Config,
    projectname: str,
) -> str:
    """
    Retrieves the unique project ID for a given project name.

    Args:
        config (Config): Configuration object containing connection details.
        projectname (str): The name of the project for which to retrieve the ID.

    Returns:
        str: The unique identifier (ID) of the specified project.

    Raises:
        ValueError: If the project name cannot be uniquely identified in the project list.

    Notes:
        - This function relies on the `getProjects` function to fetch the full project list.
        - If multiple or no entries match the given project name, an error is logged, and the first matching ID is returned.
    """
    projects = getProjects(
        config,
        filter=projectname,
        filter_type=FilterType.EQUAL,
        filter_value=FilterValue.DISPLAYNAME,
    )
    if len(projects) != 1:
        return None

    return projects[0].id


def getAttributeGroups(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[AttributeGroup]:
    """Fetches AttributeGroups metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "AttributeGroup",
        "/attributegroups",
        AttributeGroup,
        filter,
        filter_type,
        filter_value,
    )


def getMetrics(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Metric]:
    """Fetches Metrics metadata with the given filters."""
    return _generic_metadata_get(
        config, projectname, "Metrics", "", Metric, filter, filter_type, filter_value
    )


def getTiles(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Tile]:
    """Fetches Tiles metadata with the given filters."""
    return _generic_metadata_get(
        config, projectname, "Tiles", "", Tile, filter, filter_type, filter_value
    )


def getPages(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Page]:
    """Fetches Pages metadata with the given filters."""
    return _generic_metadata_get(
        config, projectname, "Pages", "", Page, filter, filter_type, filter_value
    )


def getApplications(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Application]:
    """Fetches Applications metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "Applications",
        "",
        Application,
        filter,
        filter_type,
        filter_value,
    )


def getDiagrams(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Diagram]:
    """Fetches Diagrams metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "Diagrams",
        "",
        Diagram,
        filter,
        filter_type,
        filter_value,
    )


def getDefinedColumns(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[DefinedColumn]:
    """Fetches DefinedColumns metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "Columns",
        "/defined",
        DefinedColumn,
        filter,
        filter_type,
        filter_value,
    )


def getImportedColumns(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[ImportedColumn]:
    """Fetches ImportedColumns metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "Columns",
        "/exported",
        ImportedColumn,
        filter,
        filter_type,
        filter_value,
    )


def getSubProcesses(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[SubProcess]:
    """Fetches SubProcesss metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "SubProcess",
        "/subprocesses",
        SubProcess,
        filter,
        filter_type,
        filter_value,
    )


def deleteDefinedColumns(config: Config, definedcolumns: List[str]) -> None:
    """Deletes a list of Defined Columns by their IDs."""
    _generic_metadata_delete(config, definedcolumns, "Columns")


def deleteImportedColumns(config: Config, importedcolumns: List[str]) -> None:
    """Deletes a list of Imported Columns by their IDs."""
    _generic_metadata_delete(config, importedcolumns, "Columns")


def deleteMetrics(config: Config, metrics: List[str]) -> None:
    """Deletes a list of Metrics by their IDs."""
    _generic_metadata_delete(config, metrics, "Metrics")


def deleteTiles(config: Config, tiles: List[str]) -> None:
    """Deletes a list of Tiles by their IDs."""
    _generic_metadata_delete(config, tiles, "Tiles")


def deleteAttributeGroups(config: Config, attributegroups: List[str]) -> None:
    """Deletes a list of AttributeGroups by their IDs."""
    _generic_metadata_delete(config, attributegroups, "AttributeGroup")


def deletePages(config: Config, pages: List[str]) -> None:
    """Deletes a list of Pages by their IDs."""
    _generic_metadata_delete(config, pages, "Pages")


def deleteApplications(config: Config, applications: List[str]) -> None:
    """Deletes a list of Pages by their IDs."""
    _generic_metadata_delete(config, applications, "Applications")


def deleteDiagrams(config: Config, diagrams: List[str]) -> None:
    """Deletes a list of Diagrams by their IDs."""
    _generic_metadata_delete(config, diagrams, "Diagrams")


def deleteSubprocesses(config: Config, subprocesses: List[str]) -> None:
    """Deletes a list of SubProcesses by their IDs."""
    _generic_metadata_delete(config, subprocesses, "SubProcess")


def createDefinedColumns(
    config: Config, projectname: str, definedcolumns: List[DefinedColumn]
) -> None:
    """Creates or updates a list of DefinedColumns."""
    _generic_metadata_create_or_update(
        config, projectname, definedcolumns, "Columns", getDefinedColumns
    )


def createImportedColumns(
    config: Config, projectname: str, importedcolumns: List[ImportedColumn]
) -> None:
    """Creates or updates a list of ImportedColumns."""
    _generic_metadata_create_or_update(
        config, projectname, importedcolumns, "Columns", getImportedColumns
    )


def createMetrics(config: Config, projectname: str, metrics: List[Metric]) -> None:
    """Creates or updates a list of Metrics."""
    _generic_metadata_create_or_update(
        config, projectname, metrics, "Metrics", getMetrics
    )


def createTiles(config: Config, projectname: str, tiles: List[Tile]) -> None:
    """Creates or updates a list of Tiles."""
    _generic_metadata_create_or_update(config, projectname, tiles, "Tiles", getTiles)


def createAttributeGroups(
    config: Config, projectname: str, attributegroups: List[AttributeGroup]
) -> None:
    """Creates or updates a list of AttributeGroups."""
    _generic_metadata_create_or_update(
        config, projectname, attributegroups, "AttributeGroup", getAttributeGroups
    )


def createPages(config: Config, projectname: str, pages: List[Page]) -> None:
    """Creates or updates a list of Pages."""
    _generic_metadata_create_or_update(config, projectname, pages, "Pages", getPages)


def createApplications(
    config: Config, projectname: str, applications: List[Application]
) -> None:
    """Creates or updates a list of Applications."""
    _generic_metadata_create_or_update(
        config, projectname, applications, "Applications", getApplications
    )


def createDiagrams(config: Config, projectname: str, diagrams: List[Diagram]) -> None:
    """Creates or updates a list of Diagrams."""
    _generic_metadata_create_or_update(
        config, projectname, diagrams, "Diagrams", getDiagrams
    )


def createSubProcesses(
    config: Config, projectname: str, subprocesses: List[SubProcess]
) -> None:
    """Creates or updates a list of SubProcesses."""
    _generic_metadata_create_or_update(
        config, projectname, subprocesses, "SubProcess", getSubProcesses
    )


def getProjects(
    config: Config,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Project]:
    """Fetches Projects metadata with the given filters."""

    # cannot use the generic meta data getter, since this is "above" the other object

    headers = config.connection_get_headers()

    response = requests.get(
        config.get_config_nemo_url() + "/api/nemo-projects/projects", headers=headers
    )
    if response.status_code != 200:
        log_error(
            f"request failed. Status: {response.status_code}, error: {response.text}"
        )
    data = json.loads(response.text)

    def match_filter(value: str, filter: str, filter_type: FilterType) -> bool:
        """Applies the given filter to the value."""
        if filter == "*":
            return True
        elif filter_type == FilterType.EQUAL:
            return value == filter
        elif filter_type == FilterType.STARTSWITH:
            return value.startswith(filter)
        elif filter_type == FilterType.ENDSWITH:
            return value.endswith(filter)
        elif filter_type == FilterType.CONTAINS:
            return filter in value
        elif filter_type == FilterType.REGEX:
            return re.search(filter, value) is not None
        return False

    # Apply filter to the data
    filtered_data = [
        item
        for item in data
        if match_filter(item.get(filter_value.value, ""), filter, filter_type)
    ]

    # Clean metadata and return the list of objects
    cleaned_data = clean_meta_data(filtered_data)
    return [_deserializeMetaDataObject(item, Project) for item in cleaned_data]


def deleteProjects(config: Config, projects: List[str]) -> None:
    """Deletes a list of projects by their IDs."""
    _generic_metadata_delete(config, projects, "Project")


def createProjects(config: Config, projects: List[Project]) -> None:
    """Creates or updates a list of Projects."""
    # Initialize request
    headers = config.connection_get_headers()

    for project in projects:
        logging.info(f"Create/update Project '{project.displayName}'")

        project.tenant = config.get_tenant()

        # Check if the object already exists
        existing_object = getProjects(
            config=config,
            filter=project.displayName,
            filter_type=FilterType.EQUAL,
            filter_value=FilterValue.DISPLAYNAME,
        )

        if len(existing_object) == 1:
            # Update existing object
            project.id = existing_object[0].id
            response = requests.put(
                f"{config.get_config_nemo_url()}/api/nemo-projects/projects/{project.id}",
                json=project.to_dict(),
                headers=headers,
            )
            if response.status_code != 200:
                log_error(
                    f"Request failed.\nURL: {f"{config.get_config_nemo_url()}/api/nemo-projects/projects/{project.id}"}\nStatus: {response.status_code}, error: {response.text}"
                )

        else:
            # Create new object
            response = requests.post(
                f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/Project",
                json=project.to_dict(),
                headers=headers,
            )
            if response.status_code != 201:
                log_error(
                    f"Request failed.\nURL: {f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/Project"}\nStatus: {response.status_code}, error: {response.text}"
                )


def getDependencyTree(config: Config, id: str) -> DependencyTree:
    # Initialize request
    headers = config.connection_get_headers()
    data = {"id": id}

    response = requests.get(
        f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/Metrics/DependencyTree",
        headers=headers,
        params=data,
    )

    if response.status_code != 200:
        log_error(
            f"{config.get_config_nemo_url()}/api/nemo-persistence/metadata/Metrics/DependencyTree",
        )
        return None

    data = json.loads(response.text)
    return DependencyTree.from_dict(data)


def getReports(
    config: Config,
    projectname: str,
    filter: str = "*",
    filter_type: FilterType = FilterType.STARTSWITH,
    filter_value: FilterValue = FilterValue.DISPLAYNAME,
) -> List[Report]:
    """Fetches Reports metadata with the given filters."""
    return _generic_metadata_get(
        config,
        projectname,
        "Reports",
        "/reports",
        Report,
        filter,
        filter_type,
        filter_value,
    )


def createReports(config: Config, projectname: str, reports: List[Report]) -> None:
    """Creates or updates a list of Reports."""
    _generic_metadata_create_or_update(
        config, projectname, reports, "Reports", getReports
    )


def deleteReports(config: Config, reports: List[str]) -> None:
    """Deletes a list of Reports by their IDs."""
    _generic_metadata_delete(config, reports, "Reports")
