from unittest.mock import patch

import pandas as pd
import pytest
from featuretools import EntitySet, Feature, calculate_feature_matrix, dfs

from evalml.automl import AutoMLSearch
from evalml.preprocessing import TimeSeriesSplit, split_data
from evalml.problem_types import ProblemTypes

PERIODS = 500


@pytest.mark.parametrize(
    "problem_type",
    [
        ProblemTypes.TIME_SERIES_BINARY,
        ProblemTypes.TIME_SERIES_MULTICLASS,
        ProblemTypes.TIME_SERIES_REGRESSION,
    ],
)
def test_can_run_automl_for_time_series_with_categorical_and_boolean_features(
    problem_type,
):
    X = pd.DataFrame(
        {
            "features": range(101, 101 + PERIODS),
            "date": pd.date_range("2010-10-01", periods=PERIODS),
        },
    )
    y = pd.Series(range(PERIODS))
    if problem_type == ProblemTypes.TIME_SERIES_BINARY:
        y = y % 2
    elif problem_type == ProblemTypes.TIME_SERIES_MULTICLASS:
        y = y % 3

    X.ww.init()
    X.ww["bool_feature"] = (
        pd.Series([True, False])
        .sample(n=X.shape[0], replace=True)
        .reset_index(drop=True)
    )
    X.ww["cat_feature"] = (
        pd.Series(["a", "b", "c"])
        .sample(n=X.shape[0], replace=True)
        .reset_index(drop=True)
    )

    automl = AutoMLSearch(
        X,
        y,
        problem_type=problem_type,
        problem_configuration={
            "max_delay": 5,
            "gap": 3,
            "forecast_horizon": 3,
            "time_index": "date",
        },
        optimize_thresholds=False,
        data_splitter=TimeSeriesSplit(
            forecast_horizon=3,
            gap=3,
            max_delay=3,
            n_splits=3,
        ),
    )
    automl.search()
    automl.best_pipeline.fit(X, y)
    X_valid = pd.DataFrame(
        {
            "date": pd.date_range(
                pd.Timestamp(X.date.iloc[-1]) + pd.Timedelta("4d"),
                periods=2,
            ),
        },
    )
    # Treat all features as not known-in-advanced
    automl.best_pipeline.predict(X_valid, X_train=X, y_train=y)


@pytest.mark.parametrize("sampler", ["Oversampler", "Undersampler"])
@pytest.mark.parametrize(
    "problem_type",
    [
        ProblemTypes.TIME_SERIES_BINARY,
        ProblemTypes.TIME_SERIES_MULTICLASS,
        ProblemTypes.TIME_SERIES_REGRESSION,
    ],
)
def test_can_run_automl_for_time_series_known_in_advance(
    problem_type,
    sampler,
):
    X = pd.DataFrame(
        {
            "features": range(101, 101 + PERIODS),
            "date": pd.date_range("2010-10-01", periods=PERIODS),
        },
    )
    y = pd.Series(range(PERIODS))
    if problem_type == ProblemTypes.TIME_SERIES_BINARY:
        # So that we have coverage for sampling
        y = (
            pd.Series([1] * 50 + [0] * 450)
            .sample(frac=1, random_state=0, replace=False)
            .reset_index(drop=True)
        )
    elif problem_type == ProblemTypes.TIME_SERIES_MULTICLASS:
        y = y % 3

    X.ww.init()
    X.ww["bool_feature"] = (
        pd.Series([True, False])
        .sample(n=X.shape[0], replace=True)
        .reset_index(drop=True)
    )
    X.ww["cat_feature"] = (
        pd.Series(["a", "b", "c"])
        .sample(n=X.shape[0], replace=True)
        .reset_index(drop=True)
    )

    automl = AutoMLSearch(
        X,
        y,
        problem_type=problem_type,
        problem_configuration={
            "max_delay": 5,
            "gap": 3,
            "forecast_horizon": 3,
            "time_index": "date",
            "known_in_advance": ["bool_feature", "cat_feature"],
        },
        optimize_thresholds=False,
        sampler_method=sampler,
        data_splitter=TimeSeriesSplit(
            forecast_horizon=3,
            gap=3,
            max_delay=3,
            n_splits=3,
        ),
    )
    automl.search()
    X_valid = pd.DataFrame(
        {
            "date": pd.date_range(
                pd.Timestamp(X.date.iloc[-1]) + pd.Timedelta("4d"),
                periods=2,
            ),
            "bool_feature": [True, False],
            "cat_feature": ["a", "c"],
        },
    )
    # Treat all features as not known-in-advanced
    automl.best_pipeline.predict(X_valid, X_train=X, y_train=y)


@patch(
    "evalml.pipelines.time_series_pipeline_base.TimeSeriesPipelineBase._add_training_data_to_X_Y",
)
@pytest.mark.parametrize(
    "problem_type",
    [
        ProblemTypes.TIME_SERIES_BINARY,
        ProblemTypes.TIME_SERIES_MULTICLASS,
        ProblemTypes.TIME_SERIES_REGRESSION,
    ],
)
def test_can_run_automl_for_time_series_with_exclude_featurizers(
    mock_add_X_y,
    problem_type,
):
    X = pd.DataFrame(
        {
            "features": range(101, 101 + PERIODS),
            "date": pd.date_range("2010-10-01", periods=PERIODS),
        },
    )
    y = pd.Series(range(PERIODS))
    if problem_type == ProblemTypes.TIME_SERIES_BINARY:
        # So that we have coverage for sampling
        y = (
            pd.Series([1] * 50 + [0] * 450)
            .sample(frac=1, random_state=0, replace=False)
            .reset_index(drop=True)
        )
    elif problem_type == ProblemTypes.TIME_SERIES_MULTICLASS:
        y = y % 3

    es = EntitySet()
    es.add_dataframe(dataframe_name="X", dataframe=X, index="id", make_index=True)
    features = dfs(
        entityset=es,
        target_dataframe_name="X",
        max_depth=1,
        features_only=True,
    )
    # time index must be included in input data
    features.append(Feature(es["X"].ww["date"]))
    feature_matrix = calculate_feature_matrix(entityset=es, features=features)
    # target lagged features must be present with correct start delay (gap + forecast horizon)
    feature_matrix.ww["target_delay_6"] = y.shift(6)
    feature_matrix.ww.set_time_index("date")

    problem_config = {
        "max_delay": 5,
        "gap": 3,
        "forecast_horizon": 3,
        "time_index": "date",
    }
    X_train, X_test, y_train, y_test = split_data(
        feature_matrix,
        y,
        problem_type=problem_type,
        problem_configuration=problem_config,
    )

    automl = AutoMLSearch(
        X_train,
        y_train,
        problem_type=problem_type,
        problem_configuration=problem_config,
        optimize_thresholds=False,
        exclude_featurizers=["DatetimeFeaturizer", "TimeSeriesFeaturizer"],
    )
    automl.search()
    assert mock_add_X_y.call_count == 0

    rankings = automl.rankings
    for score in rankings["ranking_score"].values:
        assert pd.notnull(score)

    num_pipelines = automl._num_pipelines()
    for pipeline_number in range(num_pipelines):
        pipeline = automl.get_pipeline(pipeline_number)
        if pipeline.estimator.name in ["ARIMA Regressor", "Prophet Regressor"]:
            assert not pipeline.should_drop_time_index

            # Make sure we can still predict on these estimators
            pipeline.fit(X_train, y_train)
            pipeline.predict_in_sample(X_test, y_test, X_train, y_train)

        else:
            assert pipeline.should_drop_time_index
        assert pipeline.should_skip_featurization
