"""Definition of measurements."""
from typing import List

from pymetadata.identifiers.miriam import BQB

from ..node import Choice, DType, InfoNode, MeasurementType
from ..units import (
    AMOUNT_UNITS,
    AUC_DIV_UNITS,
    AUC_UNITS,
    AUMC_UNITS,
    CLEARANCE_UNITS,
    CONCENTRATION_PER_DOSE_UNITS,
    CONCENTRATION_UNITS,
    DIMENSIONLESS,
    FLOW_UNITS,
    NO_UNIT,
    PRESSURE_UNITS,
    RATE_UNITS,
    RATIO_UNITS,
    TIME_UNITS,
    VD_UNITS,
)


MEASUREMENT_NODES: List[InfoNode] = [
    MeasurementType(
        "blood measurement",
        description="measurement of the blood, it's contents, cells or other factors "
        "contained within the blood.",
        parents=["measurement"],
        dtype=DType.ABSTRACT,
        annotations=[(BQB.IS, "cmo/CMO:0000035")],
    ),
    MeasurementType(
        "blood cell measurement",
        description="A value resulting from the quantification of a morphological "
        "or physiological parameter of blood cells, i.e. cells native to the "
        "circulation, including those of erythroid, lymphoid, myeloid "
        "and monocytic lineages. A cell is a membrane-enclosed protoplasmic "
        "mass constituting the smallest structural unit of an organism that "
        "is capable of independent functioning.",
        parents=["blood measurement"],
        dtype=DType.ABSTRACT,
        annotations=[(BQB.IS, "cmo/CMO:0000036")],
    ),
    MeasurementType(
        "chemistry measurement",
        description="A quantification of a parameter of the "
        "chemical composition of a tissue or compartment. "
        "If used in a group or individual characteristica the tissue is "
        "blood, i.e., this refers to blood chemistry measurement. If measured "
        "in different tissue, e.g. urine it must be encoded as output.",
        parents=["measurement"],
        dtype=DType.ABSTRACT,
        annotations=[],
    ),
    MeasurementType(
        "liver function test",
        description="Statical liver function tests (e.g., standard panel of liver "
        "enzymes, i.e., ALT and AST). The quantitative measurement of liver "
        "enzymes in the blood, which is useful in assessing the functional "
        "state of a patient's liver. Liver function tests include the "
        "measurement of albumin, alanine transaminase, aspartate "
        "transaminase, alkaline phosphatase, bilirubin, and gamma "
        "glutamyl transferase."
        "To encode numerical results for the individual enzymes use the "
        "`concentration` "
        "or `amount` with the respective substance. Also used for general "
        "reporting of liver function or hepatic function.",
        parents=["chemistry measurement"],
        dtype=DType.CATEGORICAL,
        annotations=[
            (BQB.IS, "ncit/C74954"),
        ],
        synonyms=["liver function", "hepatic function"],
    ),
    MeasurementType(
        sid="breath_test",
        name="breath test",
        description="Breath test. Any test performed on air generated by the act of exhaling.",
        parents=["chemistry measurement"],
        dtype=DType.CATEGORICAL,
        annotations=[
            (BQB.IS, "ncit/C116515"),
        ],
        synonyms=[],
    ),
    MeasurementType(
        "concentration",
        description="Concentration of substance. The quantity of a substance per unit "
        "volume or weight;"
        "See also `amount` for amount of substance, "
        "`cumulative amount` for cumulative amount, and "
        "`concentration unbound` for "
        "the unbound, i.e. free, concentration of the substance.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_UNITS,
        annotations=[
            (BQB.IS, "ncit/C41185"),
            (BQB.IS, "chmo/CHMO:0002820"),
        ],
    ),
    MeasurementType(
        "concentration_per_dose",
        name="concentration per dose",
        description="Concentration divided by dose. Dose-normalized concentration.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_PER_DOSE_UNITS,
        synonyms=["dose-normalized concentration"],
    ),
    MeasurementType(
        sid="concentration-change",
        name="concentration change",
        label="concentration change",
        description="Concentration change of substance from baseline. Often only changes "
        "from baseline are reported for concentrations. If a baseline value is "
        "available the absolute change in concentration should be calculated and "
        "encoded as concentration."
        "See also `concentration`.",
        parents=["concentration"],
        dtype=DType.NUMERIC,
        units=[DIMENSIONLESS],
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C41185"),
            (BQB.IS_VERSION_OF, "chmo/CHMO:0002820"),
        ],
    ),
    MeasurementType(
        "multiple-dosing-cmin",
        name="muliple_dosing_cmin",
        label="multiple dosing minimal concentration",
        description="Minimal concentration of substance in quasi steady state after multiple dosing interventions. "
        "This depends on the dosing scheme. See also 'multiple_dosing_cmax'.",
        parents=["concentration"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_UNITS,
        annotations=[],
        synonyms=["css", "Css", "steady-state minimal concentration"],
    ),
    MeasurementType(
        "multiple-dosing-cmax",
        name="muliple_dosing_cmax",
        label="multiple dosing maximal concentration",
        description="Maximal concentration of substance in quasi steady state after multiple dosing interventions. "
        "This depends on the dosing scheme. See also 'multiple_dosing_cmin'.",
        parents=["concentration"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_UNITS,
        annotations=[],
        synonyms=["css", "Css", "steady-state maximal concentration"],
    ),
    MeasurementType(
        "concentration-ss",
        name="concentration_ss",
        label="steady-state concentration",
        description="Concentration of substance in steady state. The quantity of a substance per unit "
        "volume or weight; steady state after infusion. In case of "
        "minimal concentration in quasi steady state in multiple dosing use 'multiple_dosing_cmin'.",
        parents=["concentration"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_UNITS,
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C41185"),
            (BQB.IS_VERSION_OF, "chmo/CHMO:0002820"),
        ],
        synonyms=["css", "Css", "steady-state concentration"],
    ),
    MeasurementType(
        "concentration_ss_per_dose",
        name="concentration_ss per dose",
        label="steady state concentration per dose",
        description="Steady state concentration divided by dose. Dose-normalized "
        "steady state concentration.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_PER_DOSE_UNITS,
        synonyms=["dose-normalized steady state concentration"],
    ),
    MeasurementType(
        "concentration-unbound",
        name="concentration_unbound",
        label="concentration unbound",
        description="Concentration of unbound substance. "
        "Describes the free unbound substance concentration, i.e., "
        "concentration of the free substance. "
        "See also `concentration`, `amount` or `fraction_unbound`.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=CONCENTRATION_UNITS,
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C41185"),
            (BQB.IS_VERSION_OF, "chmo/CHMO:0002820"),
        ],
    ),
    MeasurementType(
        "amount",
        description="Amount of substance. If substance is measured in amount units. See also 'concentration'.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=AMOUNT_UNITS,
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C25256"),  # quantity
        ],
    ),
    MeasurementType(
        sid="cumulative-amount",
        name="cumulative amount",
        label="cumulative amount",
        description="Cumulative amount of substance. It is required to provide the time "
        "period for the cumulative collection. Cumulative amount is often "
        "used for urinary measurements in which urine is collected for a "
        "given time period (see also 'recovery').",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=AMOUNT_UNITS,
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C25256"),  # quantity
        ],
    ),
    MeasurementType(
        sid="excretion-rate",
        name="excretion rate",
        label="excretion rate",
        description="Excretion rate of substance. The speed relative to time that an exogenous compound is removed "
        "from a body system."
        "Rate of removal of substance from body for instance via urinary "
        "excretion, excretion via the fecal route or via sweat. Use with the respective tissue to encode the route of "
        "excretion, e.g. with the tissue urine. For secretion within the body use 'secretion rate'.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=RATE_UNITS,
        annotations=[
            (BQB.IS, "ncit/C85534"),
        ],
    ),
    MeasurementType(
        sid="maximum-excretion-rate",
        name="maximum excretion rate",
        label="maximum excretion rate (ERmax)",
        description="Maximum excretion rate (ERmax). See also 'excretion rate'",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=RATE_UNITS,
        annotations=[],
    ),
    MeasurementType(
        sid="urinary-volume-rate",
        name="urinary volume rate",
        description="Rate of urinary volume excretion, i.e., volume of urine per time. Should be used "
        "with substance 'h2o'.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=["ml/hr", "ml/hr/(1.73*m^2)"],
        annotations=[],
        synonyms=["urinary flow rate"],
    ),
    MeasurementType(
        "dob",
        label="delta over baseline (DOB)",
        description="Delta over baseline. The ratio is reported in units delta13 CO2 "
        "relative to the PDB (Belemnitella americana) standard (1.1235% 13C) "
        "and is defined as 'delta13C(o/oo) = [(13C/12C) sample / (13C/12C) standard) -1] *1000'."
        "The value of the baseline was subtracted from each value resulting"
        "in a net change in delta (delta over baseline; DOB). DOB can be "
        "converted to an equivalent excess spepcific activity (excess 13CO2/total CO2)"
        "by multiplying DOB by 1.123 E-5.",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=[DIMENSIONLESS],
        annotations=[],
    ),
    MeasurementType(
        sid="pdr",
        name="pdr",
        label="plasma disappearance rate (PDR)",
        description="plasma disappearance rate (PDR). Calculated from backward extrapolation of k from fitted curve."
        "E.g. ICG-PDR",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=["percent/min"],
        annotations=[],
    ),
    # FIXME: this must be handled via formulas
    MeasurementType(
        sid="fractional_pdr",
        name="fractional pdr",
        label="fractional plasma disappearance rate (PDR)",
        description="Plasma disappearance rate (PDR) relative to reference value (e.g., "
        "post-operative PDR/pre-operative PDR.",
        parents=["pdr"],
        dtype=DType.NUMERIC,
        units=[DIMENSIONLESS],
        annotations=[],
    ),
    MeasurementType(
        "retention ratio",
        description="Retention ratio. E.g. ICG-R15, calculated as C_icg(15min)*C_icg(0)/100",
        parents=["chemistry measurement"],
        dtype=DType.NUMERIC,
        units=["percent"],
        annotations=[],
    ),
    MeasurementType(
        "biochemistry (categorial)",
        description="blood chemistry measurement (level); "
        "The substance is defined via the `substance` field on the "
        "measurement_type, e.g., "
        "to define that the blood insulin was elevated use "
        "`substance: insulin` and `choice: biochemistry elevated`. "
        "To specify the actual amount or concentration use the numerical types"
        "`concentration` and `amount`.",
        parents=["chemistry measurement"],
        dtype=DType.CATEGORICAL,
    ),
    MeasurementType(
        "kidney function test",
        description="Statical kidney function test(s) or renal function test (e.g., urea "
        "or creatinine clearance). A laboratory procedure that evaluates the "
        "kidney function."
        "To encode numerical results for the individual enzymes use the "
        "`concentration` "
        "or `amount` with the respective substance. Also used for general "
        "reporting of kidney function or renal function.",
        parents=["chemistry measurement"],
        dtype=DType.CATEGORICAL,
        annotations=[
            (BQB.IS, "ncit/C74972"),
        ],
        synonyms=["renal function", "kidney function"],
    ),
    MeasurementType(
        sid="abt",
        name="ABT",
        label="Aminopyrine breath test (ABT)",
        description="A breath test with 13C-labeled aminopyrine that has been used "
        "as a non-invasive measure of cytochrome P-450 metabolic activity "
        "in liver function tests. It is also used in measuring the total "
        "body water in the human body system. Production and use have been "
        "banned in France, Thailand and India. ABT is measured in  "
        "[% dose*kg/mmole].",
        parents=["liver function test", "breath test"],
        dtype=DType.NUMERIC,
        units=["percent*kg/mmole"],
        annotations=[],
        synonyms=["ABT"],
    ),
    MeasurementType(
        sid="ermbt",
        name="ermbt",
        label="Erythromycin breath test (ERMBT)",
        description="Erythromycin breath test. The erythromycin breath test (ERMBT) "
        "is a method used to measure metabolism (oxidation and elimination "
        "from the system) by a part of the cytochrome P450 system. "
        "Erythromycin is tagged with carbon-14 and given as an intravenous "
        "injection; after 20 minutes the subject blows up a balloon and the "
        "carbon dioxide exhaled that is tagged with carbon-14 shows the "
        "activity of the CYP3A4 isoenzyme on the erythromycin. "
        "Therefore, this activity can predict how other drugs that are "
        "metabolized by the CYP3A4 part of the cytochrome P450 system will "
        "be acted upon by a particular person. ",
        parents=["liver function test", "breath test"],
        dtype=DType.NUMERIC,
        units=["percent/hr"],
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C116515"),  # breath test
            (BQB.HAS_PART, "ncit/C476"),  # erythromycin
        ],
        synonyms=["EBT"],
    ),
    MeasurementType(
        sid="glomular-filtration-rate",
        name="gfr",
        label="Glomular filtration rate",
        description="Glomular Filtration Rate (GFR). An estimation of the volume of fluid "
        "filtered from the kidney glomerular capillaries into the Bowman's "
        "capsule per specified unit of time based on serum creatinine level, "
        "weight and age of organism and other factors. GFR should be encoded "
        "as creatinine via 'measurement_type=clearance', "
        "'substance=creatinine' and 'tissue=urine' if it is a creatinine "
        "clearance.",
        parents=["kidney function test"],
        dtype=DType.NUMERIC,
        synonyms=["creatinine clearance"],
        annotations=[
            (BQB.IS, "cmo/CMO:0000490"),
            (BQB.IS, "ncit/C90505"),
        ],
        units=["ml/min/(1.73*m^2)", "ml/min"],
    ),
    MeasurementType(
        sid="prothrombin-time-ratio",
        name="prothrombin time ratio",
        description="Prothrombin time ratio. Measurement of the clotting ability of "
        "fibrinogen, prothrombin, proaccelerin, proconvertin and Stuart "
        "factor, usually given in seconds to formation of clot after the "
        "addition of a tissue factor or thromboplastin.",
        parents=["liver function test"],
        dtype=DType.NUMERIC,
        units=[DIMENSIONLESS],
        annotations=[
            (BQB.IS_VERSION_OF, "cmo/CMO:0000211"),
            (BQB.IS_VERSION_OF, "ncit/C62656"),
        ],
    ),
    MeasurementType(
        sid="plasma-thromboplastin-activity",
        name="plasma thromboplastin activity",
        description="Plasma thromboplastin activity. A test for the functional "
        "intactness of the prothrombin complex that is used in "
        "controlling the amount of anticoagulant used in preventing thrombosis",
        parents=["liver function test"],
        synonyms=["Thrombotest"],
        dtype=DType.NUMERIC,
        units=[DIMENSIONLESS],
        annotations=[],
    ),
    MeasurementType(
        sid="child-pugh-score",
        name="child-pugh score",
        label="Child-Pugh score",
        description="Child-Pugh Classification. Child-Pugh score indicating one-year "
        "survival of 100% in patients with chronic liver disease and "
        "cirrhosis. This score is determined by the study of the following "
        "five factors: bilirubin, albumin, international normalized ratio, "
        "presence and degree of ascites, and presence and degree of "
        "encephalopathy.",
        parents=["liver function test"],
        units=[DIMENSIONLESS, NO_UNIT],
        dtype=DType.CATEGORICAL,
        annotations=[
            (BQB.IS, "ncit/C121007"),
        ],
    ),
    MeasurementType(
        sid="child-pugh-score-numeric",
        name="child-pugh score numeric",
        label="Child-Pugh score (numeric)",
        description="Child-Pugh Classification. Child-Pugh score indicating one-year "
        "survival of 100% in patients with chronic liver disease and "
        "cirrhosis. This score is determined by the study of the following "
        "five factors: bilirubin, albumin, international normalized ratio, "
        "presence and degree of ascites, and presence and degree of "
        "encephalopathy. Numeric values must be smaller or equal 15.",
        parents=["liver function test"],
        units=[DIMENSIONLESS],
        dtype=DType.NUMERIC,
        annotations=[
            (BQB.IS, "ncit/C121007"),
        ],
    ),
    Choice(
        "child-pugh class A",
        label="Child-Pugh class A",
        description="Child-Pugh Classification - Grade A (5-6 points) Well Compensated "
        "Disease, "
        "Child-Pugh Class A",
        parents=["child-pugh score"],
        annotations=[
            (BQB.IS, "ncit/C113691"),
        ],
    ),
    Choice(
        "child-pugh class B",
        label="Child-Pugh class B",
        description="Child-Pugh Classification - Child-Pugh Class B, Grade B (7-9 points)."
        "Significant Functional Compromise",
        parents=["child-pugh score"],
        annotations=[
            (BQB.IS, "ncit/C113692"),
        ],
    ),
    Choice(
        "child-pugh class C",
        label="Child-Pugh class C",
        description="Child-Pugh Class C, Grade C (10-15 points) Decompensated Disease.",
        parents=["child-pugh score"],
        annotations=[
            (BQB.IS, "ncit/C113694"),
        ],
    ),
    Choice(
        "child-pugh class unkown",
        label="Child-Pugh class unknown",
        description="Unknown: Could not be determined or unsure, Child-Pugh Class Unknown.",
        parents=["child-pugh score"],
        annotations=[
            (BQB.IS, "ncit/C159867"),
        ],
    ),
    MeasurementType(
        sid="inr",
        name="inr",
        label="international normalized ratio (INR)",
        description="International normalized ratio (INR).",
        parents=["liver function test"],
        dtype=DType.NUMERIC,
        annotations=[
            (BQB.IS_VERSION_OF, "cmo/CMO:0000211"),
            (BQB.IS_VERSION_OF, "ncit/C62656"),
        ],
        synonyms=["international normalized ratio"],
    ),
    MeasurementType(
        "prothrombin time",
        description="Prothrombin time. Measurement of the clotting ability of "
        "fibrinogen, prothrombin, proaccelerin, proconvertin and Stuart "
        "factor, usually given in seconds to formation of clot after the "
        "addition of a tissue factor or thromboplastin.",
        parents=["liver function test"],
        dtype=DType.NUMERIC,
        units=["sec"],
        annotations=[
            (BQB.IS, "cmo/CMO:0000211"),
            (BQB.IS, "ncit/C62656"),
        ],
    ),
    MeasurementType(
        "glycosylated hemoglobin measurement",
        name="hba1c",
        label="HbA1c (glycosylated haemoglobin)",
        description="HbA1c (glycosylated haemoglobin). A quantitative measurement of the "
        "amount of glycosylated hemoglobin present in a sample of blood.",
        parents=["blood cell measurement"],
        dtype=DType.NUMERIC,
        units=["percent"],
        annotations=[(BQB.IS, "efo/0009207"), (BQB.IS, "ncit/C64849")],
    ),
    MeasurementType(
        "hematocrit",
        description="Volume-Percentage of red blood cells in blood. The measurement depends on "
        "number and size of the red blood cells.",
        parents=["blood cell measurement"],
        dtype=DType.NUMERIC,
        units=["percent"],
        name="hematocrit",
        annotations=[],
    ),
    MeasurementType(
        sid="blood_cell_count_categorial",
        name="blood cell count (categorial)",
        description="Blood cell count. The determination of the number of red blood "
        "cells, white blood cells, and platelets in a biospecimen.",
        parents=["blood cell measurement"],
        dtype=DType.CATEGORICAL,
        annotations=[
            (BQB.IS, "ncit/C28133"),
        ],
        synonyms=[
            "cell count",
        ],
    ),
    MeasurementType(
        sid="cell_count_levels",
        name="cell count levels",
        description="cell count",
        parents=[
            "blood cell count (categorial)",
        ],
        dtype=DType.ABSTRACT,
        annotations=[
            (BQB.IS_VERSION_OF, "ncit/C28133"),
        ],
    ),
    Choice(
        sid="cell_count_normal",
        name="normal",
        label="cell count normal",
        description="Cell count in normal range.",
        parents=["cell count levels"],
        annotations=[],
    ),
    Choice(
        sid="cell_count_abnormal",
        name="abnormal",
        label="cell count abnormal",
        description="Cell count in abnormal range.",
        parents=["cell count levels"],
        annotations=[],
    ),
    Choice(
        sid="cell_count_elevated",
        name="elevated",
        label="cell count eleveated",
        description="Cell count in eleveated range.",
        parents=["cell count levels"],
        annotations=[],
    ),
    Choice(
        sid="cell_count_decreased",
        name="decreased",
        label="cell count decreased",
        description="Cell count in decreased range.",
        parents=["cell count levels"],
        annotations=[],
    ),
]

MEASUREMENT_NODES.extend(
    [
        MeasurementType(
            sid="pharmacokinetic measurement",
            description="Pharmacokinetic measurement",
            parents=["measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            "auc measurement",
            label="AUC measurement",
            description="Area under the curve (AUC), all pharmacokinetics paramters "
            "which are an area under the curve",
            parents=["pharmacokinetic measurement"],
            dtype=DType.ABSTRACT,
            units=AUC_UNITS,
        ),
        MeasurementType(
            "auc_inf",
            label="area under curve (AUC) infinity",
            description="Area under the curve (AUC), extrapolated until infinity.",
            parents=["auc measurement"],
            dtype=DType.NUMERIC,
            units=AUC_UNITS,
        ),
        MeasurementType(
            "auc_end",
            label="area under curve (AUC) end",
            description="Area under the curve (AUC), until last time point. "
            "Time period is required for calculation.",
            parents=["auc measurement"],
            dtype=DType.NUMERIC,
            units=AUC_UNITS,
        ),
        MeasurementType(
            "auc_relative",
            label="area under curve (AUC) relative",
            description="Relative area under the curve (AUC), AUC of a substance relative "
            "to other measured metabolites",
            parents=["auc measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            "auc_per_dose",
            label="area under curve (AUC) per dose",
            description="Area under the curve divided by dose (1/clearance). Dose-normalized area under the curve.",
            parents=["auc measurement"],
            dtype=DType.NUMERIC,
            units=AUC_DIV_UNITS,
            synonyms=["dose-normalized AUC"],
        ),
        MeasurementType(
            "aumc_inf",
            label="Area under first moment curve to infinity (AUMC)",
            description="Area under first moment curve (AUMC), extrapolated until infinity.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=AUMC_UNITS,
        ),
        MeasurementType(
            sid="auc_ratio",
            name="auc_ratio",
            label="AUC ratio",
            description="Ratio of arithmetic AUC. Use with substance ratios (i.e. derived substances such as px/caf)."
            "Often these ratios are given as geometric ratios and must be encoded accordingly.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=AUMC_UNITS,
        ),
        MeasurementType(
            "extraction_ratio",
            label="extraction ratio (ER)",
            description="Extraction ratio (ER).",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
            synonyms=["ER"],
        ),
        MeasurementType(
            "clearance measurement",
            description="clearance abstract class",
            parents=["pharmacokinetic measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            sid="clearance",
            label="clearance (Cl)",
            description="Total/apparent clearance of given substance (mostly from blood or "
            "plasma)."
            "This is the rate with which a volume is cleared of a given substance "
            "(see also rate of disappearance `ra`)"
            "If the clearance is based on the unbound substance use "
            "clearance_unbound. If the clearance refers to renal "
            "clearance use clearance_renal.",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_unbound",
            label="clearance unbound",
            description="Clearance of unbound substance, i.e., of the free substance"
            "not bound by proteins (see also `fraction_unbound`)",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_intrinsic",
            label="clearance intrinsic",
            description="Intrinsic clearance of substance. This is the total clearance minus "
            "the clearance by the kidneys "
            "(renal clearance). In most cases this is exclusively the liver if no "
            "other tissues are involved in the conversion/clearance.",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_intrinsic_unbound",
            label="clearance intrinsic unbound",
            description="Intrinsic clearance of unbound substance (see also "
            "`fraction_unbound`).",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_renal",
            label="clearance renal",
            description="Clearance of substance by the kidneys. For blood and plasma "
            "clearance and clearance by other tissues use `clearance` "
            "with respective tissue.",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_renal_unbound",
            label="clearance renal unbound",
            description="Clearance of unbound substance by the kidneys. For blood and "
            "plasma clearance and clearance"
            "by other tissues use `clearance` with respective tissue.",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_partial",
            label="clearance partial",
            description="Partial clearance of substance. It can occur if "
            "several path are present. The pathway is encoded "
            "by the substance",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "clearance_other",
            label="clearance other",
            description="Clearance of substance by other means not tested for in the study",
            parents=["clearance measurement"],
            dtype=DType.NUMERIC,
            units=CLEARANCE_UNITS,
        ),
        MeasurementType(
            "enantiomeric_ratio",
            label="enantiomeric ratio",
            description="Ratio of the S enantiomer to the R enantiomer (S/R)",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
        MeasurementType(
            "vd",
            label="volume of distribution",
            description="Volume of distribution. Calculated via beta from exponential decay "
            "phase. If the volume of distribution is calculated "
            "with the unbound substance use vd_unbound. If vd is in steady state"
            " use 'vd_ss' instead",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=VD_UNITS,
        ),
        MeasurementType(
            "vd_ss",
            label="volume of distribution steady state",
            description="Volume of distribution in steady state. If vd is calculated via beta "
            "from the exponential decay phase use 'vd' instead.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=VD_UNITS,
        ),
        MeasurementType(
            "vd_unbound",
            label="volume of distribution unbound",
            description="Volume of distribution for unbound substance.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=VD_UNITS,
        ),
        MeasurementType(
            "thalf",
            label="elimination half-life",
            description="Elimination half-life for substance.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            "tmax",
            label="time to maximum (Tmax)",
            description="Time to maximum; time to peak",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
            synonyms=["time to maximum", "Tmax", "Tpk", "time of peak concentration"],
        ),
        MeasurementType(
            "oro-cecal transit time",
            description="The transit time was taken as the time when breath hydrogen "
            "excretion increased to above twice the baseline value.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            "mrt",
            label="mean residence time (MRT)",
            description="Mean residence time (MRT). MRT = AUMC/AUC",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            "mat",
            label="mean absorption time (MAT)",
            description="Mean absorption time (MAT). Different methods for calculation exist. (i) MRT po - MRT iv + 1/2 T; T=duration of infusion; "
            "(ii) MAT was calculated as the difference between mean residence time (MRT) and the inverse of the slope of the "
            "terminal elimination phase.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            sid="cmax",
            name="cmax",
            label="maximum concentration (Cmax)",
            description="Maximum concentration for given substance.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=CONCENTRATION_UNITS,
            synonyms=["Cmax", "Cpk", "peak concentration"],
        ),
        MeasurementType(
            sid="cmin",
            name="cmin",
            label="minimal concentration (Cmin)",
            description="Minimal concentration for given substance. Importantly the "
            "minimum is after the peak concentration, and refers mostly to "
            "the last measured data point. See also 'cthrough' for multiple dosing.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=CONCENTRATION_UNITS,
            synonyms=["Cmin"],
        ),
        MeasurementType(
            sid="cthrough",
            name="cthrough",
            label="through concentration",
            description="Through concentration. Lowest concentration reached by a drug "
            "before the next dose is administered. See also 'cmin' for "
            "single dosing.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=CONCENTRATION_UNITS,
            synonyms=["Ctrhough", "Trough level", "Cmin"],
        ),
        MeasurementType(
            sid="caverage",
            name="caverage",
            label="average concentration",
            description="Average concentration. Calculated as AUC_enc/tau with"
            "tau being interval of measurement.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=CONCENTRATION_UNITS,
            annotations=[(BQB.IS_VERSION_OF, "ncit/C85575")],
            synonyms=["Caverage", "AUC/tau", "AUC/t"],
        ),
        MeasurementType(
            sid="ptf",
            name="ptf",
            label="percentage of peak-trough fluctuations (PTF)",
            description="Percentage of peak-trough fluctuations. A measurement of "
            "variability of concentration for Cmin and Cmax between dose time "
            "and Tau. "
            "Calculated as (Cmax-Cmin)/(AUC_end/tau) "
            "or (Cmax-Cmin)/(Caverage) with Caverage=AUC_end/tau.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
            annotations=[
                (BQB.IS, "ncit/C85581"),
            ],
            synonyms=["fluctuation index", "fluctuation", "fluctuations"],
        ),
        MeasurementType(
            "kel",
            label="elimination rate (Kel)",
            description="Elimination rate for given substance.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
            synonyms=["elimination rate constant"],
        ),
        MeasurementType(
            sid="kabs",
            label="absorption rate (Kabs)",
            description="Absorption rate for a given substance.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
        ),
        MeasurementType(
            sid="thalf_absorption",
            label="absorption half-life",
            description="Absorption half-life for substance.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            sid="lag_absorption",
            label="lag time of absorption",
            description="Lag time of absorption.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            sid="bioavailability",
            description="Bioavailability of given substance, i.e., which fraction was "
            "available in the blood/plasma after administration. This is "
            "often < 1, because only a fraction of the substance is absorbed "
            "(see 'fraction_absorbed'), due to first pass effects. AUC(po)/AUC(iv) or Ae(po)/Ae(iv).",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
            synonyms=[],
        ),
        MeasurementType(
            sid="bp_ratio",
            label="blood/plasma ratio (BP)",
            description="The ratio of blood concentration to plasma concentration of a "
            "given substance in the blood. Tissue should be coded as 'blood'",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            sid="fraction_absorbed",
            label="fraction absorbed",
            description="Fraction absorbed of given substance. This often corresponds to the "
            "bioavailability if there is no strong first pass effect (see also"
            "bioavailability).",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            sid="plasma_binding",
            label="plasma binding",
            description="Plasma binding of given substance (see also fraction unbound).",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            sid="fraction_unbound",
            label="fraction unbound",
            description="Unbound fraction of given substance (see also `plasma_binding`). Often"
            "abbreviated as `fu`.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            sid="recovery",
            description="Fraction recovered of given substance. Use in combination "
            "with the provided tissue to encode recovery for given tissue. "
            "E.g. `recovery` with tissue `urine` codes for the "
            "urinary recovery of given substance over the specified period "
            "of time. Often stored as percent excreted. To store "
            "the amount which was excreted use `cumulative amount` in "
            "combination with the respective tissue.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
            annotations=[
                (BQB.IS, "ncit/C70827"),
            ],
        ),
        MeasurementType(
            sid="recovery_rate",
            label="recovery rate",
            description="Fraction recovered of given substance per time. Use in combination "
            "with the provided tissue to encode recovery for given tissue. ",
            parents=["pharmacokinetic measurement"],
            dtype=DType.NUMERIC,
            units=["percent/min"],
            annotations=[],
        ),
        MeasurementType(
            sid="rate",
            description="A rate measurement, often in amount per time.",
            parents=["pharmacokinetic measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            sid="ra",
            label="Rate of appearance (Ra)",
            description="Rate of appearance.",
            parents=["rate"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
        ),
        MeasurementType(
            sid="egp",
            name="egp",
            label="endogeneous glucose production (EGP)",
            description="Endogenous glucose production (EGP). Production of glucose by the body, mainly via the liver, but "
            "also under fasting conditions small amounts by the kidneys.",
            parents=["ra"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
        ),
        MeasurementType(
            sid="secretion-rate",
            name="secretion rate",
            label="secretion rate",
            description="Rate of secretion. Rate of appearance of substance from a secretory tissue. "
            "E.g. insulin or glucagon secretion by the pancreas, or pepsin secretion in the stomach. "
            "For excretion outside of the body use 'excretion rate'.",
            parents=["ra"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
        ),
        MeasurementType(
            sid="rate cycling",
            label="Rate of cycling",
            description="Rate of cycling (i.e. net cycle flux).",
            parents=["rate"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
        ),
        MeasurementType(
            "rd",
            label="Rate of disappearance (Rd)",
            description="Rate of disappearance.",
            parents=["rate"],
            dtype=DType.NUMERIC,
            units=RATE_UNITS,
        ),
        MeasurementType(
            "blood index",
            description="Index based on blood measurements to characterize physiological "
            "status",
            parents=["blood measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            sid="glucose_index",
            name="glucose_index",
            label="Glucose index",
            description="Parameter, index based on blood measurements to characterize "
            "glucose metabolism.",
            parents=["blood index"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            "Matsuda index",
            description="Matsuda index",
            parents=["glucose_index"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
        MeasurementType(
            "QUICKI",
            description="quantitative insulin sensitivity check index",
            parents=["glucose_index"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
        MeasurementType(
            "HOMA-IR",
            description="homeostatic model assessment for insulin resistance",
            parents=["glucose_index"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
        MeasurementType(
            "Insulinogenic index",
            description="Insulinogenic index",
            parents=["glucose_index"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
        MeasurementType(
            "Glucose oral disposition index",
            description="Oral disposition index",
            parents=["glucose_index"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
    ]
)

MEASUREMENT_NODES.extend(
    [
        MeasurementType(
            "physiological measurement",
            description="physiological measurement",
            parents=["measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            "renal measurement",
            description="Any value resulting from the quantification of a morphological "
            "or physiological parameter pertaining to the kidneys or urine.",
            parents=["physiological measurement"],
            dtype=DType.ABSTRACT,
            annotations=[
                (BQB.IS, "cmo/CMO:0000080"),
                (BQB.IS, "efo/0004742"),
            ],
            synonyms=["renal system measurement"],
        ),
        MeasurementType(
            "urinary volume",
            description="The volume of the urine. Quantification of the amount of urine "
            "voided. Should be used with substance 'h2o'.",
            parents=["renal measurement"],
            dtype=DType.NUMERIC,
            annotations=[
                (BQB.IS, "oba/OBA:0000173"),
                (BQB.IS, "efo/0010604"),
            ],
            units=["l"],
            synonyms=["urine volume measurement", "diuresis"],
        ),
        MeasurementType(
            "gastric measurement",
            description="Any value resulting from the quantification of a morphological "
            "or physiological parameter pertaining to the gastrointestinal tract (e.g., gastric juice)",
            parents=["physiological measurement"],
            dtype=DType.ABSTRACT,
            annotations=[],
            synonyms=[],
        ),
        MeasurementType(
            "gastric juice volume",
            description="The volume of the gastric juice. A thin watery acid digestive "
            "fluid secreted by glands in the mucous membrane of the stomach. "
            "Should be used with substance 'h2o'.",
            parents=["gastric measurement"],
            dtype=DType.NUMERIC,
            annotations=[
                (BQB.IS, "bto/BTO:0000501"),
                (BQB.IS, "fma/FMA:62972"),
            ],
            units=["l"],
            synonyms=["stomach acid volume"],
        ),
        MeasurementType(
            "cardiovascular measurement",
            description="Any value resulting from the quantification of a morphological "
            "or physiological parameter pertaining to the heart and/or blood "
            "vessels.",
            parents=["physiological measurement"],
            dtype=DType.ABSTRACT,
            annotations=[(BQB.IS, "cmo/CMO:0000001")],
        ),
        MeasurementType(
            "blood pressure",
            description="Measurement of the pressure, or force per area, exerted by "
            "circulating blood against the walls of the blood vessels. "
            "The pressure is dependent on the energy of the heart action, "
            "elasticity of the vessel walls and volume and viscosity of "
            "the blood.",
            parents=["cardiovascular measurement"],
            dtype=DType.NUMERIC,
            units=PRESSURE_UNITS,
            annotations=[
                (BQB.IS, "cmo/CMO:0000003"),
            ],
        ),
        MeasurementType(
            "blood pressure (categorical)",
            description="Measurement of the pressure, or force per area, exerted by "
            "circulating blood against the walls of the blood vessels. "
            "The pressure is dependent on the energy of the heart action, "
            "elasticity of the vessel walls and volume and viscosity of "
            "the blood.",
            parents=["cardiovascular measurement"],
            dtype=DType.CATEGORICAL,
            annotations=[
                (BQB.IS, "cmo/CMO:0000003"),
            ],
        ),
        Choice(
            "blood pressure normal",
            name="normal",
            description="Normal blood pressure.",
            parents=["blood pressure (categorical)"],
            annotations=[],
        ),
        Choice(
            "blood pressure elevated",
            name="elevated",
            description="Elevated blood pressure.",
            parents=["blood pressure (categorical)"],
            annotations=[],
        ),
        MeasurementType(
            "blood pressure systolic",
            description="The peak blood Pressure during the systolic phase of the heartbeat. The maximum arterial "
            "pressure within the cardiac cycle, i.e. at the point at which the heart is in its maximal "
            "state of contraction. This is the time when the blood is forced from the ventricles of the "
            "heart into the pulmonary artery and the aorta.",
            parents=["blood pressure"],
            dtype=DType.NUMERIC,
            units=PRESSURE_UNITS,
            annotations=[
                (BQB.IS, "cmo/CMO:0000004"),
                (BQB.IS, "ncit/C25298"),
            ],
        ),
        MeasurementType(
            "blood pressure diastolic",
            description="The blood pressure during the diastolic phase of the heartbeat. The minimum arterial pressure "
            "within the cardiac cycle, usually at the point at which the heart is in a state of relaxation "
            "and expansion. This is the time when the ventricles fill with blood.",
            parents=["blood pressure"],
            dtype=DType.NUMERIC,
            units=PRESSURE_UNITS,
            annotations=[
                (BQB.IS, "cmo/CMO:0000005"),
                (BQB.IS, "ncit/C25299"),
            ],
            synonyms=["blood pressures diastolic"],
        ),
        MeasurementType(
            sid="map",
            name="MAP",
            label="Mean arterial pressure (MAP)",
            description="Mean arterial pressure (MAP). The mean pressure of the blood within the arterial circulation. "
            "The arterial pressure may be directly measured by insertion of an intra-arterial catheter "
            "connected to a transducer. The mean arterial pressure (MAP) can be calculated by subsequent "
            "analysis of the waveform. ",
            parents=["blood pressure"],
            dtype=DType.NUMERIC,
            units=PRESSURE_UNITS,
            annotations=[
                (BQB.IS, "ncit/C49679"),
                (BQB.IS, "efo/0006340"),
                (BQB.IS, "cmo/CMO:0000009"),
            ],
            synonyms=["mean arterial pressure", "MAP"],
        ),
        MeasurementType(
            sid="blood flow",
            description="Blood flow.",
            parents=["cardiovascular measurement"],
            dtype=DType.ABSTRACT,
            annotations=[(BQB.IS, "ncit/C16353")],
            synonyms=["Blood circulation"],
        ),
        MeasurementType(
            sid="hbf",
            name="hbf",
            label="Hepatic blood flow",
            description="(Total) Hepatic blood flow (HBF). The volume of blood flowing through the liver per unit of time."
            "Can be caluclated as THBF = PBF + HABF",
            parents=["blood flow"],
            dtype=DType.NUMERIC,
            units=["l/min", "l/min/kg", "l/min/m^2"],
            annotations=[(BQB.IS_VERSION_OF, "cmo/CMO:0000196")],
            synonyms=["THBF", "total hepatic blood flow"],
        ),
        MeasurementType(
            sid="pbf",
            name="pbf",
            label="portal blood flow (PBF)",
            description="Portal blood flow.",
            parents=["blood flow"],
            dtype=DType.NUMERIC,
            units=["l/min", "l/min/kg"],
            annotations=[(BQB.IS, "ncit/C56744")],
            synonyms=["PBF", "portal vein flow"],
        ),
        MeasurementType(
            sid="habf",
            label="hepatic arterial blood flow (HABF)",
            description="Hepatic arterial blood flow (HABF).",
            parents=["blood flow"],
            dtype=DType.NUMERIC,
            units=["l/min", "l/min/kg"],
            annotations=[(BQB.IS_VERSION_OF, "cmo/CMO:0000198")],
            synonyms=["HABF"],
        ),
        MeasurementType(
            sid="azygos_blood_flow",
            name="azygos_blood_flow",
            label="Azygos blood flow",
            description="Blood flow through the vena azygos.",
            parents=["blood flow"],
            dtype=DType.NUMERIC,
            units=["l/min"],
        ),
        MeasurementType(
            sid="bile_flow",
            name="bile_flow",
            label="bile flow",
            description="Bile flow.",
            parents=["physiological measurement"],
            dtype=DType.NUMERIC,
            annotations=[
                (BQB.HAS_PART, "ncit/C13192"),  # bile
                (BQB.HAS_PART, "ncit/C41141"),  # flow
            ],
            units=FLOW_UNITS,
        ),
        MeasurementType(
            sid="thalf_gastric_emptying",
            name="thalf_gastric_emptying",
            label="gastric emptying half-life",
            description="Gastric emptying half-life.",
            parents=["physiological measurement"],
            dtype=DType.NUMERIC,
            units=TIME_UNITS,
        ),
        MeasurementType(
            sid="percent_gastric_emptying",
            name="percent_gastric_emptying",
            label="gastric emptying percentage",
            description="Gastric emptying percentage. How much of the substance is still "
            "found in the stomach.",
            parents=["physiological measurement"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
        ),
        MeasurementType(
            sid="rate_gastric_emptying",
            name="rate_gastric_emptying",
            label="gastric emptying rate",
            description="Gastric emptying rate.",
            parents=["physiological measurement"],
            dtype=DType.NUMERIC,
            units=["1/min"],
        ),
        MeasurementType(
            sid="heart measurement",
            description="heart measurement",
            parents=["cardiovascular measurement"],
            dtype=DType.ABSTRACT,
            annotations=[
                (BQB.IS, "cmo/CMO:0000670"),
            ],
        ),
        MeasurementType(
            sid="cardiac-function-test",
            name="cardiac function test",
            description="Cardiac function test. A heart function measurement is a "
            "measurement of some heart function or process typically used to "
            "establish normal heart function or in process of diagnosis",
            parents=["chemistry measurement", "heart measurement"],
            dtype=DType.CATEGORICAL,
            annotations=[
                (BQB.IS, "efo/0004311"),
            ],
            synonyms=["cardiac function", "heart function"],
        ),
        MeasurementType(
            sid="heart-rate",
            name="heart rate",
            description="The number of contractions of the cardiac ventricles per "
            "unit of time. This is the resting heart rate, for heart rate in"
            "exercise use 'Exercise heart rate (EHR)'.",
            parents=["heart measurement"],
            dtype=DType.NUMERIC,
            units=["1/s"],
            annotations=[
                (BQB.IS, "cmo/CMO:0000670"),
            ],
            synonyms=["resting heart rate"],
        ),
        MeasurementType(
            sid="ehr",
            name="EHR",
            label="Exercise heart rate (EHR)",
            description="The number of contractions of the cardiac ventricles per "
            "unit of time in an exercise test. "
            "For percental change see 'EHR change'.",
            parents=["heart-rate"],
            dtype=DType.NUMERIC,
            units=["1/s"],
            annotations=[
                (BQB.IS_VERSION_OF, "cmo/CMO:0000670"),
            ],
            synonyms=["EHR"],
        ),
        MeasurementType(
            sid="ehr-change",
            name="EHR change",
            label="Exercise heart rate change (EHR)",
            description="The change in the number of contractions of the cardiac ventricles per "
            "unit of time. For absolute change see 'EHR'. Sometimes refered to as "
            "beta blockage.",
            parents=["ehr"],
            dtype=DType.NUMERIC,
            units=["percent"],
            annotations=[
                (BQB.IS_VERSION_OF, "cmo/CMO:0000742"),
            ],
            synonyms=["beta blockage"],
        ),
        MeasurementType(
            sid="ehr-change-auec",
            name="EHR change AUEC",
            label="Area under the effect curve (AUEC) of EHR",
            description="Area under the effect curve (AUEC) of EHR.",
            parents=["ehr-change"],
            dtype=DType.NUMERIC,
            units=["percent*hr"],
            annotations=[],
            synonyms=["AUEC"],
        ),
        MeasurementType(
            sid="ehr-change-average",
            name="EHR change average",
            label="Exercise heart rate change average (AUEC/tau)",
            description="The average calculated as AUC/end over the change in EHR",
            parents=["ehr-change"],
            dtype=DType.NUMERIC,
            units=["percent"],
            annotations=[],
            synonyms=["AUEC/tau", "average under the effect curve"],
        ),
        MeasurementType(
            sid="ehr-change-fluctuations",
            name="EHR change fluctuations",
            label="Exercise heart rate change fluctations",
            description="Fluctations in EHR change.",
            parents=["ehr-change"],
            dtype=DType.NUMERIC,
            units=["percent"],
            annotations=[],
        ),
        MeasurementType(
            sid="cardiac output",
            name="cardiac output",
            label="cardiac output (CO)",
            description="Cardiac output (CO). The amount of blood pumped by the heart per unit of time."
            "The total volume of blood pumped by the heart over a set period of time, conventionally one "
            "minute; it is calculated as heart rate times stroke volume (CO= HR x SV), and is additionally "
            "dependent on preload and afterload for functional output.",
            parents=["heart measurement"],
            dtype=DType.NUMERIC,
            units=["l/min"],
            annotations=[
                (BQB.IS, "ncit/C119246"),
                (BQB.IS, "omit/0003637"),
            ],
            synonyms=["CO"],
        ),
        MeasurementType(
            sid="cardiac-index",
            name="cardiac index",
            label="Cardiac index (CI)",
            description="Cardiac index (CI). The measure of an individual's cardiac output as divided by their body "
            "surface area (CI=CO/BSA). This calculation is a useful function to determine an "
            "individual's cardiac performance in relation to their body size, providing an overview of "
            "global cardiovascular function. See also 'cardiac output'.",
            parents=["heart measurement"],
            dtype=DType.NUMERIC,
            units=["l/min/m^2"],
            annotations=[
                (BQB.IS, "ncit/C119245"),
                (BQB.IS, "cmo/CMO:0002881"),
            ],
            synonyms=["CI"],
        ),
        MeasurementType(
            "organ morphological measurement",
            description="A measurement of the physical form and structure of an organ.",
            parents=["physiological measurement"],
            dtype=DType.ABSTRACT,
            annotations=[
                (BQB.IS, "cmo/CMO:0000068"),
            ],
        ),
        MeasurementType(
            "organ weight",
            description="organ weight",
            parents=["organ morphological measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            "liver weight",
            description="organ weight of liver",
            parents=["organ weight"],
            dtype=DType.NUMERIC,
            units=["g", "g/kg"],
        ),
        MeasurementType(
            "kidney weight",
            description="organ weight of kidney",
            parents=["organ weight"],
            dtype=DType.NUMERIC,
            units=["g", "g/kg"],
        ),
        MeasurementType(
            "muscle weight",
            description="organ weight of muscle",
            parents=["organ weight"],
            dtype=DType.NUMERIC,
            units=["g", "g/kg"],
        ),
        MeasurementType(
            sid="fat weight",
            label="fat mass (FM)",
            description="Fat mass (FM). Weight of fat tissue. The weight of the body fat associated with either a "
            "particular body part or the whole body.",
            parents=["organ weight"],
            dtype=DType.NUMERIC,
            units=["g", "g/kg"],
            annotations=[(BQB.IS, "ncit/C158256")],
        ),
        MeasurementType(
            "organ volume",
            description="organ volume",
            parents=["organ morphological measurement"],
            dtype=DType.ABSTRACT,
            units=["ml", "ml/kg"],
        ),
        MeasurementType(
            sid="liver_volume",
            name="liver volume",
            description="Measurement of the size of the three dimensional space occupied "
            "by the liver.",
            parents=["organ volume"],
            dtype=DType.NUMERIC,
            units=["ml", "ml/kg"],
            annotations=[(BQB.IS, "cmo/CMO:0001468")],
        ),
        MeasurementType(
            sid="liver_remnant_volume",
            name="liver remnant volume",
            description="Measurement of the size of the three dimensional space occupied "
            "by the liver remnant after hepatectomy.",
            parents=["liver volume"],
            dtype=DType.NUMERIC,
            units=["ml", "ml/kg"],
            annotations=[(BQB.IS_VERSION_OF, "cmo/CMO:0001468")],
            synonyms=["liver remnant"],
        ),
        MeasurementType(
            sid="fractional_liver_remnant_volume",
            name="fractional liver remnant volume",
            description="Measurement of the size of the three dimensional space occupied "
            "by the liver remnant after hepatectomy in percent of pre-operative"
            "liver volume. See also 'liver resection rate'.",
            parents=["liver volume"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
            annotations=[(BQB.IS_VERSION_OF, "cmo/CMO:0001468")],
        ),
        MeasurementType(
            sid="liver_resection_rate",
            name="liver resection rate",
            description="Fraction of removed liver volume in hepatectomy. The following "
            "relationship holds 'liver resection rate = 1 - fractional liver remnant volume'."
            "See also 'fractional liver remnant volume'.",
            parents=["liver volume"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
            annotations=[(BQB.IS_VERSION_OF, "ncit/C131680")],
        ),
    ]
)

MEASUREMENT_NODES.extend(
    [
        MeasurementType(
            "metabolic measurement",
            description="Abstract Class for metabolic measurements.",
            parents=["measurement"],
            dtype=DType.ABSTRACT,
        ),
        MeasurementType(
            "metabolic ratio",
            description="Ratio between substances. Often used to characterize enzyme or "
            "pathway. Use a derived substance which defines the ratio.",
            parents=["metabolic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            "metabolic_fraction",
            description="Fraction of the dose that was metabolized. Typically estimated by area under the curve."
            "The material given in the dose should be known given the intervention name, and the metabolite"
            "in question should be specified in the 'substance' column",
            parents=["metabolic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            sid="cumulative-metabolic-ratio",
            name="cumulative metabolic ratio",
            description="Cumulative ratio between substances. The ratio of the cumulative "
            "amount of substances is calculated. This is for instance used "
            "for ratios in the urine, where urine is collected over given period"
            "and then the ratios are calculated (see also 'cumulative amount' "
            "and 'recovery').",
            parents=["metabolic measurement"],
            dtype=DType.NUMERIC,
            units=RATIO_UNITS,
        ),
        MeasurementType(
            sid="metabolic-phenotype",
            name="metabolic phenotype",
            description="Ratio between substances. Often used to characterize enzyme or "
            "pathway. Use a derived substance which defines the ratio.",
            parents=["metabolic measurement"],
            dtype=DType.CATEGORICAL,
            units=[NO_UNIT],
        ),
        Choice(
            sid="mp-um",
            name="um",
            label="Ultra rapid metabolizer (UM)",
            description="Ultra rapid metabolizer of a substance determined by a "
            "metabolic measurement.",
            parents=["metabolic phenotype"],
        ),
        Choice(
            sid="mp-pm",
            name="pm",
            label="Poor metabolizer (PM)",
            description="Poor metabolizer of a substance determined by a metabolic "
            "measurement.",
            synonyms=["slow metabolizer"],
            parents=["metabolic phenotype"],
        ),
        Choice(
            sid="mp-em",
            name="em",
            label="Extensive metabolizer (EM)",
            description="extensive metabolizer of a substance determined by a metabolic "
            "measurement.",
            synonyms=["rapid metabolizer"],
            parents=["metabolic phenotype"],
        ),
        Choice(
            sid="mp-im",
            name="im",
            label="Intermediate metabolizer (IM)",
            description="intermediate metabolizer of a substance determined by a "
            "metabolic measurement.",
            parents=["metabolic phenotype"],
        ),
        MeasurementType(
            sid="biochemistry-level",
            name="biochemistry level",
            description="biochemistry in normal range",
            parents=[
                "biochemistry (categorial)",
                "liver function test",
                "kidney function test",
                "cardiac function test",
            ],
            dtype=DType.ABSTRACT,
            annotations=[(BQB.IS_VERSION_OF, "ncit/C27993")],
        ),
        Choice(
            sid="biochemistry-normal",
            name="normal",
            label="biochemistry normal",
            description="Biochemistry in normal range. Being approximately average or within "
            "certain limits; conforming with or constituting a norm or standard "
            "or level or type or social norm.",
            parents=["biochemistry level"],
            annotations=[
                (BQB.IS, "ncit/C14165"),
                (BQB.IS, "sio/SIO:000554"),
            ],
        ),
        Choice(
            sid="biochemistry-abnormal",
            name="abnormal",
            label="biochemistry abnormal",
            description="biochemistry abnormal range of result compared to healthy control "
            "(see also `elevated` and `decreased` for more "
            "precise terms).",
            parents=["biochemistry level"],
            annotations=[
                (BQB.IS, "ncit/C25401"),
                (BQB.IS, "sio/SIO:000555"),
            ],
        ),
        Choice(
            sid="biochemistry-elevated",
            name="elevated",
            label="biochemistry elevated",
            description="Biochemistry increased result (compared to healthy control). "
            "Raised or increased above normal levels or limits.",
            parents=["biochemistry abnormal"],
            annotations=[
                (BQB.IS, "ncit/C25493"),
            ],
        ),
        Choice(
            sid="biochemistry-decreased",
            name="decreased",
            label="biochemistry decreased",
            description="biochemistry decreased result (compared to healthy control). "
            "A quality that has a value that is decreased compared to normal "
            "or average.",
            parents=["biochemistry abnormal"],
            annotations=[
                (BQB.IS, "sio/SIO:000751"),
                (BQB.IS, "ncit/C25640"),
            ],
        ),
        MeasurementType(
            sid="albumin-measurement",
            name="albumin measurement",
            description="A quantitative measurement of albumin present in a sample.",
            parents=["biochemistry level"],
            annotations=[
                (BQB.IS, "ncit/C64431"),
            ],
            dtype=DType.NUMERIC,
            units=CONCENTRATION_UNITS,
        ),
        MeasurementType(
            sid="prealbumin-measurement",
            name="prealbumin measurement",
            description="The determination of the amount of prealbumin present in a sample.",
            parents=["biochemistry level"],
            annotations=[
                (BQB.IS, "ncit/C100435"),
            ],
            dtype=DType.NUMERIC,
            units=CONCENTRATION_UNITS,
        ),
        MeasurementType(
            sid="western-blot",
            name="western blot",
            description="The western blot, or western blotting, is a widely used analytical technique "
            "in molecular biology and immunogenetics to detect specific proteins in a sample "
            "of tissue homogenate or extract. Use the 'tissue' field to specify origin of "
            "sample and the 'choice' field to specify the protein.",
            parents=["chemistry measurement"],
            dtype=DType.NUMERIC_CATEGORICAL,
            units=["arbitrary_unit/mg"],
            annotations=[
                # (BQB.IS, "scdo/SCDO:0001240"),
                # (BQB.IS, "bao/0002424"),
                (BQB.IS, "ncit/C16357"),
            ],
        ),
        MeasurementType(
            "relative-peak-area",
            description="Relative peak area. Area of peak relative to a reference peak."
            "E.g. relative concentrations in mass spectrometry via relative peak"
            "areas.",
            parents=["measurement"],
            dtype=DType.NUMERIC,
            units=[DIMENSIONLESS],
            annotations=[],
        ),
    ]
)

PROTEINS = ["CYP2E1"]


for protein in PROTEINS:
    choice_protein = Choice(
        sid=protein,
        description=f"{protein} was detected during western blotting. ",
        parents=["western-blot"],
        annotations=[],
    )
    MEASUREMENT_NODES.append(choice_protein)
