"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const s3 = require("@aws-cdk/aws-s3");
const s3n = require("@aws-cdk/aws-s3-notifications");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("@aws-cdk/aws-s3");
test('check exception for Missing existingBucketObj from props for deploy = false', () => {
    const stack = new core_1.Stack();
    try {
        defaults.buildS3Bucket(stack, {});
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('s3 bucket with bucketId', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {}, 'my');
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "myS3LoggingBucketDE461344"
            }
        },
    }));
});
test('s3 bucket with bucketProps', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            bucketName: 'mybucket'
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketName: "mybucket"
    }));
});
test('s3 bucket with default props', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        },
        VersioningConfiguration: {
            Status: "Enabled"
        }
    }));
});
test('s3 bucket with life cycle policy', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            lifecycleRules: [{
                    expiration: core_1.Duration.days(365),
                    transitions: [{
                            storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: core_1.Duration.days(30)
                        }, {
                            storageClass: aws_s3_1.StorageClass.GLACIER,
                            transitionAfter: core_1.Duration.days(90)
                        }]
                }]
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    ExpirationInDays: 365,
                    Status: "Enabled",
                    Transitions: [
                        {
                            StorageClass: "STANDARD_IA",
                            TransitionInDays: 30
                        },
                        {
                            StorageClass: "GLACIER",
                            TransitionInDays: 90
                        }
                    ]
                }
            ]
        }
    }));
});
test('s3 bucket with access logging configured', () => {
    const stack = new core_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            }
        },
    }));
});
test('Check S3 Bucket policy', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Sid: "HttpsOnly",
                    Action: "*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false",
                        },
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": ["S3Bucket07682993", "Arn"],
                                    },
                                    "/*",
                                ],
                            ],
                        },
                        {
                            "Fn::GetAtt": [
                                "S3Bucket07682993",
                                "Arn"
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17",
        },
    }));
});
test('s3 bucket with LoggingBucket and versioning turned off', () => {
    const stack = new core_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket,
            serverAccessLogsPrefix: 'access-logs',
            versioned: false
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    }));
});
test('s3 bucket versioning turned off', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsPrefix: 'access-logs',
            versioned: false
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "S3LoggingBucket800A2B27"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    }));
});
test('s3 bucket with LoggingBucket and auto delete objects', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        loggingBucketProps: {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    }));
    assert_1.expect(stack).to(assert_1.haveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "S3LoggingBucket800A2B27"
        }
    }));
});
test('s3 bucket versioning turned on', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsPrefix: 'access-logs',
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LifecycleConfiguration: {
            Rules: [
                {
                    NoncurrentVersionTransitions: [
                        {
                            StorageClass: "GLACIER",
                            TransitionInDays: 90
                        }
                    ],
                    Status: "Enabled"
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "S3LoggingBucket800A2B27"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        },
        VersioningConfiguration: {
            Status: "Enabled"
        }
    }));
});
test('Suppress cfn-nag warning for s3 bucket notification', () => {
    const stack = new core_1.Stack();
    let queue;
    let bucket;
    [bucket] = defaults.buildS3Bucket(stack, {});
    [queue] = defaults.buildQueue(stack, "S3BucketNotificationQueue", {});
    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.SqsDestination(queue));
    defaults.addCfnNagS3BucketNotificationRulesToSuppress(stack, "BucketNotificationsHandler050a0587b7544547bf325f094a3db834");
    assert_1.expect(stack).to(assert_1.haveResource("AWS::Lambda::Function", {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W58",
                        reason: "Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions."
                    },
                    {
                        id: 'W89',
                        reason: `This is not a rule for the general case, just for specific use cases/industries`
                    },
                    {
                        id: 'W92',
                        reason: `Impossible for us to define the correct concurrency for clients`
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Policy", {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W12",
                        reason: "Bucket resource is '*' due to circular dependency with bucket and role creation at the same time"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition));
});
//# sourceMappingURL=data:application/json;base64,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