"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_glue_1 = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("..");
// --------------------------------------------------------------
// Test deployment with role creation
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobRole = new aws_iam_1.Role(stack, 'CustomETLJobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com')
    });
    const cfnJobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: 's3://fakescriptlocation/fakebucket',
        },
        role: _jobRole.roleArn
    }, 'testETLJob', {});
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://fakescriptlocation/fakebucket"
            },
            Role: {
                "Fn::GetAtt": [
                    "CustomETLJobRole90A83A66",
                    "Arn"
                ]
            },
            GlueVersion: "2.0",
            NumberOfWorkers: 2,
            SecurityConfiguration: "testETLJob",
            WorkerType: "G.1X"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an existing Glue Job
// --------------------------------------------------------------
test('Create a Glue Job outside the construct', () => {
    // Stack
    const stack = new core_1.Stack();
    const _existingCfnJob = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
        command: {
            name: 'pythonshell',
            pythonVersion: '2',
            scriptLocation: 's3://existingFakeLocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        allocatedCapacity: 2,
        maxCapacity: 4,
        numberOfWorkers: 2,
        workerType: 'Standard'
    });
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    defaults.buildGlueJob(stack, {
        existingCfnJob: _existingCfnJob,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            AllocatedCapacity: 2,
            Command: {
                Name: "pythonshell",
                PythonVersion: "2",
                ScriptLocation: "s3://existingFakeLocation/existingScript",
            },
            GlueVersion: "1",
            MaxCapacity: 4,
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            WorkerType: "Standard",
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Provide additional parameters other than default ones
// --------------------------------------------------------------
test('Test custom deployment properties', () => {
    // Stack
    const stack = new core_1.Stack();
    const _commandName = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _commandName,
            pythonVersion: '3',
            scriptLocation: 's3://existingFakeLocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        numberOfWorkers: 2,
        workerType: 'Standard'
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // check if Glue Job Resource was created correctly
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://existingFakeLocation/existingScript",
            },
            GlueVersion: "1",
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            SecurityConfiguration: "ETLJobSecurityConfig",
            WorkerType: "Standard",
        },
        Type: "AWS::Glue::Job"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the role is created
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazon.com"
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            Description: "Existing role"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the security config is created
    expect(stack).toHaveResourceLike('AWS::Glue::SecurityConfiguration', {
        Properties: {
            EncryptionConfiguration: {
                JobBookmarksEncryption: {
                    JobBookmarksEncryptionMode: "CSE-KMS",
                    KmsKeyArn: {
                        "Fn::Join": [
                            "", [
                                "arn:", {
                                    Ref: "AWS::Partition",
                                },
                                ":kms:", {
                                    Ref: "AWS::Region",
                                },
                                ":", {
                                    Ref: "AWS::AccountId",
                                },
                                ":alias/aws/glue",
                            ],
                        ],
                    },
                },
                S3Encryptions: [{
                        S3EncryptionMode: "SSE-S3",
                    }],
            },
            Name: "ETLJobSecurityConfig",
        },
        Type: "AWS::Glue::SecurityConfiguration",
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Do not supply parameters and error out
// --------------------------------------------------------------
test('Do no supply glueJobProps or existingCfnJob and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' }))
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Allow the construct to create the job role required
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [{
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazonaws.com"
                        }
                    }],
                Version: "2012-10-17"
            },
            Description: "Service role that Glue custom ETL jobs will assume for exeuction"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when output location is provided
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3,
            existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                versioned: false,
                bucketName: 'outputbucket',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: core_1.RemovalPolicy.DESTROY
            })
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        Type: 'AWS::S3::Bucket',
        Properties: {
            BucketEncryption: {
                ServerSideEncryptionConfiguration: [{
                        ServerSideEncryptionByDefault: {
                            SSEAlgorithm: "AES256"
                        }
                    }]
            },
            BucketName: "outputbucket"
        },
        UpdateReplacePolicy: "Delete",
        DeletionPolicy: "Delete"
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when script location not provided - throw error
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    try {
        defaults.buildGlueJob(stack, {
            glueJobProps: cfnJobProps,
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3,
                existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                    versioned: false
                })
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
        expect(error.message).toEqual('Script location has to be provided as an s3 Url location. Script location cannot be empty');
    }
});
// --------------------------------------------------------------
// Dont pass Job Command attributes and it should throw an error
// --------------------------------------------------------------
test('Test for incorrect Job Command property', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {},
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Check for CfnJob.JobCommandProperty type
// --------------------------------------------------------------
test('check for JobCommandProperty type', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {
                command: {
                    fakekey: 'fakevalue'
                }
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Supply maxCapacity with GlueVersion 2.0 and error out
// --------------------------------------------------------------
test('GlueJob configuration with glueVersion 2.0 should not support maxCapacity and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                glueVersion: '2.0',
                maxCapacity: '2'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Fail if setting maxCapacity and WorkerType/ NumberOfWorkers
// --------------------------------------------------------------
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                workerType: 'Standard'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2,
                workerType: 'G1.X'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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