"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployRegionalApiGateway(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    return defaults.RegionalLambdaRestApi(stack, fn);
}
function setupRestApi(stack, apiProps) {
    const [restApi] = defaults.GlobalRestApi(stack, apiProps);
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    // Setup the API Gateway Integration
    const apiGatewayIntegration = new api.AwsIntegration({
        service: "sqs",
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            requestParameters: {
                "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
            },
            requestTemplates: {
                "application/x-www-form-urlencoded": "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")&MessageAttribute.1.Name=queryParam1&MessageAttribute.1.Value.StringValue=$input.params(\"query_param_1\")&MessageAttribute.1.Value.DataType=String"
            },
            integrationResponses: [
                {
                    statusCode: "200",
                    responseTemplates: {
                        "text/html": "Success"
                    }
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        },
        path: '11112222' + "/" + 'thisqueuequeueName'
    });
    // Setup the API Gateway method(s)
    apiGatewayResource.addMethod('POST', apiGatewayIntegration, {
        requestParameters: {
            "method.request.querystring.query_param_1": true
        },
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ]
    });
}
test('Test override for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        description: 'Hello World'
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            Description: "Hello World",
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "LambdaRestApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test override for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn, {
        handler: fn,
        restApiName: "HelloWorld"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "EDGE"
                ]
            },
            Name: "HelloWorld"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test ApiGateway::Account resource for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test ApiGateway::Account resource for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test default RestApi deployment w/ ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        restApiName: "customRestApi"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Name: "customRestApi"
    });
});
test('Test default RestApi deployment for Cloudwatch loggroup', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        UpdateReplacePolicy: "Retain",
        DeletionPolicy: "Retain"
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::ApiGateway::Stage', {
        AccessLogSetting: {
            DestinationArn: {
                "Fn::GetAtt": [
                    "ApiAccessLogGroupCEA70788",
                    "Arn"
                ]
            },
            Format: "{\"requestId\":\"$context.requestId\",\"ip\":\"$context.identity.sourceIp\",\"user\":\"$context.identity.user\",\"caller\":\"$context.identity.caller\",\"requestTime\":\"$context.requestTime\",\"httpMethod\":\"$context.httpMethod\",\"resourcePath\":\"$context.resourcePath\",\"status\":\"$context.status\",\"protocol\":\"$context.protocol\",\"responseLength\":\"$context.responseLength\"}",
        },
    });
});
test('Test addMethodToApiResource with action', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const getRequestTemplate = "{}";
    // Add Method
    defaults.addProxyMethodToApiResource({
        action: "Query",
        service: "dynamodb",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "GET",
        requestTemplate: getRequestTemplate
    });
    // Add Method
    defaults.addProxyMethodToApiResource({
        path: '11112222' + "/" + 'thisqueuequeueName',
        service: "sqs",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "PUT",
        requestTemplate: getRequestTemplate
    });
    // Error scenario: missing action and path
    try {
        defaults.addProxyMethodToApiResource({
            service: "sqs",
            apiResource: apiGatewayResource,
            apiGatewayRole,
            apiMethod: "DELETE",
            requestTemplate: getRequestTemplate
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Test default RestApi w/ request model and validator', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const validator = restApi.addRequestValidator('default-validator', {
        requestValidatorName: 'default-validator',
        validateRequestBody: true
    });
    defaults.addProxyMethodToApiResource({
        service: "kinesis",
        action: "PutRecord",
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "{}",
        contentType: "'x-amz-json-1.1'",
        requestValidator: validator,
        requestModel: { "application/json": api.Model.EMPTY_MODEL }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RequestValidator', {
        Name: "default-validator",
        ValidateRequestBody: true
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        RequestModels: { "application/json": "Empty" }
    });
});
// -----------------------------------------------------------------------
// Test for Regional ApiGateway Creation
// -----------------------------------------------------------------------
test('Test for RegionalRestApiGateway', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack, {
        restApiName: "HelloWorld-RegionalApi"
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "HelloWorld-RegionalApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// -----------------------------------------------------------------------
// Tests for exception while overriding restApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding restApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    try {
        defaults.RegionalRestApi(stack, {
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Tests for exception while overriding LambdaRestApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding LambdaRestApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    try {
        defaults.GlobalLambdaRestApi(stack, fn, {
            handler: fn,
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Test for Integration Request Props Override
// -----------------------------------------------------------------------
test('Test for Integration Request Props Override', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    const integReqParams = { 'integration.request.path.topic-level-1': "'method.request.path.topic-level-1'" };
    const integResp = [
        {
            statusCode: "200",
            selectionPattern: "2\\d{2}",
            responseTemplates: {
                "application/json": "$input.json('$')"
            }
        }
    ];
    // Override the default Integration Request Props
    const integrationReqProps = {
        subdomain: 'abcdefgh12345',
        options: {
            requestParameters: integReqParams,
            integrationResponses: integResp,
            passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
        }
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        awsIntegrationProps: integrationReqProps
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
                "integration.request.path.topic-level-1": "'method.request.path.topic-level-1'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:abcdefgh12345.iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200",
                ResponseParameters: {
                    "method.response.header.Content-Type": true
                }
            }
        ]
    });
});
// -----------------------------------------------------------------------
// Test for Method Options Override
// -----------------------------------------------------------------------
test('Test for Method Request Props Override', () => {
    const stack = new core_1.Stack();
    const [globalApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalApi.root.addResource('hello');
    const methodReqParams = { 'method.request.path.topic-level-1': true };
    const methodResp = [
        {
            statusCode: "403"
        }
    ];
    const resourceMethodOptions = {
        requestParameters: methodReqParams,
        methodResponses: methodResp,
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        methodOptions: resourceMethodOptions
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    StatusCode: "200"
                },
                {
                    StatusCode: "500",
                    ResponseTemplates: {
                        "text/html": "Error"
                    },
                    SelectionPattern: "500"
                }
            ],
            PassthroughBehavior: "NEVER",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true
        }
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of RestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using restApiProps', () => {
    const stack = new core_1.Stack();
    const [globalRestApi] = defaults.GlobalRestApi(stack, {
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalRestApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of LambdaRestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using lambdaApiProps', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
//# sourceMappingURL=data:application/json;base64,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